<?php
/**
 * Menu model for the Navigation plugin.
 *
 * @author  Andy Carter
 * @package Navigation
 */
App::uses('EvNavigationAppModel', 'EvNavigation.Model');

class Menu extends EvNavigationAppModel {

	// Menus have a tree behaviour.
	public $actsAs = array(
		'Tree' => array(
			'level' => 'level',
		),
	);

	public $order = array('lft' => 'ASC');

	// We want to keep a record of which menu items are currently for use when
	// building our menus.
	protected $_menuActiveId = null;

	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$this->removeBehaviour('CustomCopyable');

		return;
	}

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'cannot be blank'
			),
			'unique' => array(
				'rule' => array('checkCombinationUnique', array('parent_id')),
				'message' => 'A menu item with this name already exists for this parent'
			),
		),
		'system_name' => array(
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'System names must be unique',
				'allowEmpty' => true,
			)
		),
	);

/**
 * Before save
 * @param array $options Save options
 * @return bool
 */
	public function beforeSave($options = []) {
		// If the 'checkUrl' flag is passed we want to check if the `url` field has been populated;
		// if it has then we want to clear any polymorphic relation.
		if (!empty($options['checkUrl']) && !empty($this->data[$this->alias]['url'])) {
			$this->data[$this->alias] += [
				'plugin' => null,
				'model' => null,
				'controller' => null,
				'model_id' => null,
				'action' => null
			];
		}

		// Prevent saving blank strings as they will conflict for not being unique
		// Null values are ignored by this restriction
		if (isset($this->data[$this->alias]['system_name']) && empty($this->data[$this->alias]['system_name'])) {
			$this->data[$this->alias]['system_name'] = null;
		}

		return parent::beforeSave($options);
	}

/**
 * Called after each successful save operation.
 *
 * @param bool $created True if this save created a new record
 * @param array $options Options passed from Model::save().
 * @return void
 */
	public function afterSave($created = false, $options = array()) {
		Cache::clearGroup('Menus', 'EvNavigation_Menus');
		Cache::clearGroup('EvNavigationMenuPermissions', 'EvNavigation_Permissions');

		parent::afterSave($created, $options);
	}

/**
 * Returns a menu with active menu items flagged as 'active'.
 *
 * @param int|string $id       ID or system name of menu to return
 * @param int        $activeId Optional active menu item ID
 * @return array     array of menu items
 */
	public function getMenu($id, $activeId = null) {
		$cacheKey = 'menu_data_';
		if (! empty($this->locale)) {
			$cacheKey .= $this->locale . '_';
		}
		$cacheKey .= $id . '_' . $activeId;
		return Cache::remember($cacheKey, function () use ($id, $activeId) {
			// Set the ID of the active menu item ready for searching the
			// menu with.
			$activeId = $activeId === null ? $this->_menuActiveId : $activeId;

			// Get the menu record so that we can work out its children
			// using the left and right columns.
			$menu = is_numeric($id) ? $this->findById($id) : $this->findBySystemName($id);

			if (empty($menu)) {
				return [];
			}

			// Get the menu's children.
			$data = $this->find('threaded', array(
				'conditions' => array(
					$this->alias . '.lft >' => $menu[$this->alias]['lft'],
					$this->alias . '.rght <' => $menu[$this->alias]['rght'],
					$this->alias . '.is_active' => true
				),
				'order' => array(
					$this->alias . '.lft' => 'ASC'
				),
				'joins' => array(
					array(
						'table' => 'pages',
						'alias' => 'Page',
						'type' => 'LEFT',
						'conditions' => array(
							$this->alias . '.model' => 'EvCorePage',
							'Page.id = ' . $this->alias . '.model_id'
						)
					)
				),
				'fields' => array(
					$this->alias . '.*',
					'Page.redirect_url'
				)
			));

			// Search for the active menu item in the menu data and mark it
			// as active. Then return the menu data.
			return $this->_findActive($data, $activeId);
		}, 'EvNavigation_Menus');
	}

	protected function _findActive(&$data, $activeId) {
		foreach ($data as &$item) {
			$item[$this->alias]['active'] = ($item[$this->alias]['id'] == $activeId);

			if (isset($item['children'])) {
				$this->_findActive($item['children'], $activeId);
			}
		}

		return $data;
	}

/**
 * Set the active menu item
 * @param integer $id ID of menu item to mark as active
 */
	public function setActive($id) {
		$this->_menuActiveId = $id;

		return true;
	}

/**
 * Formats a menu url into a route array based on the available data.
 *
 * Called after each find operation. Can be used to modify any results returned by find().
 * Return value should be the (modified) results.
 *
 * @param mixed $results The results of the find operation
 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
 * @return mixed Result of the find operation
 * @link https://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		$customModels = [];
		foreach ($results as &$result) {
			if (empty($result[$this->alias])) {
				continue;
			}

			$resultData = $result[$this->alias];

			// override the url with array based routing if a controller is set
			if (!empty($resultData['controller'])) {
				if (
					!empty(Configure::read('EvNavigation.use_custom_menu_routes'))
					&& !empty($resultData['model'])
					&& !empty($resultData['model_id'])
				) {
					//Attempt to find a custom route for this menu item
					$modelClass = $resultData['model'];
					if (!empty($resultData['plugin'])) {
						$modelClass = InflectorExt::camelize($resultData['plugin']) . '.' . $modelClass;
					}

					if (!isset($customModels[$modelClass])) {
						$customModels[$modelClass] = EvClassRegistry::init($modelClass);
					}

					$Model = $customModels[$modelClass];

					if (!empty($Model) && $Model->Behaviors->enabled('Navigatable')) {
						$result[$this->alias]['url'] = $Model->getCustomMenuRoute($resultData['model_id'], $resultData);
					}
				}

				if (empty($result[$this->alias]['url'])) {
					list($plugin, $controller) = pluginSplit($resultData['controller']);

					$result[$this->alias]['url'] = array(
						'plugin' => $plugin ?: $resultData['plugin'],
						'controller' => $controller,
						'action' => $resultData['action']
					);

					// add model id if one is set
					if (isset($resultData['model_id']) && !empty($resultData['model_id'])) {
						$result[$this->alias]['url'][] = $resultData['model_id'];
					}
				}
			}
		}

		return $results;
	}

	public function readForEdit($id, $query = array()) {
		$data = parent::readForEdit($id, $query);

		if (!empty($data[$this->alias]['model']) && !empty($data[$this->alias]['id'])) {
			unset($data[$this->alias]['url']);
		}

		return $data;
	}

/**
 * Returns the child menu items of a menu in tree form
 *
 * @param  mixed $parentId parent menu ID or an array of IDs
 * @param  integer $id       ID of a menu item to exclude
 * @return array
 */
	public function findTreeBranches($parentId, $id = null) {
		$conditions = array();

		if (is_numeric($parentId) && $parentId != 0) {
			$menu = $this->findById($parentId);

			$conditions = array(
				'lft >=' => $menu[$this->alias]['lft'],
				'rght <=' => $menu[$this->alias]['rght'],
				'is_active' => true
			);
		} elseif (is_array($parentId)) {

			foreach ($parentId as $thisParentId) {
				$menu = $this->findById($thisParentId);

				$conditions['OR'][] = array(
					'lft >=' => $menu[$this->alias]['lft'],
					'rght <=' => $menu[$this->alias]['rght'],
					'is_active' => true
				);
			}
		}

		if (! empty($id)) {
			$menu = $this->findById($id);

			$conditions['NOT'] = array(
				'AND' => array(
					'lft >=' => $menu[$this->alias]['lft'],
					'rght <=' => $menu[$this->alias]['rght']
				)
			);
		}

		return $this->generateTreeList(
			$conditions,
			null,
			null,
			'--'
		);
	}
}
