<?php

App::uses('BaseMigrationModelUtil', 'EvMigrationUtil.Lib');

class MenuMigrationModelUtil extends BaseMigrationModelUtil {

/**
 * Attempts to find a menu item based on its parent. If not found it is created under that parent.
 *
 * @param array $conditions       The conditions to match to be an existing record
 * @param array $parentConditions The conditions the parent must match to be the parent. If this is not found, inserting will fail.
 * @param array $data             The data to insert if it doesn't already exist
 * @return int|bool The inserted id on success, false on fail.
 */
	public function createOnceWithParent($conditions, $parentConditions, $data) {
		// Check if the parent record exists
		$parentId = $this->Model->field($this->Model->alias . '.id', $parentConditions);

		if (!empty($parentId)) {
			$key = ! empty($data[$this->Model->alias]) ? $this->Model->alias . '.parent_id' : 'parent_id';
			$data = Hash::insert($data, $key, $parentId);

			$conditions = Hash::merge($conditions, [
				$this->Model->alias . '.parent_id' => $parentId
			]);

			// Do the usual now we've identified the parent
			return $this->createOnce($conditions, $data);
		}

		return false;
	}

/**
 * Create a menu item swith children underneath them. The id is passed down to the children to be used as the parent id.
 * Children are saved from the "children" array key.
 *
 * @param array $matchFields Fields to match to check if a field already exists
 * @param array $data        An array of individual records to pass into Model->saveAssociated if the record doesn't already exist
 * @return array The id of each menu item saved, if children exist then the ids of them are also returned.
 */
	public function createAllWithChildren($matchFields, $data) {
		$results = [];

		foreach ($data as $key => $item) {
			$conditions = [];

			// Build the conditions from the field list
			foreach ($matchFields as $field) {
				if (isset($item[$this->Model->alias])) {
					$conditions[$this->Model->alias . '.' . $field] = $item[$this->Model->alias][$field];
				} else {
					$conditions[$this->Model->alias . '.' . $field] = $item[$field];
				}
			}

			$resultId = $this->createOnce($conditions, $item);

			$results[$key]['id'] = $resultId;

			if (!empty($item['children'])) {
				$item['children'] = Hash::insert($item['children'], '{n}.' . $this->Model->alias . '.parent_id', $resultId);

				$results[$key]['children'] = $this->createAllWithChildren($matchFields, $item['children']);
			}
		}

		return $results;
	}

/**
 * Deletes the records met by the given conditions including conditions for the parent
 *
 * @param array $conditions       Conditions array to delete a record by
 * @param array $parentConditions Conditions array to identify the parent
 * @param array $cascade          Delete dependants
 * @param array $callbacks        Trigger callbacks
 * @return bool                   Returns true if the delete suceeds or the record doesn't exists
 */
	public function deleteAllWithParent($conditions, $parentConditions, $cascade = true, $callbacks = false) {
		$parentId = $this->Model->field($this->Model->alias . '.id', $parentConditions);

		if (!empty($parentId)) {
			$conditions = Hash::merge($conditions, [
				$this->Model->alias . '.parent_id' => $parentId
			]);
			// Do the usual now we've identified the parent
			return $this->deleteAll($conditions, $cascade, $callbacks);
		}

		return true;
	}

/**
 * Exports a menu to a migration
 *
 * @param int $id The id passed on the command line
 * @return bool True on success
 * @throws NotFoundException If the menu item cannot be found
 * @throws Exception If the item a menu links to using a model_id cannot be found
 */
	public function getExportMigration($id) {
		$migrationUp = PHP_EOL;
		$migrationDown = PHP_EOL;
		$migration = '';

		$migration .= '$MenuUtil = MigrationUtil::init(\'EvNavigation.Menu\');' . PHP_EOL;

		$current = $this->Model->findById($id);
		if (!empty($current)) {
			$stack = [$current];

			while (!empty($current['Menu']['parent_id'])) {
				$current = $this->Model->findById($current['Menu']['parent_id']);
				array_unshift($stack, $current);
			}
		} else {
			throw new NotFoundException();
		}

		$migration .= '$parentId = null;' . PHP_EOL;
		$parentName = null;
		foreach ($stack as $menu) {

			// If a menu item is linked via a model_id then we need to create a lookup for that as it may not be the same id on another server
			$migrationUp .= '$isActive = ' . $this->_migrationValue($menu['Menu']['is_active']) . ';' . PHP_EOL;
			if (!empty($menu['Menu']['model_id'])) {

				$modelWithPlugin = (!empty($menu['Menu']['plugin']) ? $menu['Menu']['plugin'] . '.' : '') . $menu['Menu']['model'];
				$MenuModel = EvClassRegistry::init($modelWithPlugin);
				$menuModel = $MenuModel->findById($menu['Menu']['model_id']);

				if (empty($menuModel)) {
					throw new Exception('The ' . $MenuModel->alias . ' with ID "' . $menu['Menu']['model_id'] . '" for menu item with name "' . $menu['Menu']['name'] . '" could not be found');
				}

				// Check for an identifying field
				if (!empty($menuModel[$MenuModel->alias]['internal_title'])) {
					$idField = 'internal_title';
				} elseif (!empty($menuModel[$MenuModel->alias]['internal_name'])) {
					$idField = 'internal_name';
				} elseif (!empty($menuModel[$MenuModel->alias]['system_name'])) {
					$idField = 'system_name';
				} elseif (!empty($menuModel[$MenuModel->alias]['slug'])) {
					$idField = 'slug';
				} else {
					$idField = $MenuModel->displayField;
				}

				$migrationUp .= '$MenuModel = EvClassRegistry::init(\'' . $modelWithPlugin . '\');' . PHP_EOL;
				$migrationUp .= '$modelId = $MenuModel->field(\'' . $MenuModel->alias . '.id' . '\', [\'' . $MenuModel->alias . '.' . $idField . '\' => ' . $this->_migrationValue($menuModel[$MenuModel->alias][$idField]) . ']);' . PHP_EOL;
				$migrationUp .= 'if (empty($modelId)) {' . PHP_EOL;
				$migrationUp .= '	echo \'Could not find model to link for menu item with name "' . $menu['Menu']['name'] . '"\' . PHP_EOL;' . PHP_EOL;
				$migrationUp .= '	$modelId = null;' . PHP_EOL;
				$migrationUp .= '	$isActive = false;' . PHP_EOL;
				$migrationUp .= '}' . PHP_EOL;
			} else {
				$migrationUp .= '$modelId = null;' . PHP_EOL;
			}

			unset(
				$menu['Menu']['id'],
				$menu['Menu']['parent_id'],
				$menu['Menu']['lft'],
				$menu['Menu']['rght'],
				$menu['Menu']['created'],
				$menu['Menu']['modified'],
				$menu['Menu']['model_id'],
				$menu['Menu']['is_active']
			);

			if (is_array($menu['Menu']['url'])) {
				// We're only interested in specific URLs, not cake paths.
				unset ($menu['Menu']['url']);
			}

			$migrationUp .= '$parentId = $MenuUtil->createOnce([' . PHP_EOL;
			$migrationUp .= '	\'name\' => ' . $this->_migrationValue($menu['Menu']['name']) . ',' . PHP_EOL;
			$migrationUp .= '	\'parent_id\' => $parentId,' . PHP_EOL;
			$migrationUp .= '], [' . PHP_EOL;
			foreach ($menu['Menu'] as $field => $value) {
				$migrationUp .= '	\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
			}
			$migrationUp .= '	\'model_id\' => $modelId,' . PHP_EOL;
			$migrationUp .= '	\'parent_id\' => $parentId,' . PHP_EOL;
			$migrationUp .= '	\'is_active\' => $isActive,' . PHP_EOL;
			$migrationUp .= ']);' . PHP_EOL;
			$parentName = $menu['Menu']['name'];

			// DOWN
			$migrationDown .= '$parentId = $MenuUtil->Model->field(\'Menu.id\', [\'Menu.parent_id\' => $parentId, \'Menu.name\' => ' . $this->_migrationValue($menu['Menu']['name']) . ']);' . PHP_EOL;
		}

		// Only actually delete the one that was sepecifically created by this migration.
		// This may leave behind parent elements that are no longer needed but is better
		// than deleting things that may be needed elsewhere
		$migrationDown .= '$MenuUtil->Model->delete($parentId);' . PHP_EOL;

		return [
			'common' => $migration,
			'up' => $migrationUp,
			'down' => $migrationDown,
		];
	}
}
