<?php
/**
 * Menu controller for the Navigation plugin.
 *
 * @author  Andy Carter
 * @package Navigation
 */
App::uses('EvNavigationAppController', 'EvNavigation.Controller');
App::uses('ArrayUtil', 'EvCore.Lib');
App::uses('Folder', 'Utility');
App::uses('InflectorExt', 'EvInflector.Lib');

class MenusController extends EvNavigationAppController {

	public $adminSaveOptions = [
		'deep' => true,
		'checkUrl' => true
	];

	public function __construct($request = null, $response = null) {
		parent::__construct($request, $response);

		$this->adminActions[] = 'admin_move_up';
		$this->adminActions[] = 'admin_move_down';
	}

/**
 * Prepare data for populating admin dropdowns.
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$parents = $this->Menu->generateTreeList(
			null,
			null,
			null,
			'--'
		);

		// search for all the content elements in the elements/menu folder to save having to enter manually
		// can provide a nice dropdown instead
		$contentElements = array();

		$dir = new Folder(APP . 'View' . DS . 'Elements' . DS . 'menu');
		$files = $dir->find('.*\.ctp');

		foreach ($files as $file) {

			$filename = substr($file, 0, -4);
			$contentElements['menu/' . $filename] = $filename;
		}

		$this->set(compact('parents', 'contentElements'));
	}

/**
 * Prepare the admin form fields for the menu item form
 * @return array
 */
	protected function _adminFormFields() {
		$Model = $this->{$this->modelClass};

		$fields = parent::_adminFormFields();

		$fields[$Model->alias . '.url']['label'] = 'URL';
		$fields[$Model->alias . '.new_window']['label'] = 'Open Link in New Window';

		// store the array routing fields incase we are an administrator
		$adminOnly = array(
			$Model->alias . '.plugin' => $fields[$Model->alias . '.plugin'],
			$Model->alias . '.model' => $fields[$Model->alias . '.model'],
			$Model->alias . '.controller' => $fields[$Model->alias . '.controller'],
			$Model->alias . '.action' => $fields[$Model->alias . '.action'],
			$Model->alias . '.model_id' => $fields[$Model->alias . '.model_id'],
			$Model->alias . '.content_element' => $fields[$Model->alias . '.content_element'],
			$Model->alias . '.permission_level' => $fields[$Model->alias . '.permission_level'],
		);

		unset(
			$fields[$Model->alias . '.model'],
			$fields[$Model->alias . '.action'],
			$fields[$Model->alias . '.model_id'],
			$fields[$Model->alias . '.plugin'],
			$fields[$Model->alias . '.controller'],
			$fields[$Model->alias . '.content_element'],
			$fields[$Model->alias . '.permission_level'],
			$fields[$Model->alias . '.level']
		);

		if ($this->Permissions->hasSuperAdminPermission($this->Auth->user())) {

			// rejig the form, just for looks
			$fields = ArrayUtil::addAfter(
				$fields,
				$Model->alias . '.class',
				$adminOnly
			);

			$fields[$Model->alias . '.model_id']['type'] = 'string';
			$fields[$Model->alias . '.model_id']['label'] = 'Item ID';
			$fields[$Model->alias . '.controller']['info'] = 'Use Plugin Notation to link to a Plugin Controller';
			$fields[$Model->alias . '.content_element']['type'] = 'select';
		} else {
			$fields[$Model->alias . '.system_name']['type'] = 'hidden';
		}

		if (!empty(Configure::read('EvNavigation.use_acl_permissions'))) {
			//If we are using the acl permissions on menus then we don't want the permission level to be editable.
			if (isset($fields[$Model->alias . '.permission_level'])) {
				unset($fields[$Model->alias . '.permission_level']);
			}
		} else {
			//If we aren't using the acl permissions on menus then we don't want the login requirement to be editable.
			if (isset($fields[$Model->alias . '.requires_login'])) {
				unset($fields[$Model->alias . '.requires_login']);
			}
		}

		return $fields;
	}

/**
 * Menus admin listings page.
 * @return void
 */
	public function admin_index() {
		$Model = $this->{$this->modelClass}->alias;

		$this->set('toolbar', $this->_adminIndexToolbar());

		$data = $this->{$this->modelClass}->generateTreeList(null, null, null, '--');
		$this->set('data', $data);

		$this->set('title_for_layout', InflectorExt::pluralize($this->$Model->displayName));

		$this->view = 'EvNavigation.Menus/admin_index';

		return;
	}

/**
 * Menu item edit page.
 * @param int $id menu item ID
 */
	public function admin_edit($id = null) {
		$alias = $this->{$this->modelClass}->alias;
		parent::admin_edit($id);

		// If we're dealing with a menu item associated to another model then we want to warn the
		// user that supplying a URL will remove the association. We check this against the database
		// in case we've got incomplete POST data (i.e. validation has failed).
		$menuItem = $this->{$this->modelClass}->findById($id);
		if (! empty($menuItem[$alias]['model']) && !empty($menuItem[$alias]['url']) && is_array($menuItem[$alias]['url'])) {
			$linkedPageUrl = Router::url($menuItem[$alias]['url'] + ['admin' => false]);

			$this->viewVars['fields'][$alias . '.url']['displayInfo'] = 'This menu item is already <a href="' . $linkedPageUrl . '" target="_blank">linked to a page</a>, adding a URL here will replace the existing link';
		}

		// Can't add to self.
		// @TODO filter out children of self.
		unset($this->viewVars['parents'][$id]);
	}

	public function admin_move_up($id, $delta = 1) {
		if ($delta > 0) {

			if ($this->Menu->moveUp($id, abs($delta))) {

				$this->Session->setFlash(array(
					'title' => 'Item moved',
					'description' => 'Menu item has been moved up'
				), 'flash_success');

			} else {

				$this->Session->setFlash(array(
					'title' => 'Move failed',
					'description' => 'Failed to move menu item!'
				), 'flash_fail');

			}

		}

		$this->redirect(array('action' => 'index'));

		return;
	}

	public function admin_move_down($id, $delta = 1) {
		if ($delta > 0) {

			if ($this->Menu->moveDown($id, abs($delta))) {

				$this->Session->setFlash(array(
					'title' => 'Item moved',
					'description' => 'Menu item has been moved down'
				), 'flash_success');

			} else {

				$this->Session->setFlash(array(
					'title' => 'Move failed',
					'description' => 'Failed to move menu item!'
				), 'flash_fail');

			}

		}

		$this->redirect(array('action' => 'index'));

		return;
	}
}
