<?php

App::uses('AppHelper', 'View/Helper');

App::uses('Menu', 'EvNavigation.Model');

/**
 * Navigation Helper
 *
 * To create a formatted menu pass the return from Menu::getMenu($id):-
 *
 * $this->Navigation->menu($menu, $attr = array());
 *
 * You can check the active state of an individual menu item:-
 *
 * $this->Navigation->checkActive($item);
 *
 * @author  Andy Carter
 * @package Navigation
 */
class NavigationHelper extends AppHelper {

/**
 * Helper dependencies
 *
 * @var array
 * @access public
 */
	public $helpers = array('Html', 'Image', 'Session');

	public $settings = array(
		'tag' => 'li',
		'wrapTag' => 'ul',
		'childWrapTag' => 'div',
		'childrenClass' => 'menu-item--has-children',
		'activeClass' => 'menu-item--active',
		'activeTrailClass' => 'menu-item--active-trail',
		'itemClassPrefix' => 'menu-item--',
		'model' => 'Menu',
		'childrenImageClass' => 'menu-item--has-children-image',
		'depth' => null,
		'prepend' => null,
		'append' => null
	);

	public function __construct(View $View, $settings = array()) {
		parent::__construct($View, $settings);
		$this->settings = array_merge($this->settings, (array)$settings);
		$this->user_permission_level = intval($this->Session->read('Auth.User.UserGroup.level'));
		return;
	}

	/**
	 * Returns a formatted menu
	 * @param  [type] $data [description]
	 * @param  array  $attr [description]
	 * @return [type]       [description]
	 */
	public function menu($data, $attr = array()) {
		// Set the depth of menu to return.
		$attr = array_merge($this->settings, (array)$attr);
		$depth = $attr['depth'];

		// Build up the menu with the active and active_trail flags set.
		$this->buildMenu($data, $depth, $attr);

		// Return the marked up menu.
		return $this->_formatMenu($data, $attr, true);
	}

	/**
	 * Build up the menu array with the correct active/active-trail flags set.
	 * @param  array 	$data  menu data
	 * @param  integer 	$depth depth to recursively run the method for
	 *                         reduces by 1 each run (set to null to
	 *                         build the entire menu)
	 * @param 	array 	$attr  settings data merged with passed overwritten parameters
	 * @param  integer  $activeParent	ID of parent of active item
	 * @return array
	 */
	public function buildMenu(&$data, $depth = null, $attr = array(), &$activeParent = false) {
		foreach ($data as &$item) {

			$hasChildren = isset($item['children']) && !empty($item['children']);

			if ($hasChildren && ($depth === null | $depth > 1)) {

				$this->buildMenu(
						$item['children'], ($depth === null ? null : $depth - 1), $attr, $activeParent
				);
			} else {

				// We've reached the maximum depth of the current menu so
				// get rid of the children.
				unset($item['children']);
			}

			$item[$attr['model']]['active_trail'] = false;
			// Check for active states

			if ($this->checkActive($item, $attr['model'])) {

				$item[$attr['model']]['active'] = true;
				$activeParent = $item[$attr['model']]['parent_id'];
			} elseif ($activeParent === $item[$attr['model']]['id']) {

				$item[$attr['model']]['active_trail'] = true;
				$activeParent = $item[$attr['model']]['parent_id'];
			}
		}

		return $data;
	}

	/**
	 * Build up the menu recursively.
	 * @param  array   $data    array containing menu data
	 * @param  array   $attr
	 * @param  boolean $primary true on the first call, false thereafter to
	 *                          indicate child menu
	 * @return string  formatted menu
	 */
	protected function _formatMenu($data, $attr, $primary = false) {
		$return = '';

		if (!$primary && !empty($attr['childWrapTag'])) {

			$childWrapClass = '';

			if (! empty($attr['childWrapClass'])) {

				$childWrapClass = ' class="' . $attr['childWrapClass'] . '"';
			}

			$return .= "<{$attr['childWrapTag']}{$childWrapClass}>";
		}

		$id = '';
		$classes = array();

		if ($primary && isset($attr['class']) && !empty($attr['class'])) {

			$classes[] = $attr['class'];
		}

		if ($primary && isset($attr['id']) && !empty($attr['id'])) {

			$id = " id='{$attr['id']}'";
		}

		$class = implode(' ', $classes);

		if (! $primary && ! empty($attr['wrapClass'])) {
			$class .= $attr['wrapClass'];
		}

		$dataAttr = '';
		if ($primary && isset($attr['data']) && !empty($attr['data'])) {
			foreach ($attr['data'] as $k => $v) {
				$dataAttr .= ' data-' . $k . '="' . $v . '"';
			}
		}

		if (! empty($attr['wrapTag'])) {
			$return .= "<{$attr['wrapTag']}$id class='$class'{$dataAttr}>";
		}

		$allHidden = true;
		foreach ($data as $item) {
			if (! empty($item[$attr['model']]['permission_level']) && $item[$attr['model']]['permission_level'] < $this->user_permission_level) {
				continue;
			}
			$allHidden = false;

			if (isset($item['children'][0][$attr['model']]['is_menu_hidden'])) {
				$hasChildren = isset($item['children']) && !empty($item['children']) && !$item['children'][0][$attr['model']]['is_menu_hidden'];
			} else {
				$hasChildren = false;
			}

			$hasChildrenImage = false;
			if (isset($item['children'][0]['MenuImage'][0]['id'])) {
				$hasChildrenImage = true;
			}
			$children = $hasChildren ? $this->_formatMenu($item['children'], $attr, false) : null;

			// Work out the classes to apply to the current menu item.

			$classes = array();

			if (isset($item[$attr['model']]['id']) && !empty($attr['itemClassPrefix'])) {

				$classes[] = $attr['itemClassPrefix'] . $item[$attr['model']]['id'];
			}

			if (isset($item[$attr['model']]['class']) && !empty($item[$attr['model']]['class'])) {

				$classes[] = $item[$attr['model']]['class'];
			}

			if ($hasChildren && !empty($attr['childrenClass'])) {

				$classes[] = $attr['childrenClass'];
			}

			if ($hasChildrenImage && !empty($attr['childrenImageClass'])) {

				$classes[] = $attr['childrenImageClass'];
			}

			if ($item[$attr['model']]['active'] && !empty($attr['activeClass'])) {

				$classes[] = $attr['activeClass'];
			}

			if ($item[$attr['model']]['active_trail'] && !empty($attr['activeTrailClass'])) {

				$classes[] = $attr['activeTrailClass'];
			}

			$class = implode(' ', $classes);
			$return .= "<{$attr['tag']} class='$class'>";

			if (
				! empty($item[$attr['model']]['content_element']) &&
				$this->_View->elementExists($item[$attr['model']]['content_element'])
			) {
				$return .= $this->_buildContent($item, $attr);
			} else {

				$return .= $this->_buildLink($item, $attr, $hasChildren, $children);
			}

			$return .= "</{$attr['tag']}>";
		}

		if (! empty($attr['wrapTag'])) {
			$return .= "</{$attr['wrapTag']}>";
		}

		if (!$primary && !empty($attr['childWrapTag'])) {

			$return .= "</{$attr['childWrapTag']}>";
		}

		if ($allHidden) {
			$return = '';
		}

		return $return;
	}

	/**
	 * Check for active state of a menu item. Used by _formatMenu() and can be
	 * called from within a view when wanting to play with the unprocessed
	 * menu array.
	 * @param  array $item 	menu item
	 * @param  string $menu_model the menu model as defined by settings or as overwritten by user
	 * @return boolean      true if active
	 */
	public function checkActive($item, $menuModel) {
		if (isset($item[$menuModel]['active']) && $item[$menuModel]['active']) {
			return true;
		} else {

			return $this->_checkActiveByRoute($item, $menuModel);
		}
	}

	/**
	 * Check the menu item against the current route.
	 * @param  array $item	menu item
	 * @param  string $menu_model the menu model as defined by settings or as overwritten by user
	 * @return boolean		true if there is a match
	 */
	protected function _checkActiveByRoute($item, $menuModel) {
		$isActive = false;

		if (($pos = strrpos($this->here, '/page:')) !== false) {
			$this->here = substr($this->here, 0, $pos);
		}
		// Check if the menu item's URL matches the current page.
		if (!$isActive && $item[$menuModel]['url']) {

			$isActive = Router::normalize($this->here) === Router::normalize($item[$menuModel]['url']);
		}

		// Check if the menu item's pattern matches the current page.
		if (!$isActive && $item[$menuModel]['pattern']) {

			$isActive = preg_match($item[$menuModel]['pattern'], Router::normalize($this->here));
		}

		return $isActive;
	}

	/**
	 * build a link menu item
	 *
	 * @param 	array 	menu item from database we are building link for
	 * @param  	array   $attr
	 * @param 	bool 	whether the item has children
	 * @param 	string 	the formatted child elements
	 * @return 	string
	 */
	protected function _buildLink($item, $attr, $hasChildren, $children) {
		$return = '';

		$addImage = '';
		if (isset($item['MenuImage'][0])) {
			$addImage = $this->Image->resize($item['MenuImage'][0], array('width' => 100, 'height' => 100, 'crop' => true));
		}

		$name = $item[$attr['model']]['name'];

		if (isset($attr['prepend']) && ! empty($attr['prepend'])) {
			$name = $attr['prepend'] . $name;
		}

		if (isset($attr['append']) && ! empty($attr['append'])) {
			$name = $name . $attr['append'];
		}

		$linkAttr = array(
			'escape' => false,
			'target' => (isset($item[$attr['model']]['new_window']) && $item[$attr['model']]['new_window'] ? '_blank' : null)
		);

		if (isset($hasChildren) && $hasChildren === true && ! empty($attr['childrenLinkAttr'])) {
			$linkAttr = array_merge($linkAttr, $attr['childrenLinkAttr']);
		}

		$return .= $this->Html->link(
			$addImage .	$name,
			$item[$attr['model']]['url'],
			$linkAttr
		);

		if ($hasChildren) {
			$return .= $children;
		}

		return $return;
	}

	/**
	 * build a content element li rather then a link
	 *
	 * @param 	array 	menu item from database we are building link for
	 * @param  array   $attr
	 * @return 	string
	 */
	protected function _buildContent($item, $attr) {
		return $this->_View->element(
			$item[$attr['model']]['content_element'],
			array(
				'item' => $item,
				'attr' => $attr
			)
		);
	}
}
