<?php
/**
 * Menu model for the Navigation plugin.
 *
 * @author  Andy Carter
 * @package Navigation
 */
App::uses('EvNavigationAppModel', 'EvNavigation.Model');

class Menu extends EvNavigationAppModel {

	// Menus have a tree behaviour.
	public $actsAs = array(
		'Tree' => array(
			'level' => 'level',
		),
	);

	public $order = array('lft' => 'ASC');

	// We want to keep a record of which menu items are currently for use when
	// building our menus.
	protected $_menuActiveId = null;

	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$this->removeBehaviour('CustomCopyable');

		return;
	}

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'cannot be blank'
			),
			'unique' => array(
				'rule' => array('checkCombinationUnique', array('parent_id')),
				'message' => 'A menu item with this name already exists for this parent'
			)
		)
	);

/**
 * Before save
 * @param array $options Save options
 * @return bool
 */
	public function beforeSave($options = []) {
		// If the 'checkUrl' flag is passed we want to check if the `url` field has been populated;
		// if it has then we want to clear any polymorphic relation.
		if (!empty($options['checkUrl']) && !empty($this->data[$this->alias]['url'])) {
			$this->data[$this->alias] += [
				'plugin' => null,
				'model' => null,
				'controller' => null,
				'model_id' => null,
				'action' => null
			];
		}

		return parent::beforeSave($options);
	}

/**
 * Called after each successful save operation.
 *
 * @param bool $created True if this save created a new record
 * @param array $options Options passed from Model::save().
 * @return void
 */
	public function afterSave($created = false, $options = array()) {
		Cache::clearGroup('Menus', 'EvNavigation_Menus');

		parent::afterSave($created, $options);
	}

/**
 * Returns a menu with active menu items flagged as 'active'.
 * @param  integer $id       ID of menu to return
 * @param  integer $activeId optional active menu item ID
 * @return array             array of menu items
 */
	public function getMenu($id, $activeId = null) {
		return Cache::remember('menu_data_' . $id . '_' . $activeId, function () use ($id, $activeId) {
			// Set the ID of the active menu item ready for searching the
			// menu with.
			$activeId = $activeId === null ? $this->_menuActiveId : $activeId;

			// Get the menu record so that we can work out its children
			// using the left and right columns.
			$menu = $this->findById($id);

			// Get the menu's children.
			$data = $this->find('threaded', array(
				'conditions' => array(
					'lft >' => $menu[$this->alias]['lft'],
					'rght <' => $menu[$this->alias]['rght'],
					'is_active' => true
				),
				'order' => array(
					'lft' => 'ASC'
				)
			));

			// Search for the active menu item in the menu data and mark it
			// as active. Then return the menu data.
			return $this->_findActive($data, $activeId);
		}, 'EvNavigation_Menus');
	}

	protected function _findActive(&$data, $activeId) {
		foreach ($data as &$item) {
			$item[$this->alias]['active'] = ($item[$this->alias]['id'] == $activeId);

			if (isset($item['children'])) {
				$this->_findActive($item['children'], $activeId);
			}
		}

		return $data;
	}

/**
 * Set the active menu item
 * @param integer $id ID of menu item to mark as active
 */
	public function setActive($id) {
		$this->_menuActiveId = $id;

		return true;
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as &$result) {
			// override the url with array based routing if a controller is set
			if (
				empty($result[$this->alias]['url'])
				&& isset($result[$this->alias]['controller'])
				&& !empty($result[$this->alias]['controller'])
			) {

				list($plugin, $controller) = pluginSplit($result[$this->alias]['controller']);

				$result[$this->alias]['url'] = array(
					'plugin' => $plugin ?: $result[$this->alias]['plugin'],
					'controller' => $controller,
					'action' => $result[$this->alias]['action']
				);

				// add model id if one is set
				if (isset($result[$this->alias]['model_id']) && !empty($result[$this->alias]['model_id'])) {

					$result[$this->alias]['url'][] = $result[$this->alias]['model_id'];
				}
			}
		}

		return $results;
	}

/**
 * Returns the child menu items of a menu in tree form
 *
 * @param  mixed $parentId parent menu ID or an array of IDs
 * @param  integer $id       ID of a menu item to exclude
 * @return array
 */
	public function findTreeBranches($parentId, $id = null) {
		$conditions = array();

		if (is_numeric($parentId) && $parentId != 0) {
			$menu = $this->findById($parentId);

			$conditions = array(
				'lft >=' => $menu[$this->alias]['lft'],
				'rght <=' => $menu[$this->alias]['rght'],
				'is_active' => true
			);
		} elseif (is_array($parentId)) {

			foreach ($parentId as $thisParentId) {
				$menu = $this->findById($thisParentId);

				$conditions['OR'][] = array(
					'lft >=' => $menu[$this->alias]['lft'],
					'rght <=' => $menu[$this->alias]['rght'],
					'is_active' => true
				);
			}
		}

		if (! empty($id)) {
			$menu = $this->findById($id);

			$conditions['NOT'] = array(
				'AND' => array(
					'lft >=' => $menu[$this->alias]['lft'],
					'rght <=' => $menu[$this->alias]['rght']
				)
			);
		}

		return $this->generateTreeList(
			$conditions,
			null,
			null,
			'--'
		);
	}
}
