<?php
/**
 * Navigatable Behaviour
 *
 * Associates a model with the Menu.
 */
App::uses('Menu', 'EvNavigation.Model');
App::uses('InflectorExt', 'EvInflector.Lib');
App::uses('AppBehavior', 'Model/Behavior');

class NavigatableBehavior extends AppBehavior {

	public $settings = array();

	/**
	 * Initialise the behavior - store the settings against the model
	 *
	 * @see ModelBehavior::setup()
	 */
	public function setup(Model $Model, $settings = array()) {
		// set the defaults
		$defaults = array(
			'is_menu_hidden' => false,
			'parent_id' => false,
			'name' => false,
			'auto_create' => false,
			'is_breadcrumb_hidden' => false,
			'plugin' => false,
			'controller' => InflectorExt::tableize($Model->alias),
			'action' => 'view',
			'model' => $Model->alias,
			'redirect' => null
		);

		// if a config exists, load up the config which defines the route parameters
		if (isset($settings['config']) && ! empty($settings['config'])) {

			$route = Configure::read($settings['config'] . '.route');

			// check to see if multiple routes based on models
			if (isset($route[$Model->alias])) {

				$routeSettings = $route[$Model->alias];
			} elseif (
				isset($route['controller']) &&
				isset($route['action']) &&
				isset($route['plugin'])
			) {
				// or standard for just one model
				$routeSettings = $route;
			}

			// check to make sure we found what we want
			if (isset($routeSettings)) {

				$defaults = array_merge($defaults, $routeSettings);
			}
		}

		$this->settings[$Model->alias] = array_merge(
			$defaults,
			(array)$settings
		);

		return;
	}

	/**
	 * get menu item for a given item. Used on readForEdit only
	 * to prevent tons of queries happen
	 *
	 * @param Object 	Model Object behaviour was called from
	 * @param array 	array of data we want to get the menu item for
	 * @return array 	Either array of menu data or blank array
	 */
	public function getMenuItem(Model $Model, $data) {
		$Menu = EvClassRegistry::init('EvNavigation.Menu');

		if (isset($data[$Model->alias]) && ! empty($data[$Model->alias]['id'])) {

			$conditions = array(
				'model' => $this->settings[$Model->alias]['model'],
				'model_id' => $data[$Model->alias]['id']
			);

			// plugin is set, add to the conditions so we bring out the correct item
			if (isset($this->settings[$Model->alias]['plugin']) && ! empty($this->settings[$Model->alias]['plugin'])) {

				$conditions['plugin'] = $this->settings[$Model->alias]['plugin'];
			}

			$menuItem = $Menu->find('first', array(
				'conditions' => $conditions
			));

			if (! empty($menuItem['Menu']['id'])) {

				return $menuItem['Menu'];
			}
		}

		return array();
	}

	/**
	 * Prepare the data for saving.
	 *
	 * @see ModelBehavior::beforeSave()
	 */
	public function beforeSave(Model $Model, $options = array()) {
		$response = parent::beforeSave($Model, $options);

		if (isset($Model->data[$Model->alias]['Menu'])) {

			$Model->data['Menu'] = $Model->data[$Model->alias]['Menu'];
			unset($Model->data[$Model->alias]['Menu']);
		}

		return $response;
	}

	/**
	 * Save the menu
	 *
	 * @see ModelBehavior::afterSave()
	 */
	public function afterSave(Model $Model, $created, $options = array()) {
		Cache::clearGroup('Menus', 'EvNavigation_Menus');

		parent::afterSave($Model, $created, $options);

		// If we've got menu data we want to save it, unless this is a new menu
		// item that has been set as in active (basically it's not getting
		// added to the menu in this case).

		if ($this->settings[$Model->alias]['auto_create']) {
			if ($this->settings[$Model->alias]['name'] == ':displayField') {
				$name = $Model->data[$Model->alias][$Model->displayField];
			} else {
				$name = $this->settings[$Model->alias]['name'];
			}

			$Model->data['Menu']['name'] = $name;
			$Model->data['Menu']['plugin'] = $this->settings[$Model->alias]['plugin'];
			$Model->data['Menu']['parent_id'] = $this->settings[$Model->alias]['parent_id'];
			$Model->data['Menu']['is_active'] = true;
			$Model->data['Menu']['url'] = Router::url(array(
				'admin' => false,
				'controller' => $this->settings[$Model->alias]['controller'] || InflectorExt::tableize($Model->alias),
				'action' => $this->settings[$Model->alias]['action'],
				$Model->id
			));
		}

		if (isset($Model->data['Menu']) && ($Model->data['Menu']['is_active'] || $Model->data['Menu']['id'])) {

			$Model->data['Menu']['plugin'] = $this->settings[$Model->alias]['plugin'];
			$Model->data['Menu']['controller'] = $this->settings[$Model->alias]['controller'];
			$Model->data['Menu']['action'] = $this->settings[$Model->alias]['action'];
			$Model->data['Menu']['model'] = $this->settings[$Model->alias]['model'];
			$Model->data['Menu']['model_id'] = $Model->id;
			if (
				$this->settings[$Model->alias]['redirect']
				&& ! empty($Model->data[$Model->alias][$this->settings[$Model->alias]['redirect']])
			) {
				$Model->data['Menu']['url'] = $Model->data[$Model->alias][$this->settings[$Model->alias]['redirect']];
			}

			$Menu = new Menu();
			$Menu->create();
			$Menu->save($Model->data);
		}

		return;
	}

	/**
	 * delete menu item
	 *
	 */
	public function afterDelete(Model $Model) {
		parent::afterDelete($Model);

		$conditions = array(
			'Menu.model' => $this->settings[$Model->alias]['model'],
			'Menu.model_id' => $Model->id,
		);

		if (! empty($this->settings[$Model->alias]['plugin'])) {

			$conditions['Menu.plugin'] = $this->settings[$Model->alias]['plugin'];
		}

		$Menu = new Menu();
		$Menu->deleteAll($conditions);
	}
}
