<?php

/**
 * This is the base migration util containing generic migration helper methods
 * Class/Plugin specific migration helpers can extend from this one
 */
class BaseMigrationModelUtil {

	public $Model = null;

/**
 * Constructor for a model migration helper
 *
 * @param string $modelName Name of base model for the helper which will be loaded through EvClassRegistry::init
 */
	public function __construct($modelName) {
		$this->Model = EvClassRegistry::init($modelName);
	}

/**
 * Function to only create a record if it doesn't already exist
 *
 * @param array $conditions Conditions array to check if the record already exists or not
 * @param array $data       Data to pass into Model->saveAssociated if the record doesn't already exist
 * @return int|bool         Primary key of the created or existing record, false if it fails
 */
	public function createOnce($conditions, $data) {
		// Check if the record already exists
		$existingId = $this->Model->field($this->Model->primaryKey, $conditions);
		if (!empty($existingId)) {
			return $existingId;
		}

		// Record didn't exist so create it
		$this->Model->create();
		$success = (bool)$this->Model->saveAssociated($data, ['deep' => true]);
		if (!$success) {
			debug($this->Model->validationErrors);
		}

		return $success ? $this->Model->id : false;
	}

/**
 * Function to only create a group of records if they don't already exist
 *
 * @param array $matchFields Fields to match to check if a field already exists
 * @param array $data        An array of individual records to pass into Model->saveAssociated if the record doesn't already exist
 * @return array          	 An array of ids to match the original array order
 */
	public function createAllOnce($matchFields, $data) {
		$results = [];

		foreach ($data as $key => $item) {
			$conditions = [];

			// Build the conditions from the field list
			foreach ($matchFields as $field) {
				if (isset($item[$this->Model->alias])) {
					$conditions[$this->Model->alias . '.' . $field] = $item[$this->Model->alias][$field];
				} else {
					$conditions[$this->Model->alias . '.' . $field] = $item[$field];
				}
			}

			$results[$key] = $this->createOnce($conditions, $item);
		}

		return $results;
	}

/**
 * Update a single record with data. If a record can't be found with the provided conditions then nothing will be
 * updated.
 *
 * @param array $conditions Conditions array to check if the record already exists or not.
 * @param array $data       Data to pass into Model->saveAssociated if the record doesn't already exist.
 * @return int|bool Primary key of the existing record, false if it fails.
 */
	public function updateSingle($conditions, $data) {
		$existingRecord = $this->Model->find(
			'first',
			[
				'conditions' => $conditions
			]
		);
		if (empty($existingRecord)) {
			debug('Could not find a record of ' . $this->Model->alias . ' to update with conditions: ' . json_encode($conditions));
			return;
		}

		if (empty($data[$this->Model->alias])) {
			$data = [$this->Model->alias => $data];
		}

		$data = Hash::merge($existingRecord, $data);

		$success = $this->Model->saveAssociated($data, ['deep' => true]);

		if (!$success) {
			debug($this->Model->validationErrors);
		}

		return $success ? $this->Model->id : false;
	}

/**
 * Update a group of records. Records are only updated if they are found, no new records will be created.
 *
 * @param array $matchFields Fields to match to check if a field already exists.
 * @param array $data        An array of individual records to pass into Model->saveAssociated if the record doesn't already exist.
 * @return array An array of ids to match the original array order.
 */
	public function updateMultiple($matchFields, $data) {
		$results = [];

		foreach ($data as $key => $item) {
			$conditions = [];

			// Build the conditions from the field list
			foreach ($matchFields as $field) {
				if (isset($item[$this->Model->alias])) {
					$conditions[$this->Model->alias . '.' . $field] = $item[$this->Model->alias][$field];
				} else {
					$conditions[$this->Model->alias . '.' . $field] = $item[$field];
				}
			}

			$results[$key] = $this->updateSingle($conditions, $item);
		}

		return $results;
	}

/**
 * Deletes the first record met by the given conditions
 *
 * @param array $conditions Conditions array to delete a record by
 * @return bool             Returns true if the delete suceeds or the record doesn't exists
 */
	public function deleteIfExists($conditions) {
		// Check if the record already exists
		if ($record = $this->Model->find('first', ['conditions' => $conditions])) {
			return $this->Model->delete($record[$this->Model->alias][$this->Model->primaryKey]);
		}

		return true;
	}

/**
 * Deletes all records that meet the conditions, alais for Model::delete
 *
 * @param array $conditions Conditions array to delete records by
 * @param bool $cascade Set to true to delete records that depend on this record
 * @param bool $callbacks Run callbacks
 * @return bool True on success, false on failure
 * @link http://book.cakephp.org/2.0/en/models/deleting-data.html#deleteall
 */
	public function deleteAll($conditions, $cascade = true, $callbacks = false) {
		return $this->Model->deleteAll($conditions, $cascade, $callbacks);
	}

/**
 * Convert a value to one that can be safely added to a migration
 *
 * @param mixed $val A value
 * @return string The string version to be added to a migration
 */
	protected function _migrationValue($val) {
		if ($val === null) {
			return 'null';
		} elseif (is_bool($val)) {
			return $val ? 'true' : 'false';
		} elseif (is_numeric($val)) {
			return $val;
		} elseif (is_array($val)) {
			return '\'' . json_encode($val) . '\'';
		} else {
			return '\'' . str_replace('\'', '\\\'', $val) . '\'';
		}
	}

/**
 * Exports a page and creates a migration
 *
 * @param string $identifier The id/system_name passed on the command line
 * @return bool True on success
 * @throws Exception If this function has not been overridden and an exporter defined
 */
	public function getExportMigration($identifier) {
		throw new Exception('No export method has been defined for this model');
	}

}
