<?php

App::uses('AppShell', 'Console/Command');
App::uses('MigrationUtil', 'EvMigrationUtil.Lib');

class MigrationShell extends AppShell {

	protected $_migrationPath = ROOT . '/app/Config/Migration/';

/**
 * Calls export on a model's migration util.
 *
 * @return void
 */
	public function export() {
		$modelName = $this->args[0];
		$identifierString = $this->args[1];
		$Util = MigrationUtil::init($modelName);

		$identifiers = explode(',', $identifierString);

		// NOTE: Migrations cannot have the same timestamp or they will not be run. They should also be run in order of their timestamp
		// - If multiple migrations are being created then a timestamp offset will be added.
		// - If a migration already exists the timestamp will be offset to be after the last migration.

		// Get all migration filenames
		$filenames = glob($this->_migrationPath . '*.php');
		$currentTime = time();
		$timeOffset = 0;

		$this->out('Looking for migrations...');

		if (!empty($filenames)) {
			// Find the last timestamp used
			$lastFilename = end($filenames);

			$this->out('Found last migration at ' . $lastFilename);

			$match = [];
			preg_match('/\\/([0-9]+)_[^\\/\\\\]*\\.php/', $lastFilename, $matches);

			if (!empty($matches[1])) {
				$this->out('Last migration was at ' . $matches[1] . '. Current time is ' . $currentTime);

				if ($matches[1] >= $currentTime) {
					// Adjust the time offset to be after the last timestamp used
					$timeOffset = ($matches[1] - $currentTime) + 1;
					$this->out('Adding a ' . $timeOffset . ' second time offset to compensate');
				}

			}
		}

		foreach ($identifiers as $identifier) {
			$migration = $Util->getExportMigration($identifier);
			if (!empty($migration)) {
				$this->out('Exporting ' . $modelName . ' ' . $identifier . ' as a migration...');
				$this->_createMigration('Create' . InflectorExt::classify(str_replace('.', '_', $modelName . '_' . $identifier)), $migration, $currentTime + $timeOffset);
			}
			$timeOffset++;
		}

		$this->out('Done');
	}

/**
 * Add the available options
 *
 * @return Parser
 */
	public function getOptionParser() {
		$parser = parent::getOptionParser();
		$parser->addSubcommand('export', [
			'help' => 'Exports a model',
			'parser' => [
				'description' => [
					'This will export a model',
				],
				'arguments' => [
					'ModelName' => [
						'help' => 'The model name including a plugin prefix. (E.g. EvCore.Page or EvSitesSettings.SiteSetting)',
						'required' => true
					],
					'identifier' => [
						'help' => 'An identifier for the item to export. This varies per model. Multiple, comma separated identifiers can be used and this will create one migration per item.',
						'required' => true
					]
				]
			]
		]);
		return $parser;
	}

/**
 * Creates a migration with the $before code in the before function
 *
 * @param string $className The class name for the migration
 * @param string $migration The before code
 * @param int $timestamp The timestamp to use
 * @return bool True on success
 * @throws Exception If the template file is missing
 */
	protected function _createMigration($className, $migration, $timestamp) {
		$underscoredName = InflectorExt::underscore($className);

		$migrationTemplate = file_get_contents( App::pluginPath('EvMigrationUtil') . 'Lib/DefaultMigrationTemplate.php');
		if (!empty($migrationTemplate)) {
			$migrationTemplate = str_replace('DefaultMigrationTemplate', $className, $migrationTemplate);
			$migrationTemplate = str_replace('default_migration_template', $underscoredName, $migrationTemplate);

			if (!empty($migration['common'])) {
				$migrationTemplate = str_replace('\'MIGRATION_COMMON\';', $this->_indentBy(2, $migration['common']), $migrationTemplate);
			}

			if (!empty($migration['up'])) {
				$migrationTemplate = str_replace('\'MIGRATION_UP\';', $this->_indentBy(3, $migration['up']), $migrationTemplate);
			}

			if (!empty($migration['down'])) {
				$migrationTemplate = str_replace('\'MIGRATION_DOWN\';', $this->_indentBy(3, $migration['down']), $migrationTemplate);
			}

			return file_put_contents( $this->_migrationPath . ($timestamp) . '_' . $underscoredName . '.php', $migrationTemplate) !== false;
		} else {
			throw new Exception('Default migration template not found');
		}

		return false;
	}

/**
 * Convert a value to one that can be safely added to a migration
 *
 * @param int $num The number of tabs to pad each line
 * @param string $str The string to indent
 * @return string The string version to be added to a migration
 */
	protected function _indentBy($num, $str) {
		$padding = str_pad('', $num, "\t");
		return preg_replace('/\n(?=[^\n])/m', "\n" . $padding, $str);
	}

}
