<?php

class MetaDataLib {

	protected $_settings = []; //The meta settings used in the meta component.

	protected $_dataModel = null; //The model the meta data is associated to.

/**
 * Meta data constructor. Takes the settings form the meta component
 *
 * @param array $metaSettings MetaComponent settings.
 * @param obj   $dataModel    The model the meta data is associated with.
 */
	public function __construct($metaSettings, $dataModel = null) {
		$this->_settings = $metaSettings;
		$this->_dataModel = $dataModel;
	}

/**
 * Set the title for layout, either using the meta title, display field, or model name.
 *
 * @param array $data  The data passed through from the controller.
 * @param obj   $Model The model to check available fields.
 * @return string The meta title to set.
 */
	public function getTitleForLayout($data, $Model = null) {
		if (!empty($data['MetaData']['title'])) {
			return $data['MetaData']['title'];
		}

		return $this->getFallbackTitle($data, $Model);
	}

/**
 * If the meta data title isn't set then attempt to check other available fields. If the titleField setting is
 * available then that is used, if not then the model displayField is checked for availability and if that can't be
 * found then the model displayName will be used.
 *
 * @param array $data  The data passed through from the controller.
 * @param obj   $Model The model to check available fields.
 * @return string The fallback meta title to set.
 */
	public function getFallbackTitle($data, $Model = null) {
		if ($Model === null) {
			if ($this->_dataModel === null) {
				return ''; //Can't get fallback.
			}

			$Model = $this->_dataModel;
		}

		$titleForLayout = '';
		if (
			!empty($this->_settings['titleField']) &&
			!empty($data[$Model->alias][$this->_settings['titleField']])
		) {
			$titleForLayout = $data[$Model->alias][$this->_settings['titleField']];
		} elseif (isset($data[$Model->alias][$Model->displayField])) {
			$titleForLayout = $data[$Model->alias][$Model->displayField];
		} else {
			$titleForLayout = InflectorExt::pluralize($Model->displayName);
		}

		if (
			!empty($this->_settings['titleIncludesSiteTitle']) &&
			!empty(Configure::read('SiteSetting.general.site_title'))
		) {
			$titleForLayout .= ' | ' . Configure::read('SiteSetting.general.site_title');
		}

		return $titleForLayout;
	}

/**
 * Set the description for layout, either using the meta description, overridden description field, body or description
 * field.
 *
 * @param array $data  The data passed through from the controller.
 * @param obj   $Model The model to check available fields.
 * @return string The meta description to set.
 */
	public function getDescriptionForLayout($data, $Model = null) {
		if (!empty($data['MetaData']['description'])) {
			return $data['MetaData']['description'];
		}

		return $this->getFallbackDescription($data, $Model);
	}

/**
 * If the meta data description isn't set then attempt to get the description from other available fields. By default
 * the other fields checked are the "body" and "description" fields but other can be added through the settings.
 *
 * @param array $data  The data passed through from the controller.
 * @param obj   $Model The model to check available fields.
 * @return string The fallback meta title to set.
 */
	public function getFallbackDescription($data, $Model = null) {
		if ($Model === null) {
			if ($this->_dataModel === null) {
				return ''; //Can't get fallback.
			}

			$Model = $this->_dataModel;
		}

		$defaultFields = [
			'body',
			'description'
		];
		$fields = array_merge((array)$this->_settings['descriptionField'], $defaultFields);

		foreach ($fields as $field) {
			if (isset($data[$Model->alias][$field])) {

				// Strip the tags first as they're not needed
				$string = strip_tags($data[$Model->alias][$field]);

				// Trim the content so the regex doesn't crash apache. This was an issue with MEDIUMTEXT
				// fields in the db as trying to regex a huge string caused apache to segfault.
				$string = substr($string, 0, $this->_settings['descriptionLength'] * 4);

				// Replace new lines and replace whitespace with a single space
				$string = preg_replace(
					[
						"/\r|\n/",
						"/\s+/"
					],
					" ",
					$string
				);

				// Finally make it the correct length with ellipsis if required
				return CakeText::truncate($string, $this->_settings['descriptionLength']);
			}
		}

		return null;
	}

/**
 * Set the description for layout, using the meta keywords or null.
 *
 * @param array $data  The data passed through from the controller.
 * @param obj   $Model The model to check available fields.
 * @return string The meta keywords to set.
 */
	public function getKeywordsForLayout($data, $Model = null) {
		if (!empty($data['MetaData']['keywords'])) {
			$keywordsForLayout = $data['MetaData']['keywords'];
		} else {
			$keywordsForLayout = null;
		}

		return $keywordsForLayout;
	}
}
