<?php

App::uses('AppShell', 'Console/Command');

class UpdateMetaShell extends AppShell {

/**
 * Gets the option parser instance and configures it.
 *
 * @return ConsoleOptionParser
 * @link http://book.cakephp.org/2.0/en/console-and-shells.html#Shell::getOptionParser
 */
	public function getOptionParser() {
		$parser = parent::getOptionParser();

		$parser->addSubcommand(
			'replaceDescriptionWithToken',
			[
				'help' => 'Replace text in the meta data description with a meta data token',
				'parser' => $this->_getReplaceDescriptionWithTokenParser(),
			]
		);

		$parser->addSubcommand(
			'replaceTitleWithToken',
			[
				'help' => 'Replace text in the meta data title with a meta data token',
				'parser' => $this->_getReplaceTitleWithTokenParser(),
			]
		);

		return $parser;
	}

/**
 * Gets the option parser instance for the `replaceDescriptionWithToken` subCommand and configures it.
 *
 * @return ConsoleOptionParser
 * @link http://book.cakephp.org/2.0/en/console-and-shells.html#Shell::getOptionParser
 */
	protected function _getReplaceDescriptionWithTokenParser() {
		$parser = parent::getOptionParser();

		$parser->addArguments(
			[
				'modelClass' => [
					'help' => 'The class of the model you want to replace meta data for',
					'required' => true,
					'index' => 0,
				],
				'regex' => [
					'help' => 'The regex to use on the description to find text to replace',
					'required' => true,
					'index' => 1,
				],
				'token' => [
					'help' => 'The token to replace the found text in the description',
					'required' => true,
					'index' => 2,
				],
			]
		);

		return $parser;
	}

/**
 * Replace all the prices in product meta descriptions with the Product.price meta token.
 *
 * @return void.
 */
	public function replaceDescriptionWithToken() {
		return $this->_replaceMetaFieldWithToken($this->args[0], 'description', $this->args[1], $this->args[2]);
	}

/**
 * Gets the option parser instance for the `replaceTitleWithToken` subCommand and configures it.
 *
 * @return ConsoleOptionParser
 * @link http://book.cakephp.org/2.0/en/console-and-shells.html#Shell::getOptionParser
 */
	protected function _getReplaceTitleWithTokenParser() {
		$parser = parent::getOptionParser();

		$parser->addArguments(
			[
				'modelClass' => [
					'help' => 'The class of the model you want to replace meta data for',
					'required' => true,
					'index' => 0,
				],
				'regex' => [
					'help' => 'The regex to use on the description to find text to replace',
					'required' => true,
					'index' => 1,
				],
				'token' => [
					'help' => 'The token to replace the found text in the description',
					'required' => true,
					'index' => 2,
				],
			]
		);

		return $parser;
	}

/**
 * Replace all the prices in product meta descriptions with the Product.price meta token.
 *
 * @return void.
 */
	public function replaceTitleWithToken() {
		return $this->_replaceMetaFieldWithToken($this->args[0], 'title', $this->args[1], $this->args[2]);
	}

/**
 * Replace the text in a meta data field with a token.
 *
 * @param string $modelClass The class of the model to replace meta data for.
 * @param string $field      The meta data field to replace.
 * @param string $regex      The regex to use to find text to replace.
 * @param string $token      The token to replace the found text with.
 * @return null|bool Null if nothing was changed, true if successfully changed, false otherwise.
 */
	protected function _replaceMetaFieldWithToken($modelClass, $field, $regex, $token) {
		$Model = EvClassRegistry::init($modelClass);
		$MetaData = EvClassRegistry::init('MetaData.MetaData');

		$modelTable = $Model->table;
		if (!empty($Model->tablePrefix)) {
			$modelTable = $Model->tablePrefix . $modelTable;
		}

		$modelMetas = $MetaData->find(
			'all',
			[
				'fields' => [
					$MetaData->alias . '.id',
					$MetaData->alias . '.' . $field,
				],
				'joins' => [
					[
						'table' => $modelTable,
						'alias' => $Model->alias,
						'conditions' => [
							$Model->alias . '.id = ' . $MetaData->alias . '.model_id',
						]
					]
				],
				'conditions' => [
					$MetaData->alias . '.model' => $Model->alias,
					$MetaData->alias . '.' . $field . ' IS NOT NULL',
					$MetaData->alias . '.' . $field . ' !=' => '',
					$MetaData->alias . '.' . $field . ' REGEXP "' . $regex . '"',
				]
			]
		);

		if (empty($modelMetas)) {
			$this->out('Could not find any ' . InflectorExt::pluralize($Model->alias) . ' to have meta ' . InflectorExt::pluralize($field) . ' replaced');
			return null;
		}

		$this->out('Replacing meta ' . InflectorExt::pluralize($field) . ' in ' . count($modelMetas) . ' ' . InflectorExt::pluralize($Model->alias));

		foreach ($modelMetas as $modelMeta) {
			$metaField = $modelMeta[$MetaData->alias][$field];

			$metaField = preg_replace(
				'/' . $regex . '/',
				$token,
				$metaField
			);

			if (empty($metaField)) {
				continue;
			}

			$modelMeta[$MetaData->alias][$field] = $metaField;

			$MetaData->save($modelMeta);
		}

		return true;
	}
}
