<?php

App::uses('EvManualPaymentAppController', 'EvManualPayment.Controller');

class ManualPaymentController extends EvManualPaymentAppController {

	public $adminActions = array(
		'admin_payments'
	);

/**
 * Constructor.
 *
 * @param CakeRequest $request Request object for this controller. Can be null for testing,
 *  but expect that features that use the request parameters will not work.
 * @param CakeResponse $response Response object for this controller.
 */
	public function __construct(CakeRequest $request, CakeResponse $response) {
		parent::__construct($request, $response);

		$this->components[] = 'EvTransactions.Transactions';
	}

/**
 * An admin index page that shows all the current pending manual payments.
 *
 * @return void.
 */
	public function admin_payments() {
		$this->Paginator->settings = array(
			'limit' => 25,
			'conditions' => array(
				'Transaction.payment_method' => 'EvManualPayment.ManualPayment',
				'Transaction.status' => 'pending'
			)
		);
		$this->loadModel('EvTransactions.Transaction');
		$data = $this->Paginator->paginate('Transaction');
		$this->set('data', $data);
		$this->set('primaryModel', 'Transaction');

		$this->set('columns', $this->_adminIndexColumns());
		$this->set('actions', $this->_adminIndexActions());

		$this->view = 'admin_index';
	}

/**
 * Defines the columns displayed in the admin_index results table using a columns whitelist.
 *
 * @return array The index columns.
 */
	protected function _adminIndexColumns() {
		$columns = array(
			'Transaction.id' => array(
				'type' => 'integer',
				'label' => 'Id'
			),
			'Transaction.status' => array(
				'type' => 'string',
				'label' => 'Status'
			),
			'Transaction.created' => array(
				'type' => 'datetime',
				'label' => 'Date'
			)
		);

		return $columns;
	}

/**
 * Defines which actions are available to each row of the admin_index table results. Manual payments are just to be
 * viewed so no actions are available from the index.
 *
 * @return array The actions available on the index results.
 */
	protected function _adminIndexActions() {
		return [];
	}

/**
 * Complete a manual payment transaction. The transaction is made pending and the order made successful once.
 *
 * @return void.
 */
	public function complete() {
		$this->set('title_for_layout', 'Order Complete');

		if (CakePlugin::loaded('EvCheckout') && !empty($this->request->query['transaction'])) {
			if (empty($this->OrderManager)) {
				$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');
			}

			$orderId = $this->_transactionAlreadyProcessed($this->request->query['transaction']);

			if (empty($orderId)) {
				$this->loadModel('EvTransactions.Transaction');
				$this->Transaction->clear();
				$this->Transaction->id = $this->request->query['transaction'];
				$this->Transaction->save(
					[
						'Transaction' => array(
							'status' => 'pending',
							'message' => 'Offline payment - awaiting manual confirmation of payment'
						)
					]
				);

				$orderId = $this->OrderManager->buildOrder($this->request->query['transaction']);

				// if we've passed in an order id - trigger the success hook to ensure the usual order events occur
				$this->getEventManager()->dispatch(
					new CakeEvent('EvCheckout.Controller.Order.success', $this, array(
						'orderId' => $orderId
					))
				);
			}

			// On newer versions of checkout, check the transaction belongs to the current user before showing any details
			App::uses('OrderLib', 'EvCheckout.Lib');
			if (class_exists('OrderLib') && method_exists('OrderLib', 'checkTransactionAccess')) {
				$isAuthorised = OrderLib::checkTransactionAccess($this->request->query['transaction'], $this->Auth->user('User.id'));
				if (!$isAuthorised) {
					// Authorisation failed, show a basic success page with no details
					$this->OrderPages = $this->loadComponent('EvCheckout.OrderPages');
					$this->OrderPages->showUnauthorisedSuccess($this->request->query['transaction']);
					return;
				}
			}
		}

		$this->view = 'EvManualPayment.ManualPayment/complete';
	}

/**
 * Check if the transaction has already been successful.
 *
 * @param int $transactionId The id of the transaction to check.
 * @return int|bool The id of the transaction is returned if it has already been processed. False otherwise.
 */
	protected function _transactionAlreadyProcessed($transactionId) {
		$this->loadModel('EvTransactions.Transaction');

		$transaction = $this->Transaction->find(
			'first',
			[
				'conditions' => [
					'Transaction.id' => $transactionId,
					'Transaction.model' => 'EvCheckout.Order',
					'Transaction.status' => 'pending',
				],
			]
		);

		if (!empty($transaction['Transaction']['model_id'])) {
			return $transaction['Transaction']['model_id'];
		}

		return false;
	}

}
