<?php
App::uses('AppHelper', 'View/Helper');

class PointsHelper extends AppHelper {

	/**
	 * userPoints
	 *
	 * Returns the current (usable) points that a user has on their account.
	 * @return int returns the total number of points available
	 */
	public function userPoints() {
		$user = CakeSession::read('Auth.User');

		if (! $user) {
			return 0;
		}

		$pointsModel = EvClassRegistry::init('EvLoyaltyPoints.Point');

		return $pointsModel->totalPoints($user['User']['id']);
	}

	/**
	 * calculatePoints
	 *
	 * You pass this a price and it pulls the PointValue record for the currency, and works out the
	 * number of points that will be given for this item.
	 *
	 * @param  integer $price the total price (this will be rounded down to a whole number using floor)
	 * @return integer the total number of points that will be issued for this purchase
	 */
	public function calculatePoints($price) {
		$currencyId = CakeSession::read('EvCurrency.currencyId');
		$pointValueModel = EvClassRegistry::init('EvLoyaltyPoints.PointValue');

		$pointValue = $pointValueModel->find('first', array(
			'conditions' => array(
				'currency_id' => $currencyId,
				'minimum_spend <=' => $price
			)
		));

		if (! empty($pointValue)) {
			$pointsPerUnit = $pointValue['PointValue']['points_earnt_per_unit'];

			if (Configure::read('EvLoyaltyPoints.roundDown')) {
				// Round the price down (we don't support partial points)
				$price = floor($price);
			}
			$totalPoints = $pointsPerUnit * $price;

			return $totalPoints;
		}

		return 0;
	}

	/**
	 * Used to work out the monetary value of points.
	 * @param  int $points number of points
	 * @return decimal value of the points
	 */
	public function pointsValue($points) {
		$currencyId = CakeSession::read('EvCurrency.currencyId');
		$pointValueModel = EvClassRegistry::init('EvLoyaltyPoints.PointValue');

		$pointValue = $pointValueModel->find('first', array(
			'conditions' => array(
				'currency_id' => $currencyId
			)
		));

		$totalValue = 0;

		if (! empty($pointValue)) {
			$pointsPerUnit = $pointValue['PointValue']['points_per_unit'];
			$totalValue = ($points / $pointsPerUnit);
		}
		return $totalValue;
	}

	/**
	 * calculateDiscount
	 *
	 * Calculates what discount will be given on an order based on how many points the user has
	 * available to them. This is used for display purposes only and is not part of the actual
	 * discount application process.
	 *
	 * @param  integer $orderTotal the total value of the order
	 * @param  integer $currencyId  the id of the currency (used to work out the point value)
	 * @return integer  returns the discount that will be applied to this order (not currency formatted)
	 */
	public function calculateDiscount($orderTotal, $currencyId) {
		// floor the order total - we only give discounts on whole numbers
		$orderTotal = floor($orderTotal);

		$totalDiscount = 0; // stores the total discount being applied to the order
		$userPoints = $this->userPoints(); // total points the user has

		$pointValueModel = EvClassRegistry::init('EvLoyaltyPoints.PointValue');

		$pointValue = $pointValueModel->find('first', array(
			'conditions' => array(
				'currency_id' => $currencyId,
				'minimum_spend <=' => $orderTotal
			)
		));

		if (! empty($pointValue)) {
			// work out how many points can be used on the orderTotal
			$userPointsToCash = $userPoints / $pointValue['PointValue']['points_per_unit'];

			$totalDiscount = $userPointsToCash;
			if ($totalDiscount < 0) {
				$totalDiscount = $orderTotal;
			}

			if ($totalDiscount > $orderTotal) {
				$totalDiscount = $orderTotal;
			}
		}
		return $totalDiscount;
	}

	public function discountApplied() {
		return CakeSession::read('EvLoyaltyPoints.discountApplied');
	}

	public function discountRemaining() {
		$user = CakeSession::read('Auth.User');

		$pointsModel = EvClassRegistry::init('EvLoyaltyPoints.Point');
		$userPoints = $pointsModel->totalPoints($user['User']['id']);

		$pointsUsed = CakeSession::read('EvLoyaltyPoints.pointsUsed');

		return $userPoints - $pointsUsed;
	}

	public function discountsAvailable() {
		$user = CakeSession::read('Auth.User');

		if (! $user) {
			return false;
		}

		if ($this->discountApplied()) {
			return true;
		}

		// Check if the user has any points to use
		$pointsModel = EvClassRegistry::init('EvLoyaltyPoints.Point');

		if ($pointsModel->totalPoints($user['User']['id']) > 0) {
			return true;
		}

		return false;
	}

}
