<?php

App::uses('EvClassRegistry', 'EvCore.Lib');

class JupixSync {

	const SERVICE_NAME = 'Jupix',
		RESIDENTIAL_DEPARTMENTS = [
			'Sales',
			'Lettings'
		],
		TEMP_FILEPATH = ROOT . '/app/tmp',
		IMAGE_DIR = ROOT . '/app/webroot/files/image',
		DOCUMENT_DIR = ROOT . '/app/webroot/files/document';

/**
 * The main sync method, loads the data from the file and attempts to save each property it finds. If the property has already been saved
 * then it will attempt to update the property instead.
 * @param  string  $filepath Location of the Jupix XML property feed
 * @param  boolean $isUrl    Whether the file is local or not
 * @return boolean           Whether the save/update was successful
 */
	public function syncProperties($filepath, $isUrl = true) {
		$data = [];
		$saved = false;

		//Load data from file
		if ($isUrl) {
			try {
				$data = simplexml_load_file($filepath, null, LIBXML_NOCDATA);
				$data = json_decode(json_encode($data), true);
			} catch (Exception $e) {
				if (CakePlugin::loaded('EvErrbit')) {
					Errbit::notify($e);
				}
				CakeLog::write('jupixSync', 'Could not read jupix feed from url');
			}
		} else {
			//local file
			if (file_exists($filepath)) {
				try {
					//Load the xml file and process it into an object
					$data = simplexml_load_file($filepath, null, LIBXML_NOCDATA);

					$data = json_decode(json_encode($data), true);
				} catch (Exception $e) {
					if (CakePlugin::loaded('EvErrbit')) {
						Errbit::notify($e);
					}
					CakeLog::write('jupixSync', 'Could not read jupix feed from file');
				}
			}
		}

		//Process and save each property
		if (!empty($data)) {
			$residentialPropertyIdsInFeed = [];

			foreach ($data['property'] as $property) {
				//Check if the property is a residential department
				if (in_array($property['department'], self::RESIDENTIAL_DEPARTMENTS)) {
					//Check if the property is already in the system
					$updateProperty = self::_getProperty($property['propertyID'], 'EvProperty.ResidentialProperty');

					if (!empty($updateProperty)) {
						//Work out last modified date and see if we need to update this property
						$dateLastModified = DateTime::createFromFormat(
							'Y-m-d G:i:s',
							$property['dateLastModified'] . ' ' . $property['timeLastModified']
						);

						$recordedDateLastModified = DateTime::createFromFormat(
							'Y-m-d G:i:s',
							$updateProperty['ResidentialProperty']['service_last_modified']
						);

						if ($recordedDateLastModified < $dateLastModified) {
							CakeLog::write('jupixSync', 'Updating property: ' . $property['propertyID']);
							$saved = self::_saveResidentialProperty($property, $updateProperty);
						} else {
							CakeLog::write('jupixSync', 'Found but did not update property: ' . $property['propertyID'] . '. Last recorded modification: ' . $recordedDateLastModified->format('Y-m-d H:i:s') . '. Last Jupix modification: ' . $dateLastModified->format('Y-m-d H:i:s'));
						}
					} else {
						CakeLog::write('jupixSync', 'Saving property: ' . $property['propertyID']);
						$saved = self::_saveResidentialProperty($property);
					}
				}

				$residentialPropertyIdsInFeed[] = $property['propertyID'];
			}

			if (!empty($residentialPropertyIdsInFeed)) {
				self::_updateMissingResidentialProperties($residentialPropertyIdsInFeed);
			}
		}

		return $saved;
	}

/**
 * Look for a current property. If a property is found and the routable plugin is loaded then add the route.
 * @param  int    $propertyId    The id of the property to be found.
 * @param  obj    $propertyModel The property model to find on.
 * @return array                 The found property.
 */
	protected function _getProperty($propertyId, $propertyModel) {
		$PropertyModel = EvClassRegistry::init($propertyModel);
		$property = $PropertyModel->find(
			'first',
			[
				'conditions' => [
					'service' => self::SERVICE_NAME,
					'service_id' => $propertyId
				],
			]
		);

		if (!empty($property)) {
			if ($PropertyModel->Behaviors->loaded('Routable')) {
				$propertyRoute = $PropertyModel->getRoute($property);

				if (!empty($propertyRoute)) {
					$property['Route'] = $propertyRoute;
				}
			}
		}

		return $property;
	}

/**
 * Save a residential property.
 * @param  array  $property       The array of property data to save.
 * @param  array  $updateProperty The property to update if found.
 * @return bool                   Return the successfullness of the save.
 */
	protected function _saveResidentialProperty($property, $updateProperty = null) {
		//Find the branch or create one if it doesn't exist
		$branch = self::_getPropertyData('EvProperty.Branch', $property['branchID']);
		if (empty($branch)) {
			$branch = self::_createNewBranch($property['branchID'], $property['branchName']);
		}
		if (!empty($branch)) {
			$branchId = $branch['Branch']['id'];
		}

		//Find the region or create one if it doesn't exist
		$region = self::_getPropertyData('EvProperty.Region', $property['regionID']);
		if (empty($region)) {
			$region = self::_createNewRegion($property['regionID']);
		}
		if (!empty($region)) {
			$regionId = $region['Region']['id'];
		}

		//Find the property type
		$propertyType = self::_getPropertyData('EvProperty.PropertyType', $property['propertyType'], $property['department']);
		$propertyTypeId = $propertyType['PropertyType']['id'];

		//Find the property style
		$propertyStyle = self::_getPropertyData('EvProperty.PropertyStyle', $property['propertyStyle']);
		$propertyStyleId = $propertyStyle['PropertyStyle']['id'];

		//Find the property age
		$propertyAge = self::_getPropertyData('EvProperty.PropertyAge', $property['propertyAge']);
		$propertyAgeId = $propertyAge['PropertyAge']['id'];

		//Find the property availability
		if (!empty($property['availability'])) {
			$propertyAvailability = self::_getPropertyData('EvProperty.PropertyAvailability', $property['availability'], $property['department']);
			$propertyAvailabilityId = $propertyAvailability['PropertyAvailability']['id'];
		}

		//Find the price qualifier
		if (!empty($property['priceQualifier'])) {
			$priceQualifier = self::_getPropertyData('EvProperty.PriceQualifier', $property['priceQualifier'], $property['department']);
			$priceQualifierId = $priceQualifier['PriceQualifier']['id'];
		}

		//Find the property tenure
		if (!empty($property['propertyTenure'])) {
			$propertyTenure = self::_getPropertyData('EvProperty.PropertyTenure', $property['propertyTenure'], $property['department']);
			$propertyTenureId = $propertyTenure['PropertyTenure']['id'];
		}

		//Find the sale type
		if (!empty($property['saleBy'])) {
			$saleType = self::_getPropertyData('EvProperty.SaleType', $property['saleBy'], $property['department']);
			$saleTypeId = $saleType['SaleType']['id'];
		}

		//Find the rent frequency
		if (!empty($property['rentFrequency'])) {
			$rentFrequency = self::_getPropertyData('EvProperty.RentFrequency', $property['rentFrequency'], $property['department']);
			$rentFrequencyId = $rentFrequency['RentFrequency']['id'];
		}

		//Create property
		$propertyData = [
			'ResidentialProperty' => [
				//Property data fields
				'service' => self::SERVICE_NAME,
				'service_id' => $property['propertyID'],
				'client_name' => $property['clientName'],
				'branch_id' => (!empty($branchId) ? $branchId : null),
				'department' => $property['department'],
				'reference_number' => (isset($property['referenceNumber']) ? $property['referenceNumber'] : null),
				'display_address' => $property['displayAddress'],
				'bedrooms' => $property['propertyBedrooms'],
				'bathrooms' => $property['propertyBathrooms'],
				'ensuites' => $property['propertyEnsuites'],
				'reception_rooms' => $property['propertyReceptionRooms'],
				'kitchens' => $property['propertyKitchens'],
				'display_property_type' => $property['displayPropertyType'],
				'property_type_id' => $propertyTypeId,
				'property_style_id' => $propertyStyleId,
				'property_age' => $propertyAgeId,
				'floor_area' => $property['floorArea'],
				'floor_area_units' => $property['floorAreaUnits'],

				//Sales property fields
				'property_availability_id' => (isset($propertyAvailabilityId) ? $propertyAvailabilityId : null),
				'price' => (isset($property['price']) ? $property['price'] : null),
				'show_price' => (isset($property['forSalePOA']) && $property['forSalePOA'] == 1 ? true : false),
				'price_qualifier_id' => (isset($priceQualifierId) ? $priceQualifierId : null),
				'property_tenure_id' => (isset($propertyTenureId) ? $propertyTenureId : null),
				'sale_type_id' => (isset($saleTypeId) ? $saleTypeId : null),
				'is_development_oppurtunity' => (isset($property['developmentOpportunity']) && $property['developmentOpportunity'] == 1 ? true : false),
				'is_investment_oppurtunity' => (isset($property['investmentOpportunity']) && $property['investmentOpportunity'] == 0 ? true : false),
				'estimated_rental_income' => (isset($property['estimatedRentalIncome']) ? $property['estimatedRentalIncome'] : null),

				//Letting property fields
				'rent' => (isset($property['rent']) ? $property['rent'] : null),
				'rent_frequency_id' => (isset($rentFrequencyId) ? $rentFrequencyId : null),
				'show_rent' => (isset($property['toLetPOA']) && $property['toLetPOA'] == 1 ? true : false),
				'is_student_property' => (isset($property['studentProperty']) && $property['studentProperty'] == 1 ? true : false),
				'letting_fee_policy_title' => (isset($property['lettingFeePolicyHeadline']) ? $property['lettingFeePolicyHeadline'] : null),
				'letting_fee_policy' => (isset($property['lettingFeePolicyDetails']) ? $property['lettingFeePolicyDetails'] : null),

				//Property info fields
				'latitude' => $property['latitude'],
				'longitude' => $property['longitude'],
				'region_id' => (!empty($regionId) ? $regionId : null),
				'is_featured' => ($property['featuredProperty'] == 0 ? false : true),
				'service_last_modified' => $property['dateLastModified'] . ' ' . $property['timeLastModified'],
				'summary' => self::_extractMainSummary($property),
				'description' => $property['fullDescription'],

				'is_active' => true,
				'modified' => date('Y-m-d H:i:s')
			]
		];

		$ResidentialProperty = EvClassRegistry::init('EvProperty.ResidentialProperty');
		if (!empty($updateProperty)) {
			$propertyData = Hash::merge($updateProperty, $propertyData);
		}

		$ResidentialProperty->create();
		$propertyResult = $ResidentialProperty->save($propertyData);

		if ($propertyResult) {
			$propertyResult['ResidentialProperty']['id'] = $ResidentialProperty->id;

			//Create address to save for property
			self::_savePropertyAddress($property, $propertyResult, $updateProperty);

			self::_savePropertyFeatures($property, $propertyResult, $updateProperty);

			self::_savePropertyFlags($property, $propertyResult, $updateProperty);

			self::_savePropertyLinks($property, $propertyResult, $updateProperty);

			self::_savePropertyFiles($property, $propertyResult, $updateProperty, 'images', 'GeneralImage');

			self::_savePropertyFiles($property, $propertyResult, $updateProperty, 'floorplans', 'FloorplanImage');

			self::_savePropertyFiles($property, $propertyResult, $updateProperty, 'epcGraphs', 'EpcGraphImage');

			self::_savePropertyFiles($property, $propertyResult, $updateProperty, 'epcFrontPages', 'EpcFrontPageImage');

			self::_savePropertyFiles($property, $propertyResult, $updateProperty, 'brochures', 'BrochureDocument', false);
		}

		return $propertyResult;
	}

/**
 * Get the id of the associated property data using the provided jupix index and department
 * @param  string $model      The name of the model to search the id for.
 * @param  int    $serviceId  The index of the data to search for.
 * @param  string $department The department of the property
 * @return int                The id of the property data.
 */
	protected function _getPropertyData($model, $serviceId, $department = null) {
		$conditions = [
			'service' => self::SERVICE_NAME,
			'service_id' => $serviceId
		];

		if (!empty($department)) {
			$conditions['department'] = $department;
		}

		return EvClassRegistry::init($model)->find(
			'first',
			[
				'fields' => [
					'id'
				],
				'conditions' => $conditions
			]
		);
	}

/**
 * The property belongs to a branch that doesn't exist yet so create it first.
 * @param  int    $serviceId  The id of the branch on Jupix
 * @param  string $branchName The name of the new branch
 * @return array              The new branch.
 */
	protected function _createNewBranch($serviceId, $branchName) {
		$Branch = EvClassRegistry::init('EvProperty.Branch');
		$Branch->create();
		$newBranch = $Branch->save(
			[
				'service' => self::SERVICE_NAME,
				'service_id' => $serviceId,
				'name' => $branchName
			]
		);

		$newBranch['Branch']['id'] = $Branch->id;

		return $newBranch;
	}

/**
 * The property belongs to a region that doesn't exists yet so create it first.
 * @param  int    $serviceId The id of the region on Jupix.
 * @return array             The new region.
 */
	protected function _createNewRegion($serviceId) {
		$Region = EvClassRegistry::init('EvProperty.Region');
		$Region->create();
		$newRegion = $Region->save(
			[
				'service' => self::SERVICE_NAME,
				'service_id' => $serviceId,
				'name' => ''
			]
		);

		$newRegion['Region']['id'] = $Region->id;

		return $newRegion;
	}

/**
 * Save the address of a property. If an address already exists then update it.
 * @param  array $jupixProperty  The property from the jupix feed. Contains all the property data.
 * @param  array $savedProperty  The property that is currently being saved. Contains the data of the current property
 * @param  array $updateProperty The property that already exists.
 * @return array                 The saved address.
 */
	protected function _savePropertyAddress($jupixProperty, $savedProperty, $updateProperty) {
		$propertyId = $savedProperty['ResidentialProperty']['id'];

		$propertyAddressData = [
			'model' => 'ResidentialProperty',
			'model_id' => $propertyId,
			'name' => (!empty($jupixProperty['addressName']) ? $jupixProperty['addressName'] : null),
			'address1' => (!empty($jupixProperty['addressNumber']) ? $jupixProperty['addressNumber'] . ' ': null) . (!empty($jupixProperty['addressStreet']) ? $jupixProperty['addressStreet'] : null),
			'address2' => (!empty($jupixProperty['address2']) ? $jupixProperty['address2'] : null),
			'address3' => null,
			'city' => $jupixProperty['address3'],
			'county' => $jupixProperty['address4'],
			'state' => null,
			'post_code' => trim($jupixProperty['addressPostcode']),
			'country_id' => 225, //All Jupix properties are in the UK
			'is_active' => true,
		];

		if (empty($propertyAddressData['address1'])) {
			$propertyAddressData['address1'] = 'N/A';
		}

		$Address = EvClassRegistry::init('EvAddressBook.Address');

		if (!empty($updateProperty)) {
			$foundAddress = $Address->find(
				'first',
				[
					'conditions' => [
						'model' => 'Property',
						'model_id' => $propertyId
					]
				]
			);

			if (!empty($foundAddress)) {
				$propertyAddressData['id'] = $foundAddress['Address']['id'];
			}
		}

		$Address->create();
		return $Address->save($propertyAddressData);
	}

/**
 * Save the features of a property. If the property is being updated, check to see if any features need deleting from
 * the property. Only saves new property features and removes old ones.
 * @param  array $jupixProperty  The property from the jupix feed. Contains all the property data.
 * @param  array $savedProperty  The property that is currently being saved. Contains the data of the current property
 * @param  array $updateProperty The property that already exists.
 * @return array                 The saved features.
 */
	protected function _savePropertyFeatures($jupixProperty, $savedProperty, $updateProperty) {
		$propertyId = $savedProperty['ResidentialProperty']['id'];

		$PropertyFeature = EvClassRegistry::init('EvProperty.PropertyFeature');

		$propertyFeatures = [];
		if (!empty($updateProperty)) {
			$propertyFeatures = $PropertyFeature->find(
				'all',
				[
					'conditions' => [
						'residential_property_id' => $propertyId
					],
					'order' => 'PropertyFeature.sequence ASC'
				]
			);
		}

		$featuresToDelete = [];

		//Features are stored as separate fields. Jupix has a max of 10
		for ($featureNum = 0; $featureNum < 10; $featureNum++) {
			$feature = $jupixProperty['propertyFeature' . ($featureNum + 1)];

			if (!empty($propertyFeatures[$featureNum])) {
				//Already has a property feature stored
				if (!empty($feature)) {
					$propertyFeatures[$featureNum]['PropertyFeature']['name'] = $feature;
				} else {
					$featuresToDelete[] = $propertyFeatures[$featureNum]['PropertyFeature']['id'];
					unset($propertyFeatures[$featureNum]);
				}
			} else {
				if (!empty($feature)) {
					$propertyFeatures[] = [
						'residential_property_id' => $propertyId,
						'name' => $feature,
						'sequence' => $featureNum
					];
				}
			}
		}

		if (!empty($updateProperty) && !empty($featuresToDelete)) {
			$PropertyFeature->deleteAll(['PropertyFeature.id' => $featuresToDelete]);
		}

		if (!empty($propertyFeatures)) {
			return $PropertyFeature->saveMany($propertyFeatures);
		}

		return true;
	}

/**
 * Save the flags of a property. There is no way to compare flags so they are all deleted and saved again. Could possibly
 * work like features except the feeds have shown they aren't really used.
 * @param  array $jupixProperty  The property from the jupix feed. Contains all the property data.
 * @param  array $savedProperty  The property that is currently being saved. Contains the data of the current property
 * @param  array $updateProperty The property that already exists.
 * @return array                 The saved flags.
 */
	protected function _savePropertyFlags($jupixProperty, $savedProperty, $updateProperty) {
		$propertyId = $savedProperty['ResidentialProperty']['id'];

		$PropertyFlag = EvClassRegistry::init('EvProperty.PropertyFlag');

		//There isn't a way to track which flags are which so have to delete them all and save them again
		if (!empty($updateProperty)) {
			$PropertyFlag->deleteAll(['PropertyFlag.residential_property_id' => $propertyId]);
		}

		if (!empty($jupixProperty['flags']['flag'])) {
			$propertyFlags = [];

			if (is_array($jupixProperty['flags']['flag'])) {
				foreach ($jupixProperty['flags']['flag'] as $flag) {
					$propertyFlags[] = [
						'residential_property_id' => $propertyId,
						'name' => $flag,
					];
				}
			} else {
				$propertyFlags[] = [
					'residential_property_id' => $propertyId,
					'name' => $jupixProperty['flags']['flag'],
				];
			}

			if (!empty($propertyFlags)) {
				return $PropertyFlag->saveMany($propertyFlags);
			}
		}

		return true;
	}

/**
 * Save the links of a property. Links are checked as they are absolute so matching links are kept and any others are removed.
 * @param  array $jupixProperty  The property from the jupix feed. Contains all the property data.
 * @param  array $savedProperty  The property that is currently being saved. Contains the data of the current property
 * @param  array $updateProperty The property that already exists.
 * @return array                 The saved links.
 */
	protected function _savePropertyLinks($jupixProperty, $savedProperty, $updateProperty) {
		$propertyId = $savedProperty['ResidentialProperty']['id'];

		$PropertyLink = EvClassRegistry::init('EvProperty.PropertyLink');

		$propertyLinks = [];
		if (!empty($updateProperty)) {
			$propertyLinks = $PropertyLink->find(
				'all',
				[
					'conditions' => [
						'residential_property_id' => $propertyId
					]
				]
			);

			if (!empty($propertyLinks)) {
				$currentLinks = Hash::combine($propertyLinks, '{n}.PropertyLink.url', '{n}.PropertyLink');
			}
		}

		if (!empty($jupixProperty['externalLinks'])) {

			foreach ($jupixProperty['externalLinks'] as $link) {
				if (!empty($link['url'])) {
					$propertyLinkData = [
						'residential_property_id' => $propertyId,
						'url' => $link['url'],
						'description' => $link['description'],
						'service_last_modified' => $link['modified'],
					];

					if (!empty($currentLinks) && in_array($link['url'], array_keys($currentLinks))) {
						if ($currentLinks[$link['url']]['modified'] != $link['modified']) {
							$propertyLinkData['id'] = $currentLinks[$link['url']]['id'];
						} else {
							$propertyLinkData = [];
						}

						$linksToDelete[] = $currentLinks[$link['url']]['id'];
						unset($currentLinks[$link['url']]);
					}

					if (!empty($propertyLinkData)) {
						$propertyLinks[] = $propertyLinkData;
					}
				}

			}
		}

		if (!empty($updateProperty) && !empty($linksToDelete)) {
			$PropertyLink->deleteAll(['PropertyLink.id' => $linksToDelete]);
		}

		if (!empty($propertyLinks)) {
			return $PropertyLink->saveMany($propertyLinks);
		}

		return true;
	}

/**
 * Save the address of a property. If an address already exists then update it.
 * @param  array  $jupixProperty  The property from the jupix feed. Contains all the property data.
 * @param  array  $savedProperty  The property that is currently being saved. Contains the data of the current property
 * @param  array  $updateProperty The property that already exists.
 * @param  array  $fileGroup      The name of file in Jupix. So that each file can be extracted.
 * @param  string $attachmentType The name of the slot that the file will occupy locally.
 * @param  array  $isImage        Is the file and image or a document.
 * @return array                  The saved address.
 */
	protected function _savePropertyFiles($jupixProperty, $savedProperty, $updateProperty, $fileGroup, $attachmentType, $isImage = true) {
		$propertyId = $savedProperty['ResidentialProperty']['id'];

		$singleFileGroup = Inflector::singularize($fileGroup);

		if (!empty($updateProperty)) {
			if ($isImage) {
				$Model = EvClassRegistry::init('EvCore.Image');
				$serviceUrlField = 'link_url';
			} else {
				$Model = EvClassRegistry::init('EvCore.Document');
				$serviceUrlField = 'caption';
			}
			$currentPropertyFiles = $Model->find(
				'all',
				[
					'conditions' => [
						'model' => 'ResidentialProperty',
						'model_id' => $propertyId,
						'attachment_type' => $attachmentType
					]
				]
			);

			$currentPropertyFiles = Hash::extract(
				$currentPropertyFiles,
				'{n}.' . $Model->alias . '.id'
			);

			$Model->deleteAll([$Model->alias . '.id' => $currentPropertyFiles], false, true);
		}

		$propertyFiles = [];
		if (!empty($jupixProperty[$fileGroup][$singleFileGroup])) {
			if (is_array($jupixProperty[$fileGroup][$singleFileGroup])) {
				foreach ($jupixProperty[$fileGroup][$singleFileGroup] as $file) {
					self::_savePropertyFile($file, $attachmentType, $propertyId, $updateProperty, $isImage);
					$propertyFiles[] = $file;
				}
			} else {
				if (!empty($jupixProperty[$fileGroup][$singleFileGroup])) {
					self::_savePropertyFile($jupixProperty[$fileGroup][$singleFileGroup], $attachmentType, $propertyId, $updateProperty, $isImage);
					$propertyFiles[] = $jupixProperty[$fileGroup][$singleFileGroup];
				}
			}
		}
	}

/**
 * Download a Jupix file and save it into the correct directory and update the database.
 * @param  string  $fileUrl        The url of the file to download.
 * @param  string  $attachmentType The slot the file will occupy. Also used to check for a file to update.
 * @param  int     $propertyId     The id of the property to associate the file with.
 * @param  array   $updateProperty The existing property
 * @param  boolean $isImage        Whether the file is an image or not
 * @return array                   The saved file.
 */
	protected function _savePropertyFile($fileUrl, $attachmentType, $propertyId, $updateProperty, $isImage = true) {
		if ($isImage) {
			$Model = EvClassRegistry::init('EvCore.Image');
		} else {
			$Model = EvClassRegistry::init('EvCore.Document');
		}

		$updated = null;
		try {
			$file = file_get_contents($fileUrl);
			if ($file === false) {
				$file = null;
			}
		} catch (Exception $e) {
			if (CakePlugin::loaded('EvErrbit')) {
				Errbit::notify($e);
			}
			CakeLog::write('jupixSync', 'Could not find ' . ' ' . $attachmentType . ' : ' . $fileUrl . ' for property: ' . $propertyId);

			$file = null;
		}

		if (!empty($file)) {
			$pathInfo = pathinfo($fileUrl);
			$extension = null;
			$filename = null;
			if (!empty($pathInfo)) {
				$extension = (!empty($pathInfo['extension']) ? $pathInfo['extension'] : null);
				$filename = (!empty($pathInfo['filename']) ? $pathInfo['filename'] : null);
			}

			$downloaded = false;
			if (!empty($filename) && !empty($extension)) {
				$tempFilepath = self::TEMP_FILEPATH . DS . $filename . '.' . $extension;
				$downloaded = file_put_contents($tempFilepath, $file);
			}

			if ($downloaded !== false) {
				$tempFile = new File($tempFilepath);
				$tempFileInfo = $tempFile->info();

				$fileData = [
					$Model->alias => [
						'model' => 'ResidentialProperty',
						'model_id' => $propertyId,
						'filename' => $tempFileInfo['filename'] . '.' . $tempFileInfo['extension'],
						'attachment_type' => $attachmentType,
						'type' => $tempFileInfo['mime'],
						'ext' => $tempFileInfo['extension'],
						'size' => $tempFileInfo['filesize'],
						'sequence' => 0,
						'is_active' => true,
					]
				];

				if ($isImage) {
					$fileData[$Model->alias]['link_url'] = $fileUrl;
				} else {
					$fileData[$Model->alias]['caption'] = $fileUrl;
				}

				//Image downloaded successfully, add it to the database
				$Model->create();
				$saved = $Model->save($fileData);

				if ($saved) {
					if ($isImage) {
						$fileDirectory = self::IMAGE_DIR;
					} else {
						$fileDirectory = self::DOCUMENT_DIR;
					}

					mkdir($fileDirectory . DS . $Model->id, 0777, true);

					$newFilepath = $fileDirectory . DS . $Model->id . DS . $filename . '.' . $extension;
					copy($tempFilepath, $newFilepath);

					$updated = $Model->save(
						[
							$Model->alias => [
								'id' => $Model->id,
								'filename' => $tempFileInfo['filename'] . '.' . $tempFileInfo['extension'],
								'attachment_type' => $attachmentType,
								'size' => $tempFileInfo['filesize'],
								'dir' => $Model->id,
							]
						]
					);
				}

				unlink($tempFilepath);
			}
		}

		return $updated;
	}

/**
 * Find any properties that aren't in the current jupix feed and set them to inactive. Also remove all the associated
 * images and documents. If they are made active again then those files will be redownloaded.
 * @param  array  $foundPropertyIds The service ids of the properties currently in the jupix feed
 * @return bool                     Whether any properties were found and successfully updated
 */
	protected function _updateMissingResidentialProperties($foundPropertyIds) {
		$Model = EvClassRegistry::init('EvProperty.ResidentialProperty');

		$missingProperties = $Model->find(
			'all',
			[
				'conditions' => [
					'ResidentialProperty.service_id !=' => $foundPropertyIds,
					'ResidentialProperty.is_active' => true
				]
			]
		);

		$result = false;

		if (!empty($missingProperties)) {
			$missingPropertyIds = Hash::extract($missingProperties, '{n}.ResidentialProperty.id');

			//Delete Files
			EvClassRegistry::init('EvCore.Image')->deleteAll(
				[
					'Image.model' => $Model->alias,
					'Image.model_id' => $missingPropertyIds,
				]
			);

			EvClassRegistry::init('EvCore.Document')->deleteAll(
				[
					'Document.model' => $Model->alias,
					'Document.model_id' => $missingPropertyIds,
				]
			);

			//Set properties to inactive
			$result = $Model->updateAll(
				[
					'ResidentialProperty.is_active' => false,
				],
				[
					'ResidentialProperty.id' => $missingPropertyIds,
				]
			);
		}

		return $result;
	}

/**
 * Set the service last modified field for every property of this service to the date of the epoch.
 */
	public function resetProperties() {
		$Model = EvClassRegistry::init('EvProperty.ResidentialProperty');

		$result = $Model->updateAll(
			[
				'ResidentialProperty.service_last_modified' => "'1970-01-01 00:00:00'",
			],
			[
				'ResidentialProperty.service' => self::SERVICE_NAME,
			]
		);

		return $result;
	}

/**
 * Extract the main summary from a jupix property to be saved.
 *
 * @param array $property The decoded property from the jupix feed.
 * @return string The main summary to save.
 */
	protected function _extractMainSummary($property) {
		$mainSummary = '';

		if (!empty($property['mainSummary'])) {
			if (is_array($property['mainSummary'])) {
				//The main summary shouldn't be an array but accidents happen, see if the first element can be used.
				$mainSummaryElement = array_shift($property['mainSummary']);
				if (!empty($mainSummaryElement) && is_string($mainSummaryElement)) {
					$mainSummary = $mainSummaryElement;
				}
			} else {
				$mainSummary = $property['mainSummary'];
			}
		}

		return $mainSummary;
	}
}
