# EvInventory

## Installation

Simply add EvInventory to the `install.php` file within the Config directory and run `Console/cake installer core` once you have added all the other plugins you will need.

If this has already been run, simply run `Console/cake installer plugin EvInventory `.

There is no installer for this plugin, only zuul... i mean migrations.

## How it works

This plugin works by attaching a behavior to the required model and then simply triggering certain events using CakePHPs [Event System](http://book.cakephp.org/2.0/en/core-libraries/events.html).

A listener built into the plugin listens for the `EvInventory.Inventory.reduce` event and from there calls the reduce method with the given parameters.

Other listeners are defined which listen for `EvInventory.Behavior.Inventory.saved`, these are there to perform certain actions upon an Inventory item being saved, usually this is updating stock levels so these listeners listen for the warning level (if set) and any out of stock inventories.

Other functionality that is set via the admin for displaying warning messages or out of stock messages are handled by means of a View Helper which in the template should be passed the inventory data, this will then return a boolean value allowing you to calculate whether or not it needs to show a message.

E.G.

	// work out if we need to show a delayed delivery message (allows purchase when OOS) or low stock message.
	if ($this->Inventory->showDelayedMessage($Variant) || $this->Inventory->showLowStockMessage($Variant)):

	// works out if we should allow them to purchase the item or not
	if ($this->Inventory->allowPurchase($Variant)):

For the full list of methods available, see the `InventoryHelper`.

## Setup

To setup the plugin once installed, simply attach a behavior to the model that requires the use of an inventory.

	if (CakePlugin::loaded('EvInventory')) {
		$this->actsAs[] = 'EvInventory.Inventories';
	}

The behavior will handle the processing of saving the inventory data.

Due to the nature of not knowing how the inventory may want to be implemented, this plugin **does not** currently make use of the `formInject` functionality. Instead, in this instance you will need to redefine a template and include the following element:

	if (CakePlugin::loaded('EvInventory')):
		echo $this->element(
			'EvInventory.stockControl'
		);
	endif;

You can also pass into the second optional data parameter, a value for `fieldPrefix`. This will basically prepend a prefix to the field name of each inventory item input within the element. This is useful for instances such as EvShop where on the Variants management page there are multiple Inventories (one for each variant). It simply allows you to setup the form elements into CakePHPs format so it can automatically handle the saving of related data.

Example from the Variant manage template showing the `fieldPrefix` in use:

	if (CakePlugin::loaded('EvInventory')):
		echo $this->element(
			'EvInventory.stockControl',
			array('fieldPrefix' => 'Variants.' . $key)
		);
	endif;

## Advanced Setup

### Custom Actions

All the actions that a user can select for inventory warning level or out of stock actions are set in the config file as an array in a `'keyName' => 'Nice Label'` format. For any custom actions, if you ensure the `keyName` is also a valid method name, you can then create a method within the `Lib/InventoryActions.php` with this keyName. When processing the warning levels or out of stock levels if this action is triggered it will first look within this file to check if a method exists with the same name as the `keyName` define in the config. If not, it simply does nothing and is assumed it will be handled within the View.

### Warning Emails

If you are using the warning / out of stock warning emails then further configuration will be needed in order to define how to retrieve certain data so the email can display things such as the product name that is out of stock and also a link to the management page.

The first attribute to define is the relationships from the inventory back to it's original item. This is done with the Model name as the array key and then the standard CakePHP belongs to syntax within, this allows the inventory plugin to be used on multiple different models in one project.

Example code from Brighton Tools:

	'InventoryBelongsTo' => array(
		'Variant' => array(
			'className' => 'EvShop.Variant',
			'foreignKey' => 'model_id',
			'conditions' => array(
				"Inventory.model = 'EvShop.Variant'"
			)
		)
	),

The second attribute is essentially the "contains" for a find on the inventory model so it can actually retrieve a product name / item name for it to later display in the email.

Example code from Brighton Tools:

	'InventoryContains' => array(
		'Variant' => array(
			'Product'
		)
	),

As it's the Variant that is linked to the Inventory, we have to get the Variant first before getting the Product data. This array is simply merged into the existing "contain" on the Inventory find.

The third attribute is a definition of multiple array values which are essential Hash syntax paths and a separator element, defining how to separate multiple paths if defined. These are all grouped by the model name that is attached to the Inventory database row. This is so when looking up the inventory row, the model column can be then used to work out what sort of model this inventory row is attached to and therefore get the display name data.

The following code is an example from Brighton Tools, and shows the full example. The first parameter is a assocative element with the key of 'separator' so the code knows to use that value to separate the rest of the array elements (**default is a single space**), the next two elements are a Hash path passed to `Hash::get` to retrieve the product name and then the variant name.

	'InventoryNamePath' => array(
		'EvShop.Variant' => array(
			'separator' => ': ',
			'Variant.Product.name',
			'Variant.name'
		)
	),

The attributes will be processed in the order they are defined in this array.

The fourth attribute is similar to the third attribute but is used to retrieve the admin link for the given model. It's again given a key to match what would be in the inventory row model column, then within it's simply a CakePHP routing array. If parameters are needed such as IDs, define an array containing Hash syntax paths and when processing, before the route is passed to the `link` method, the processor loops over all the elements within and uses `Hash::get` to retrieve any required IDs.

Example below from Brighton Tools:

	'InventoryUrlArray' => array(
		'EvShop.Variant' => array(
			'admin' => true,
			'plugin' => 'ev_shop',
			'controller' => 'variants',
			'action' => 'manage',
			array(
				'Variant.Product.id'
			)
		)
	)

### Synchronising inventories

In some cases you may have inventories that should have the same state even though the inventory belongs to a different record. In this scenario the inventories need to be kept synchronised so that each inventory can be treated as one. To enable synchronisation and to define which inventories need to be synchronised, the `synchonise` setting in the config needs enabling. By default the setting is set to `false` and any empty value will disable any synchronisation from occuring. To enable synchronisation the setting needs to look like:

```
	'synchronise' => [
		'ModelAlias' => [
			'field1',
			'field2',
			'field3',
			'etc',
		]
	]
```

The above setting would synchronise any inventory that belongs to a `ModelAlias` and has matching `field1`, `field2`, `field3` and etc. A real example might be if you are using EvShop and allow products to be copied. In this scenario the products are different but the variants are identical in terms of stock allocation due to their SKUs being the same. To synchronise across them the `synchronise` config setting would like the following:

```
	'synchronise' => [
		'Variant' => [
			'sku',
		]
	]
```

The above setting would synchronise any inventory that belongs to a Variant and has matching SKUs.

## Usage

The behavior will automatically setup a relationship to the inventory from the main model and contain it on any find calls.

If the inventory data is not required then disabling of the callbacks on the find query will prevent the behavior from running.

## Reducing Stock Levels

If building an application with custom inventories integration then you can use the following code as a basis to trigger the reduction of inventory stock levels.

	if (CakePlugin::loaded('EvInventory')) {
		// dispatch event to lower stock levels
		$this->getEventManager()->dispatch(
			new CakeEvent('EvInventory.Inventory.reduce', $this, array(
				'model' => $item['model'],
				'model_id' => $item['model_id'],
				'quantity' => $item['quantity']
			))
		);
	}

If called from a component, change the line `$this->getEventManager()->dispatch` to `$this->_controller->getEventManager()->dispatch`.

The event always requires the 3 parameters above. If any are missing in any combination the event will fail to run correctly.

## Checking if Stock Exists

If building a custom integration with inventories and you need to check there is enough stock to fulfil a request, you can use the code below as example.

	$Model = EvClassRegistry::init($item['BasketItem']['model']); // e.g.  EvShop.Variant
	if (CakePlugin::loaded('EvInventory') && $Model->hasBehavior('EvInventory.Inventories')) {
		App::uses('InventoryLib', 'EvInventory.Lib');

		if (
			InventoryLib::hasEnoughStock(
				$Model->readForEdit($item['BasketItem']['model_id']),
				$item['BasketItem']['quantity']
			)
		) {
			return true;
		}
	}

## Events

### EvInventory.Behavior.Inventory.saved

Called by the behavior when the inventory items are updated by the product (e.g. when stock levels are updated)

The `subject` for the event is the Model that the behaviour is attached to.

#### Parameters

* `id` - The model item id numbers inventory we just updated.

### EvInventory.Inventory.reduce

Called by any system when it wants to reduce the stock number. Out of the box, this is used by `EvCheckout` when an order is complete to reduce the stock of the purchased Variant.

The `subject` for this event may vary depending what is calling it, within `EvCheckout` it is `BasketBuilder` as this is the file that tranformers the checkout into an Order.

#### Parameters

* `model` - The model name of the inventory we are reducing
* `model_id` - The model id we are reducing
* `quantity` - The amount by which we are reducing the stock.

## Getting the inventory on the listing page

If you need to access the inventory on the product listing page you have to call:

	$this->Product->Variant->linkInventory();

before the listing is created. You also need to include the following option in the listing:

	'contain' => array(
		'Variant' => array(
			'Inventory'
		)
	)

## Glossary
oos : Out Of Stock - Refers to the JSON array that contains the actions that need to be made when an item goes out of stock.
