<?php
App::uses('EvInventoryAppModel', 'EvInventory.Model');
App::uses('InventoryActions', 'EvInventory.Lib');
App::uses('InventoryLib', 'EvInventory.Lib');

/**
 * EvInventoryInventory Model
 *
 */
class Inventory extends EvInventoryAppModel {

/**
 * Constructor. Binds the model's database table to the object.
 *
 * Setup the dynamic relationships.
 *
 * @param bool|int|string|array $id Set this ID for this model on startup,
 * can also be an array of options, see above.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 * @return void.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$relationships = Configure::read('EvInventory.InventoryBelongsTo');
		if (! empty($relationships)) {
			$belongsTo = array();
			foreach ($relationships as $key => $value) {
				$belongsTo[$key] = $value;
			}

			if (! empty($belongsTo)) {
				$this->bindModel(
					array(
						'belongsTo' => $belongsTo
					),
					false
				);
			}
		}
	}

/**
 * given the model name and the id get it's inventory
 *
 * TO-DO: Refactor out at some point for better testable methods
 *
 * @param string $modelName The model name it belongs to
 * @param int    $modelId   The models item id
 * @return array|bool $inventory Bool false on fail or inventory array
 */
	public function getInventory($modelName, $modelId) {
		$inventoryContain = Configure::read('EvInventory.InventoryContains');

		if (empty($inventoryContain)) {
			$inventoryContain = array();

			$relationships = Configure::read('EvInventory.InventoryBelongsTo');

			foreach ($relationships as $key => $value) {
				if (is_array($value)) {
					$inventoryContain[] = $key;
				} else {
					$inventoryContain[] = $value;
				}
			}
		}

		$params = [
			'conditions' => array(
				'Inventory.model' => $modelName,
				'Inventory.model_id' => $modelId
			),
			'contain' => $inventoryContain
		];

		$inventory = $this->find('first', $params);

		if (empty($inventory)) {
			return false;
		}

		return $inventory;
	}

/**
 * warning level reached, process the warning levl actions
 *
 * @param array $inventory The inventory row
 * @return bool
 */
	public function warningLevelReached($inventory) {
		if (! is_array($inventory['Inventory']['warning_action'])) {
			return true;
		}

		$ActionsLib = new InventoryActions($inventory, 'warning');

		foreach ($inventory['Inventory']['warning_action'] as $action) {
			if (method_exists($ActionsLib, $action)) {
				$ActionsLib->$action();
			}
		}

		return true;
	}

/**
 * out of stock, process the out of stock actions
 *
 * @param array $inventory The inventory row
 * @return bool
 */
	public function outOfStock($inventory) {
		if (! is_array($inventory['Inventory']['oos_action'])) {
			return true;
		}

		$ActionsLib = new InventoryActions($inventory, 'outofstock');

		foreach ($inventory['Inventory']['oos_action'] as $action) {
			if (method_exists($ActionsLib, $action)) {
				$ActionsLib->$action();
			}
		}

		return true;
	}

/**
 * Called before each save operation, after validation. Return a non-true result
 * to halt the save.
 *
 * JSON encode the action arrays if they haven't been already.
 *
 * @param array $options Options passed from Model::save().
 * @return bool True if the operation should continue, false if it should abort
 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#beforesave
 * @see Model::save()
 */
	public function beforeSave($options = array()) {
		if (! empty($this->data[$this->alias]['warning_action']) && is_array($this->data[$this->alias]['warning_action'])) {
			$this->data[$this->alias]['warning_action'] = json_encode($this->data[$this->alias]['warning_action']);
		}
		if (! empty($this->data[$this->alias]['oos_action']) && is_array($this->data[$this->alias]['oos_action'])) {
			$this->data[$this->alias]['oos_action'] = json_encode($this->data[$this->alias]['oos_action']);
		}

		// Everything is OK, return true so the save can continue
		return parent::beforeSave($options);
	}

/**
 * Invalidate productOption paths if they are being used
 *
 * @param bool $created If a new record was created
 * @param array $options The options used when saving
 * @return void
 */
	public function afterSave($created, $options = array()) {
		$InventoryLib = new InventoryLib();

		if (
			!empty($this->data[$this->alias]['model']) &&
			!empty($this->data[$this->alias]['model_id'])
		) {
			$modelClassName = $this->data[$this->alias]['model'];
			$modelId = $this->data[$this->alias]['model_id'];

			$syncModel = EvClassRegistry::init($modelClassName);
			$InventoryLib->syncStock($syncModel, $modelId);

			$this->data = $this->getInventory($modelClassName, $modelId);
		}

		$Event = new CakeEvent('EvInventory.Inventory.Saved', $this, array(
			'data' => $this->data
		));
		$this->getEventManager()->dispatch($Event);
		parent::afterSave($created, $options);
	}

/**
 * After find callback. Can be used to modify any results returned by find.
 *
 * @param mixed $results The results of the find operation
 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
 * @return mixed An array value will replace the value of $results - any other value will be ignored.
 */
	public function afterFind($results, $primary = false) {
		foreach ($results as $key => $item) {
			if (! empty($item[$this->alias]['warning_action'])) {
				$results[$key][$this->alias]['warning_action'] = json_decode($item[$this->alias]['warning_action'], true);
			} else {
				$results[$key][$this->alias]['warning_action'] = [];
			}

			if (! empty($item[$this->alias]['oos_action'])) {
				$results[$key][$this->alias]['oos_action'] = json_decode($item[$this->alias]['oos_action'], true);
			} else {
				$results[$key][$this->alias]['oos_action'] = [];
			}
		}

		return $results;
	}
}
