<?php

App::uses('AppBehavior', 'Model/Behavior');

App::uses('InventoryLib', 'EvInventory.Lib');

class InventoriesBehavior extends AppBehavior {

/**
 * bind the inventory model to the model
 *
 * @param Model &$Model The model the link inventory to.
 * @return void.
 */
	public function linkInventory(&$Model) {
		$Model->bindModel(
			array(
				'hasOne' => array(
					'Inventory' => array(
						'className' => 'EvInventory.Inventory',
						'foreignKey' => 'model_id',
						'conditions' => array(
							'Inventory.model' => EvClassRegistry::getNameFromModel($Model)
						),
						'type' => 'LEFT',
					)
				)
			),
			false
		);
	}

/**
 * beforeBeforeSave callback, in before saveAll is called, allowing you to modify the whole data array before
 * save processes.
 *
 * @param Model &$Model The model being saved.
 * @param array $data The data that is going to be saved.
 * @param array $options The save options being used.
 * @return array|bool Bool false to stop save, return modified data array to proceed
 */
	public function beforeBeforeSave(&$Model, $data, $options = array()) {
		$this->linkInventory($Model);

		$modelName = EvClassRegistry::getNameFromModel($Model);

		foreach ($data as $key => $item) {
			if (isset($item['Inventory'])) {
				$data[$key]['Inventory']['model'] = $modelName;
			}
		}

		return $data;
	}

/**
 * afterSave is called after a model is saved.
 *
 * @param Model $Model Model using this behavior
 * @param bool $created True if this save created a new record
 * @param array $options Options passed from Model::save().
 * @return bool
 * @see Model::save()
 */
	public function afterSave(Model $Model, $created, $options = array()) {
		// dispatch product saved event
		$Model->getEventManager()->dispatch(
			new CakeEvent('EvInventory.Behavior.Inventory.saved', $Model, array(
				'id' => $Model->id
			))
		);
	}

/**
 * beforeFind can be used to cancel find operations, or modify the query that will be executed.
 * By returning null/false you can abort a find. By returning an array you can modify/replace the query
 * that is going to be run.
 *
 * @param Model $Model Model using this behavior
 * @param array $query Data used to execute this query, i.e. conditions, order, etc.
 * @return bool|array False or null will abort the operation. You can return an array to replace the
 *   $query that will be eventually run.
 */
	public function beforeFind(Model $Model, $query) {
		if (isset($query['list'])) {
			return $query;
		}

		$this->linkInventory($Model);

		$query['contain'][] = 'Inventory';

		return $query;
	}

/**
 * Method called by the reduceStock listener.
 *
 * @param object $Model    The model calling this behavior.
 * @param int    $modelId  The model ID to look for an update.
 * @param int    $quantity The quantity to remove.
 * @return bool
 */
	public function reduceStock($Model, $modelId, $quantity) {
		$modelName = EvClassRegistry::getNameFromModel($Model);

		$Inventory = EvClassRegistry::init('EvInventory.Inventory');

		$result = $Inventory->updateAll(
			array(
				'Inventory.stock' => 'COALESCE(Inventory.stock, 0) - ' . $quantity
			),
			array(
				'Inventory.model' => $modelName,
				'Inventory.model_id' => $modelId
			)
		);

		if ($result) {
			$result = $this->syncStock($Model, $modelId);
		}

		if ($result) {
			$itemInventory = $Inventory->getInventory($modelName, $modelId);

			if ($itemInventory['Inventory']['stock'] <= $itemInventory['Inventory']['warning_level'] && $itemInventory['Inventory']['stock'] > 0) {
				$Inventory->warningLevelReached($itemInventory);
			}

			if ($itemInventory['Inventory']['stock'] <= 0) {
				$Inventory->outOfStock($itemInventory);
			}
		}

		return $result;
	}

/**
 * Provides a central function to increase stock
 *
 * @param object $Model    The model calling this behavior
 * @param int 	 $modelId  The model ID to look for an update
 * @param int 	 $quantity The quantity to add
 *
 * @return bool
 */
	public function addStock($Model, $modelId, $quantity) {
		$modelName = EvClassRegistry::getNameFromModel($Model);
		$Inventory = EvClassRegistry::init('EvInventory.Inventory');

		$result = $Inventory->updateAll(
			array(
				'Inventory.stock' => 'COALESCE(Inventory.stock, 0) + ' . $quantity
			),
			array(
				'Inventory.model' => $modelName,
				'Inventory.model_id' => $modelId
			)
		);

		if ($result) {
			$result = $this->syncStock($Model, $modelId);
		}

		return $result;
	}

/**
 * Synchronise the stock of multiple items. Inventories are found that have the same data as the
 * inventory to synchronise as defined in the config.
 *
 * @param Model $Model   The model that is being updated.
 * @param int   $modelId The id of the model that is being updated.
 * @return bool True if the stock was synchronised successfully, false otherwise.
 */
	public function syncStock($Model, $modelId) {
		return InventoryLib::syncStock($Model, $modelId);
	}
}
