<?php

class InventoryLib {

/**
 * Check the given model has enough stock to handle the requirement.
 *
 * @param array $data     The model data - if it actsAs Inventories then Inventory element should be available.
 * @param int   $quantity The quantity we want.
 * @return bool True if there is enough stock, false otherwise.
 */
	public static function hasEnoughStock($data, $quantity) {
		//Check the amount of stock meets requirements.
		if (!empty($data['Inventory']['stock']) && $data['Inventory']['stock'] >= $quantity) {
			return true;
		}

		//Check to see if selling can be allowed. The out of stock actions should be an array.
		if (!is_array($data['Inventory']['oos_action'])) {
			$data['Inventory']['oos_action'] = [];
		}

		//If the out of stock actions contain "stop" then the inventory is out of stock.
		if (in_array('stop', $data['Inventory']['oos_action'])) {
			return false;
		}

		return true;
	}

/**
 * Check if the an item can be purchased.
 *
 * @param array $data Item array containing subarray of Inventory data.
 * @return bool True if the item can be purchased. False otherwise.
 */
	public function allowPurchase($data) {
		return self::hasEnoughStock($data, 1);
	}

/**
 * Check if we need to show the delayed message for an item.
 *
 * @param array $data Data array containing subarray of Inventory data.
 * @return bool True if the delayed message should be shown, false otherwise.
 */
	public function showDelayedMessage($data) {
		if (!empty($data['Inventory']['stock']) && $data['Inventory']['stock'] > 0) {
			return false;
		}

		//The out of stock actions should be an array.
		if (!is_array($data['Inventory']['oos_action'])) {
			$data['Inventory']['oos_action'] = [];
		}

		//If the out of stock options contain "delayed" then the inventory is in stock with a delay.
		if (in_array('delayed', $data['Inventory']['oos_action'])) {
			return true;
		}

		return false;
	}

/**
 * Check if we need to show the low stock message for an item.
 *
 * @param array $data Data array containing subarray of Inventory data
 * @return bool True if the low stock message should be shown, false otherwise.
 */
	public function showLowStockMessage($data) {
		if (empty($data['Inventory']['warning_level'])) {
			return false;
		}

		if (!empty($data['Inventory']['stock']) && $data['Inventory']['stock'] > $data['Inventory']['warning_level']) {
			return false;
		}

		//The warning actions should be an array.
		if (!is_array($data['Inventory']['warning_action'])) {
			$data['Inventory']['warning_action'] = [];
		}

		//If the out of stock options contain "delayed" then the inventory is in stock with a delay.
		if (!empty($data['Inventory']['stock']) && in_array('warning', $data['Inventory']['warning_action'])) {
			return true;
		}

		return false;
	}

/**
 * Synchronise the stock of multiple items. Inventories are found that have the same data as the
 * inventory to synchronise as defined in the config.
 *
 * @param Model $Model   The model that is being updated.
 * @param int   $modelId The id of the model that is being updated.
 * @return bool True if the stock was synchronised successfully or no changes have occurred, false otherwise.
 */
	public function syncStock($Model, $modelId) {
		$syncOptions = Configure::read('EvInventory.synchronise');

		if (empty($syncOptions)) {
			//There are no synchronisation options so return without changes.
			return true;
		}

		$Inventory = EvClassRegistry::init('EvInventory.Inventory');

		//Get the inventory that we are attempting to synchronise.
		$modelName = EvClassRegistry::getNameFromModel($Model);
		$syncInventory = $Inventory->getInventory($modelName, $modelId);

		if (empty($syncOptions[$Model->alias])) {
			return true;
		}

		//Extract the values from the inventory we are syncing so that they can be used to find
		//other inventories. If some of the data is missing then stop syncing as the correct
		//inventories won't be able to be found.
		$canSync = true;
		$syncConditions = [];
		foreach ($syncOptions[$Model->alias] as $syncField) {
			if (empty($syncInventory[$Model->alias][$syncField])) {
				$canSync = false;
				break;
			}

			$syncConditions[$Model->alias . '.' . $syncField] = $syncInventory[$Model->alias][$syncField];
		}

		if (!$canSync) {
			return true;
		}

		//Add the join condition.
		$syncConditions[] = $Model->alias . '.' . $Model->primaryKey . ' = ' . $Inventory->alias . '.model_id';

		$syncInventoriesIds = $Inventory->find(
			'list',
			[
				'fields' => [
					'id',
					'id',
				],
				'joins' => [
					[
						'table' => $Model->tablePrefix . $Model->table,
						'alias' => $Model->alias,
						'conditions' => $syncConditions,
					],
				],
				'conditions' => [
					$Inventory->alias . '.id !=' => $syncInventory[$Inventory->alias]['id'],
				]
			]
		);

		if (empty($syncInventoriesIds)) {
			return true;
		}

		return $Inventory->updateAll(
			[
				$Inventory->alias . '.stock' => $syncInventory[$Inventory->alias]['stock'],
			],
			[
				$Inventory->alias . '.id' => $syncInventoriesIds,
			]
		);
	}
}
