<?php

App::uses('AuditLogPresenter', 'EvAuditLog.Lib');

class InventoryAuditLogPresenter extends AuditLogPresenter {

/**
 * The property name is the name of the database field that changed. This method turns the field name
 * into something readable.
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @return string Humanised property name.
 */
	public function presentPropertyName($audit, $auditDelta) {
		switch ($auditDelta['property_name']) {
			case 'warning_action':
				return 'Actions to take when warning stock level is reached';
			case 'oos_action':
				return 'Actions to take when stock level reaches 0';
			default:
				return InflectorExt::humanize($auditDelta['property_name']);
		}
	}

/**
 * Present a audit delta value. If the value is null return an empty string. If the model is present
 * on the audit then an attempt to use a plugin presenter will occur, otherwise the value is returned
 * without any modification.
 *
 * @param array $audit      The audit data.
 * @param array $auditDelta The aduit delta array.
 * @param string $value 	The default value
 * @return string The value to present.
 */
	public function presentValue($audit, $auditDelta, $value) {
		switch ($auditDelta['property_name']) {
			case 'warning_action':
				$actions = json_decode($value, true);

				if (empty($actions)) {
					$value = 'No actions';
					break;
				}

				$value = $this->_presentWarningActions($actions);
				break;
			case 'oos_action':
				$actions = json_decode($value, true);

				if (empty($actions)) {
					$value = 'No actions';
					break;
				}

				$value = $this->_presentOosActions($actions);
				break;
		}

		return parent::presentValue($audit, $auditDelta, $value);
	}

/**
 * If the property should appear in the log
 *
 * @param array $audit The audit data
 * @param array $auditDelta The delta data
 * @return bool True if it should appear, false if not
 */
	public function shouldPresentProperty($audit, $auditDelta) {
		switch ($auditDelta['property_name']) {
			case 'id':
			case 'model':
			case 'model_id':
				return false;
			default:
				return parent::shouldPresentProperty($audit, $auditDelta);
		}
	}

/**
 * Present the warning actions. List out each action as shown in the admin.
 *
 * @param array $actions A list of actions to present.
 * @return string The list to present.
 */
	protected function _presentWarningActions($actions) {
		$actionValues = '<ul>';

		foreach ($actions as $action) {
			$actionValues .= '<li>';

			$actionValues .= Configure::read('EvInventory.warningActions.' . $action);

			$actionValues .= '</li>';
		}

		return $actionValues .= '</ul>';
	}

/**
 * Present the out of stock actions. List out each action as shown in the admin.
 *
 * @param array $actions A list of actions to present.
 * @return string The list to present.
 */
	protected function _presentOosActions($actions) {
		$actionValues = '<ul>';

		foreach ($actions as $action) {
			$actionValues .= '<li>';

			$actionValues .= Configure::read('EvInventory.oosActions.' . $action);

			$actionValues .= '</li>';
		}

		return $actionValues .= '</ul>';
	}
}
