<?php

App::uses('Component', 'Controller');

App::uses('Impersonate', 'EvImpersonate.Lib');

class ImpersonateComponent extends Component {

	protected $_controller = null;

	public $defaultSettings = [
		// Define the model used by the Auth component.
		'userModel' => [
			'className' => 'EvCore.User',
			'alias' => 'User',
		],
		// Define the user group model used by the Auth component.
		// Accepted comparisons are < <= = != >= >
		'userGroupModel' => [
			'className' => 'EvCore.UserGroup',
			'alias' => 'UserGroup',
			'comparisonField' => 'level',
			'comparison' => '<', // impersonator [comparison] user
		],
		// Define any additional parameters for the query used to retrieve the
		// user.
		'params' => [
			'conditions' => [
				'User.user_group_id >' => 2
			],
			'contain' => [
				'UserGroup'
			]
		],
		// Define the query parameters for checking the current user can
		// impersonate other users.
		'canImpersonateParams' => [
			'conditions' => [
				'User.user_group_id <=' => 2
			]
		],
		//Define the permissions path for checking if the current user can impersonate
		//other users. %id will be replaced by the group id
		'canImpersonatePermission' => 'impersonations',
		'canImpersonateUserPermission' => 'impersonations/impersonateGroup%id',
		// Define the redirect route/url to go to after starting to impersonate
		// a user.
		'impersonateRedirect' => '/',
		// Define the redirect route/url to go to after ending an impersonation
		// session.
		'unimpersonateRedirect' => [
			'admin' => true,
			'plugin' => 'ev_core',
			'controller' => 'users',
			'action' => 'index'
		],
		//Define whether the unimpersonate method should logout the current user
		//if there is currently nobody impersonating them.
		'unimpersonateLogout' => true,
	];

/**
 * Called before the Controller::beforeFilter().
 *
 * @param Controller $controller Controller with components to initialize
 * @return void.
 * @see Component::initialize()
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);
		$this->_controller = $controller;
		$this->settings = Hash::merge($this->defaultSettings, $this->settings);

		$this->_controller->helpers[] = 'EvImpersonate.Impersonate';
	}

/**
 * Impersonate another user.
 *
 * @param int   $userId ID of the user we want to impersonate.
 * @param array $params Additional query parameters for user to impersonate.
 * @return void|bool Redirects if setting redirects are set. Otherwise true/false depending on impersonation success.
 */
	public function impersonate($userId, array $params = []) {
		// If we are already impersonating someone get the impersonator not the person being impersonated
		if ($this->isImpersonating()) {
			$impersonator = $this->getImpersonator();
		} else {
			$impersonator = $this->_controller->Auth->user();
		}

		if ($this->_canImpersonate($impersonator) === true) {
			// Get the user to impersonate.
			$this->_controller->loadModel($this->settings['userModel']['className']);
			if (!empty($this->settings['params'])) {
				$params = Hash::merge($this->settings['params'], $params);
			}
			$defaultParams = [
				'conditions' => [
					$this->_controller->{$this->settings['userModel']['alias']}->escapeField() => $userId
				]
			];
			$user = $this->_controller->{$this->settings['userModel']['alias']}->find('first', Hash::merge($defaultParams, $params));

			if (!empty($user) && $this->_canImpersonateUser($impersonator, $user)) {
				if ($this->isImpersonating()) {
					if ($this->getImpersonated($this->settings['userModel']['alias'] . '.id') == $user[$this->settings['userModel']['alias']]['id']) {
						// We are already impersonating this user - nothing else to do.
						if (!empty($this->settings['impersonateRedirect'])) {
							return $this->_controller->redirect($this->settings['impersonateRedirect']);
						} else {
							return true;
						}
					} else {
						// Already impersonating another user - unimpersonate the old one first.
						$this->unimpersonate();
					}
				}

				// Store the current auth user as the impersonator.
				$this->_controller->Session->write('Auth.Impersonator', $impersonator);

				$fullUser = $this->_controller->{$this->settings['userModel']['alias']}->getUserForLogin($user['User']['id']);

				// Change to recognised auth user to the impersonated user.
				$this->_controller->Auth->login($fullUser);

				// Redirect to ensure we are in the impersonated users session.
				if (!empty($this->settings['impersonateRedirect'])) {
					return $this->_controller->redirect($this->settings['impersonateRedirect']);
				} else {
					return true;
				}
			}
		}

		// We have failed to impersonate.
		// Unimpersonate any currently impersonated users
		$this->unimpersonate();

		$this->_controller->Session->setFlash('Unable to impersonate user!', 'flash_fail');
		if (!empty($this->settings['unimpersonateRedirect'])) {
			return $this->_controller->redirect($this->settings['unimpersonateRedirect']);
		}

		return false;
	}

/**
 * Stop impersonating a user and return to the original session.
 *
 * @return void|bool Redirects the user if setting redirects are set. Otheriwse true/false depeding on unimpersonation success.
 */
	public function unimpersonate() {
		if ($this->isImpersonating()) {
			// Restore the impersonator's login as the current user.
			$user = $this->getImpersonator();
			$currentUser = $this->_controller->Auth->user();

			//Login the impersonating user back in.
			if ($user['User']['id'] != $currentUser['User']['id']) {
				$originalLoggedIn = $this->_controller->Auth->login($user);
			} else {
				//The user is already logged in, tidy up impersonating data.
				$this->removeImpersonator();
				$originalLoggedIn = $user;
			}

			if (!empty($originalLoggedIn) && $originalLoggedIn['User']['id'] != $currentUser['User']['id']) {
				$this->_controller->Flash->warning(
					[
						'title' => 'You are no longer impersonating',
						'description' => 'You are no longer impersonating the user: ' . $currentUser['User']['email'],
					]
				);
			} else {
				//Failed to remove the impersonating user so login the current user again to ensure the correct
				//user is being impersonated.
				$this->_controller->Auth->login($currentUser);
				$this->_controller->Flash->fail(
					[
						'title' => 'Failed to unimpersonate',
						'description' => 'You are still impersonating the user: ' . $currentUser['User']['email'],
					]
				);
			}

			if (!empty($this->settings['unimpersonateRedirect'])) {
				return $this->_controller->redirect($this->settings['unimpersonateRedirect']);
			} else {
				return $originalLoggedIn;
			}
		}

		if ($this->settings['unimpersonateLogout']) {
			return $this->_controller->Auth->logout();
		}

		return null;
	}

/**
 * Check if an impersonator is allowed to impersonate another user.
 *
 * @param array $impersonator User data from the auth session.
 * @return bool
 */
	public function canImpersonate($impersonator) {
		return $this->_canImpersonate($impersonator);
	}

/**
 * Check if an impersonator is allowed to impersonate a given user.
 *
 * @param array $impersonator User data from the auth session.
 * @param array $user User data for a user to impersonate
 * @return bool
 */
	public function canImpersonateUser($impersonator, $user) {
		return $this->_canImpersonateUser($impersonator, $user);
	}

/**
 * Check if an impersonator is allowed to impersonate another user by checking a specified user
 * permission. This temporarily sets the permission setting the permission provided and is
 * reset after the check. If no permission is provided then the check is carried out as normal.
 *
 * @param array  $impersonator The user data of the user that will be impersonating.
 * @param string $permission   The permission to check. Uses setting value if no permission is provided.
 * @return bool True if the user can impersonate, false otherwise.
 */
	public function canImpersonateByPermission($impersonator, $permission = null) {
		if ($permission !== null) {
			$originalPermission = $this->settings['canImpersonatePermission'];
			$this->settings['canImpersonatePermission'] = $permission;
		}

		$canImpersonate = $this->_canImpersonatePermission($impersonator);

		if (!empty($originalPermission)) {
			$this->settings['canImpersonatePermission'] = $originalPermission;
		}

		return $canImpersonate;
	}

/**
 * Check if the impersonator is allowed to impersonate other users.
 *
 * @param array $impersonator User data from the auth session.
 * @return bool
 */
	protected function _canImpersonate($impersonator) {
		if (!empty(Configure::read('EvImpersonate.useAclPermissions'))) {
			return $this->_canImpersonatePermission($impersonator);
		} else {
			return $this->_canImpersonateUserLevel($impersonator);
		}
	}

/**
 * Check if the impersonator is allowed to impersonate a given user.
 *
 * @param array $impersonator User data from the auth session.
 * @param array $user User data for the user to impersonate.
 * @return bool
 */
	protected function _canImpersonateUser($impersonator, $user) {
		if (!empty(Configure::read('EvImpersonate.useAclPermissions'))) {
			return $this->_canImpersonateUserPermission($impersonator, $user);
		} else {
			return $this->_canImpersonateUserUserLevel($impersonator, $user);
		}
	}

/**
 * Check if a user can impersonate other users by checking their user data.
 *
 * @param array $impersonator User data from the auth session.
 * @return bool
 */
	protected function _canImpersonateUserLevel($impersonator) {
		$primaryKey = $this->_controller->{$this->settings['userModel']['alias']}->primaryKey;
		$alias = $this->_controller->{$this->settings['userModel']['alias']}->alias;
		$defaultParams = [
			'conditions' => [
				$this->_controller->{$this->settings['userModel']['alias']}->escapeField() => $impersonator[$alias][$primaryKey]
			]
		];
		return (bool)$this->_controller->{$this->settings['userModel']['alias']}->find(
			'first',
			Hash::merge($defaultParams, $this->settings['canImpersonateParams'])
		);
	}

/**
 * Check if a user can impersonate another users by checking their user data.
 *
 * @param array $impersonator User data from the auth session.
 * @param array $user User data for the user to impersonate.
 * @return bool
 */
	protected function _canImpersonateUserUserLevel($impersonator, $user) {
		// Check we have the correct permissions
		if (!$this->_canImpersonateUserLevel($impersonator)) {
			return false;
		}

		// Compare the user group levels to
		$comparisonField = $this->settings['userGroupModel']['comparisonField'];
		$alias = $this->settings['userGroupModel']['alias'];

		$impersonatorLevel = $impersonator[$alias][$comparisonField];
		$userLevel = $user[$alias][$comparisonField];

		switch($this->settings['userGroupModel']['comparison']) {
			case '<':
				return $impersonatorLevel < $userLevel;
			case '<=':
				return $impersonatorLevel <= $userLevel;
			case '=':
				return $impersonatorLevel == $userLevel;
			case '>=':
				return $impersonatorLevel >= $userLevel;
			case '>':
				return $impersonatorLevel > $userLevel;
			case '!=':
				return $impersonatorLevel != $userLevel;
		}

		return false;
	}

/**
 * Check if a user cna impersonate other users by checking their permissions.
 *
 * @param array $impersonator User data from the auth session.
 * @return bool
 */
	protected function _canImpersonatePermission($impersonator) {
		if (empty($impersonator['User']['user_group_id'])) {
			return false;
		}

		return $this->_controller->Acl->check(
			[
				'model' => 'UserGroup',
				'foreign_key' => $impersonator['User']['user_group_id']
			],
			$this->settings['canImpersonatePermission']
		);
	}

/**
 * Check if a user cna impersonate other users by checking their permissions.
 *
 * @param array $impersonator User data from the auth session.
 * @param array $user User data for the user to impersonate.
 * @return bool
 */
	protected function _canImpersonateUserPermission($impersonator, $user) {
		if (!$this->_canImpersonatePermission($impersonator) || empty($impersonator['User']['user_group_id']) || empty($user['User']['user_group_id'])) {
			return false;
		}
		return $this->_controller->Acl->check(
			[
				'model' => 'UserGroup',
				'foreign_key' => $impersonator['User']['user_group_id']
			],
			str_replace('%id', $user['User']['user_group_id'], $this->settings['canImpersonateUserPermission'])
		);
	}

/**
 * Check if the current user is impersonating another user by checking the session data.
 *
 * @return bool.
 */
	public function isImpersonating() {
		return Impersonate::isImpersonating();
	}

/**
 * Get the current impersonator.
 *
 * @param string $path The path to get a specific field from the impersonating user.
 * @return mixed The impersonating user, or specific field if $path is provided.
 */
	public function getImpersonator($path = null) {
		return Impersonate::getImpersonator($path);
	}

/**
 * Get the current impersonated user. This would be the user currently in the auth session.
 *
 * @param string $path The path to get a specific field from the impersonated user.
 * @return mixed The impersonated user, or specific field if $path is provided.
 */
	public function getImpersonated($path = null) {
		return Impersonate::getImpersonated($path);
	}

/**
 * Remove the current impersonator from the session. This is normally handled by the unimpersonate
 * method but this method can be used separately to tidy up the session data to stop the user
 * impersonating themself.
 *
 * @return bool. True if removed, otherwise false.
 */
	public function removeImpersonator() {
		return Impersonate::removeImpersonator();
	}
}
