<?php

App::uses('AppHelper', 'View/Helper');

App::uses('ImgCaching', 'EvImg.Lib');

class ImgHelper extends AppHelper {

	public $helpers = array(
		'Html'
	);

	public function __construct(View $View, $settings = []) {
		if ($settings == null) {
			$settings = array();
		}

		parent::__construct($View, $settings);

		$defaults = [
			'uploadDirectory' => 'files' . DS . 'image',
			'remoteCacheDirectory' => 'files' . DS . 'remote_cache',
			'cacheDirectory' => 'files' . DS . 'img_cache',
			'memoryLimit' => '1024M',
			'quality' => 75,
			'fullBase' => false,
			'webp' => [
				'quality' => 80, // Smaller than JPEG@75% and better overall quality
				'quality_png_to_webp' => 90, // PNG > WEBP conversion at 80 has visible artifacts
			],
		];

		$this->settings = array_merge($defaults, $settings);

		if (Configure::read('EvImg.lazyload')) {
			$this->Html->script('EvImg.lazyload.js', ['block' => 'script']);
		}

		return;
	}

	/**
	 * Resizes an image.
	 *
	 * @param array|string Image to resize
	 * @param array $dimensions Dimensions for resized image
	 * @param array $options Img tag attributes
	 * @param string $context Image context (defines a directory to place image)
	 * @return string fully qualified img tag
	 */
	public function resize($image, $dimensions = null, $options = [], $context = null) {
		$imageCacheKey = [
			'image' => $image,
			'dimensions' => $dimensions,
			'options' => $options,
			'context' => $context,
		];

		if (!empty($image['alt']) && empty($options['alt'])) {
			$options['alt'] = $image['alt'];
		} elseif (empty($options['alt']) && !empty($image['name'])) {
			$options['alt'] = $image['name'];
		}

		// Make sure we apply the dimension attributes to the output image to
		// improve browser performance (this allows the browser to know how
		// much space to allocate the image before it fully renders).
		if (empty($options['width']) && !empty($dimensions['width'])) {
			$options['width'] = $dimensions['width'];
		}
		if (empty($options['height']) && !empty($dimensions['height'])) {
			$options['height'] = $dimensions['height'];
		}

		if (Configure::read('EvImg.webp')) {
			$image = $this->_generateWebpImage($image, $dimensions, $options, $context);
		} else {
			$fullBase = isset($options['fullBase']) ? $options['fullBase'] : $this->settings['fullBase'];
			$path = $this->path($image, $dimensions, $context, $fullBase);
			unset($options['fullBase']);

			if (Configure::read('EvImg.forceImageSizeAttributes')) {
				$options = $this->_addImageSizeAttributes($path, $options);
			}

			$this->_lazyLoadImage($image, $dimensions, $options, $context, $path);
			$image = $this->Html->image($path, $options);
		}

		if ($this->_canCache('resize')) {
			$this->_setCache('resize', $imageCacheKey, $image);
		}

		return $image;
	}

	/**
	 * Wrapper method so that we can directly get a path to a cached resized image. This is to get
	 * round issues with the second parameter needing to be passed by reference.
	 *
	 * @param array|string Image to resize
	 * @param array $dimensions Dimensions for resized image
	 * @param string $context Image context (defines a directory to place image)
	 * @return string Path to resized image
	 */
	public function path($image, $dimensions = null, $context = null, $fullBase = false, $isWebp = false) {
		$path = $this->_path($image, $dimensions, $context, $isWebp);
		return $this->Html->assetUrl($path, compact('fullBase'));
	}

/**
 * Returns the path to a cached resized image.
 *
 * @param array|string $image       The image to resize.
 * @param array        &$dimensions Dimensions for resized image.
 * @param string       $context     Image context (defines a directory to place image).
 * @return string Path to resized image.
 */
	protected function _path($image, &$dimensions = null, $context = null, $isWebp = false) {
		$imageCacheKey = [
			'image' => $image,
			'dimensions' => $dimensions,
			'context' => $context,
		];

		if ($isWebp) {
			$imageCacheKey['webp'] = 'webp';
		}

		if ($this->_hasCache('imgPath', $imageCacheKey)) {
			return $this->_getCache('imgPath', $imageCacheKey);
		}

		$isPdf = !empty($dimensions['pdf']);

		$cacheFilepath = $this->settings['cacheDirectory'] . DS;

		if (! empty($image)) {
			if (is_array($image)) {

				if (strpos($image['filename'], '://') !== false) {
					$originalFilepath = $this->_cacheRemoteImage($image['filename']);
					$originalFilename = basename($originalFilepath);
					$originalFilepath = str_replace($originalFilename, '', $originalFilepath);
				} else {
					$originalFilename = $image['filename'];
					$originalFilepath = $this->settings['uploadDirectory'] . DS . $image['id'] . DS;
				}

				$cacheFilepath .= ($context !== null ? $context : $image['id']) . DS;

				$isPdf = in_array($image['type'], ['application/pdf', 'application/postscript']);

			} elseif (strpos($image, '://') !== false) {

				$originalFilepath = $this->_cacheRemoteImage($image);
				$originalFilename = basename($originalFilepath);
				$originalFilepath = str_replace($originalFilename, '', $originalFilepath);
				$cacheFilepath .= $context !== null ? $context . DS : '';

			} else {

				$parts = pathinfo($image);
				$originalFilename = $parts['basename'];
				$originalFilepath = DS . $parts['dirname'] . DS;
				$cacheFilepath .= $context !== null ? $context . DS : '';

			}
		}

		// If no image exists, setup no image placeholder
		if (
			empty($image) ||
			! file_exists(WWW_ROOT . $originalFilepath . $originalFilename)
		) {
			$originalFilepath = 'img' . DS;

			$originalFilename = 'no-image.png';

			// check whether we have a themed version available
			if (! empty($this->theme)) {
				$themeDir = 'theme' . DS . $this->theme;

				// check whether the no-image.png file exists for the theme,
				// replacing the $originalFilepath when available
				if (file_exists($themeDir . DS . $originalFilepath . $originalFilename)) {
					$originalFilepath = $themeDir . DS . $originalFilepath;
				}
			}
		}

		$originalFileFullPath = WWW_ROOT . $originalFilepath . $originalFilename;

		// Create a filename for the cached image.
		$cacheOptions = array_intersect_key(
			$dimensions,
			array_flip(array('width', 'height', 'crop', 'fit', 'blur'))
		);

		// Flatten options into a single-level array so we can implode them all
		$cacheOptionsFlattened = [];
		array_walk_recursive($cacheOptions, function ($v) use (&$cacheOptionsFlattened) {
			$cacheOptionsFlattened[] = $v;
		});

		$cacheFilename = $cacheFilepath . implode('_', $cacheOptionsFlattened) . '_' . $originalFilename;

		$convertToPng = $isWebp === false && $this->_shouldConvertToPng($originalFileFullPath);
		if ($convertToPng) {
			$cacheFilename .= '.png';
		} elseif ($isWebp) {
			$cacheFilename .= '.webp';
		}

		if (!file_exists(WWW_ROOT . $cacheFilename)) {

			$Image = new imagick();
			try {
				if ($isPdf === true) {
					$Image->setResolution(300, 300);
					$Image->setBackgroundColor('#ffffff');
					$Image->readImage($originalFileFullPath . '[0]');
					$Image = $Image->mergeImageLayers(Imagick::LAYERMETHOD_FLATTEN);
				} else {
					$Image->readImage($originalFileFullPath);

					$orientation = $Image->getImageOrientation();

					switch ($orientation) {
						case imagick::ORIENTATION_BOTTOMRIGHT:
							$Image->rotateimage('#000', 180); // rotate 180 degrees
							break;

						case imagick::ORIENTATION_RIGHTTOP:
							$Image->rotateimage('#000', 90); // rotate 90 degrees CW
							break;

						case imagick::ORIENTATION_LEFTBOTTOM:
							$Image->rotateimage('#000', -90); // rotate 90 degrees CCW
							break;
					}
				}
			} catch (ImagickException $e) {
				CakeLog::write('error', 'EvImg: Invalid Image Format for file ' . $originalFilepath . $originalFilename);

				// Show placeholder image instead.
				$Image->readImage(WWW_ROOT . DS . 'img' . DS . 'no-image.png');
			}

			$height = $Image->getImageHeight();
			$width = $Image->getImageWidth();

			if (!empty($dimensions['background'])) {
				$Image->setbackgroundcolor($dimensions['background']);
				$Image->setimagebackgroundcolor($dimensions['background']);
			}

			if (!empty($dimensions['crop'])) {

				if (!empty($dimensions['cropPosition'])) {
					$this->_cropFromPosition($Image, $dimensions, $width, $height);
				} else {
					// default to centre if no crop position is specified
					$Image->cropThumbnailImage($dimensions['width'], $dimensions['height']);
				}

			} elseif (!empty($dimensions['fill'])) {

				// Fully resize to given width and/or height even if original image is smaller
				$Image->thumbnailImage(
					!empty($dimensions['width']) ? (int)$dimensions['width'] : 0,
					!empty($dimensions['height']) ? (int)$dimensions['height'] : 0,
					false,
					true
				);

			} elseif (empty($dimensions['width'])) {

				if ((int)$dimensions['height'] < $height) {
					$Image->thumbnailImage(0, (int)$dimensions['height']);
				}

			} elseif (empty($dimensions['height'])) {

				if ((int)$dimensions['width'] < $width) {
					$Image->thumbnailImage((int)$dimensions['width'], 0);
				}

			} elseif (!empty($dimensions['fit'])) {
				// Check if the original image is smaller than the desired size. If it is, pad around the image with a
				// border.
				if ($width <= $dimensions['width'] && $height < $dimensions['height']) {

					$background = (isset($dimensions['background']) ? $dimensions['background'] : 'white');

					$Image->scaleImage($dimensions['width'], $dimensions['height'], true);

					$borderWidth = max(($dimensions['width'] - $Image->getImageWidth()) / 2, 0);
					$borderHeight = max(($dimensions['height'] - $Image->getImageHeight()) / 2, 0);

					$Image->borderImage($background, $borderWidth, $borderHeight);
				} else {
					// Resize image to given dimensions but doesn't stretch larger than original
					$Image->thumbnailImage($dimensions['width'], $dimensions['height'], true, true);
				}
			} elseif ($height <= $width) {

				if ((int)$dimensions['width'] < $width) {
					$Image->thumbnailImage((int)$dimensions['width'], 0);
				}

			} else {

				if ((int)$dimensions['height'] < $height) {
					$Image->thumbnailImage(0, (int)$dimensions['height']);
				}

			}

			if (!empty($dimensions['blur'])) {
				$radius = isset($dimensions['blur']['radius']) ? $dimensions['blur']['radius'] : 5;
				$sigma = isset($dimensions['blur']['sigma']) ? $dimensions['blur']['sigma'] : 3;
				$Image->blurImage($radius, $sigma);
			}

			if (!file_exists(WWW_ROOT . $cacheFilepath)) {
				// cache directory doesn't exist - create it now
				mkdir(WWW_ROOT . $cacheFilepath, 0777, true);
			}

			$dimensions = $Image->getImageGeometry();

			if ($isWebp && mime_content_type($originalFileFullPath) === 'image/png') {
				$Image->setImageCompressionQuality($this->settings['webp']['quality_png_to_webp']);
			} else {
				$Image->setImageCompressionQuality($isWebp ? $this->settings['webp']['quality'] : $this->settings['quality']);
			}

			// Strip image of all profiles and comments to reduce size.
			$Image->stripImage();

			// Convert non-websafe formats to png. This is anything other than png, jpg, svg and gif.
			if ($convertToPng) {
				$Image->setImageFormat('png');
			}

			// Force webp output?
			// $Image->setImageFormat('webp') is not always recognised even if webp support is available
			// Prefixing the output with 'webp:' will always have desired result
			$outputPrefix = '';
			if ($isWebp) {
				$outputPrefix = 'webp:';
			}

			$Image->writeImage($outputPrefix . WWW_ROOT . $cacheFilename);

		} else {
			// Correct image dimensions to be used on the <img/> tag.
			$Image = new imagick();
			$Image->readImage(WWW_ROOT . $cacheFilename);
			$dimensions = $Image->getImageGeometry();
		}

		$cdnAssetsUrl = Configure::read('cdn_assets_url');
		if (! empty($cdnAssetsUrl)) {
			//If a cdn asset url is available, use that instead of the base url.
			$cacheFilename = $cdnAssetsUrl . '/' . $cacheFilename;
		} else {
			$cacheFilename = '/' . $cacheFilename;
		}

		// Return the image path (make sure we use forward and not back slashes
		// for the image, this is an issue when using a Windows server).
		$cacheFilename = str_replace('\\', '/', $cacheFilename);

		if ($this->_canCache('imgPath')) {
			$this->_setCache('imgPath', $imageCacheKey, $cacheFilename);
		}

		return $cacheFilename;
	}

	public function getDimensions($image) {
		$Image = new imagick();
		$Image->readImage($image['filepath']);
		$dimensions = $Image->getImageGeometry();

		return $dimensions;
	}

/**
 * Replaces `src` on all img tags with a `data-src` attribute to allow for lazy loading
 * of images. Images already lazy loaded will not be modified.
 *
 * @param string $htmlString The html string to replace images in.
 * @param array  $options    Img tag attributes.
 * @return string.
 */
	public function lazyLoadImageTags($htmlString, $options = []) {
		if (
			!$this->_canLazyLoad($options)
			|| strpos($htmlString, '<img') === false
		) {
			return $htmlString;
		}

		// We select two groups, then keep the first group and replace the second group (src=) with our replacement string.
		// We use a negative look-behind to ignore any instances of data-src that would cause multiple instances of the
		// replacement string to be included.

		$regex = '/';
		$regex .= '(';

		// Group 1:
		// Targets img tag and selects everything up to the src attribute.
		// Ignoring any with data-lazy="true" already present.
		$regex .= '<img';
		// Negative lookahead for any number of data-lazy attributes on the img tag.
		// To prevent the src being overwritten with the placeholder when the img
		// has already been processed.
		$regex .= '(?![^>]+data-lazy)';
		// Grab everything from the start of the img tag to the first `src` attribute.
		$regex .= '.*?';
		$regex .= ')';

		// Group 2:
		// Targets and selects the src attribute itself, up to the start of it's value.
		$regex .= '(src=)';
		$regex .= '/';

		// $1 signifies the first group from the above regex
		// that is appended to the start of our new attributes.
		$replacement = '$1data-lazy="true" src="' . $this->_getLazyLoadPlaceholder($options) . '" data-src=';

		return preg_replace($regex, $replacement, $htmlString);
	}

/**
 * Determines if a given file will be converted to a PNG.
 *
 * @param string $filepath The filepath to the image to check
 * @return boolean true for convert, false to use the current format
 */
	protected function _shouldConvertToPng($filepath) {
		// Check the mime type. If it is not a jpg, gif svg or png convert to png.
		// For mimetypes see here: https://www.sitepoint.com/mime-types-complete-list/
		switch (mime_content_type( $filepath )) {
			case 'image/png':
			case 'image/jpeg':
			case 'image/gif':
			case 'image/svg+xml':
				// Do nothing. These formats will be preserved;
				return false;
			default:
				// Any other format should be converted to png
				return true;
		}
	}

/**
 * Resizes image to user defined dimensions, either by width or height depending on which is greater
 * @param  $Image - Imagick instance
 * @param  $dimensions - user defined dimensions
 * @param  $width - image width
 * @param  $height - image height
 * @return $Image - modified Imagick instance
 */
	protected function _scaleDown($Image, $dimensions, $width, $height) {
		if ((int)$dimensions['width'] < $width) {
			$Image->thumbnailImage((int)$dimensions['width'], 0);
		} elseif ((int)$dimensions['height'] < $height) {
			$Image->thumbnailImage(0, (int)$dimensions['height']);
		}

		return $Image;
	}

/**
 * Handles cropping an image from a specified position
 * The default crop method used doesn't allow for dimensions to be passed through so we break it up into 2 Imagick methods
 * Imagick::thumbnailImage & Imagick::cropImage
 * @param  $Image - Imagick instance
 * @param  $dimensions - user defined dimensions
 * @param  $width - image width
 * @param  $height - image height
 * @return $Image - modified Imagick instance
 */
	protected function _cropFromPosition($Image, $dimensions, $width, $height) {
		// cropPosition can either be a string value (top|middle|bottom) - or it can be an array with x and y keys
		if (is_array($dimensions['cropPosition'])) {
			// default x and y to 0 if neither are passed
			$x = !empty($dimensions['cropPosition']['x']) ? $dimensions['cropPosition']['x'] : 0;
			$y = !empty($dimensions['cropPosition']['y']) ? $dimensions['cropPosition']['y'] : 0;

			$this->_scaleDown($Image, $dimensions, $width, $height);
			$Image->cropImage($dimensions['width'], $dimensions['height'], $x, $y);
		} else {
			// crop from particular position according to passed value
			switch($dimensions['cropPosition']) {
				case 'top':
					$this->_scaleDown($Image, $dimensions, $width, $height);
					$Image->cropImage($dimensions['width'], $dimensions['height'], 0, 0);
					break;
				case 'bottom':
					$this->_scaleDown($Image, $dimensions, $width, $height);
					// Get the height after resize
					$height = $Image->getImageHeight();
					// 0, 0 is top left. So in order to crop from the bottom we set the y co-ordinate as (image height - expected height)
					$Image->cropImage($dimensions['width'], $dimensions['height'], 0, $height - $dimensions['height']);
					break;
				case 'middle':
				default:
					$Image->cropThumbnailImage($dimensions['width'], $dimensions['height']);
					break;
			}
		}

		return $Image;
	}

/**
 * Locally cache a remote image so that we can resize it.
 * @param string $file File path
 * @return string Locally hosted file path
 */
	protected function _cacheRemoteImage($file) {
		if (! file_exists(WWW_ROOT . $this->settings['remoteCacheDirectory'])) {
			mkdir(WWW_ROOT . $this->settings['remoteCacheDirectory'], 0777, true);
		}
		try {
			$filepath = $this->settings['remoteCacheDirectory'] . DS . md5($file);
			if (! file_exists($filepath)) {
				// Surpress errors when copying, we will catch an exception and return the
				// placeholder image if the file cannot be retrieved.
				@copy($file, $filepath);
			}
		} catch (Exception $e) {
			$filepath = 'img' . DS . 'no-image.png';
		}

		return $filepath;
	}

/**
 * Check if a cache can be created for a specific cache type.
 *
 * @param string $cacheType The type of cache to find in the config.
 * @return bool.
 */
	protected function _canCache($cacheType) {
		return ImgCaching::canCache($cacheType);
	}

/**
 * Set a cached value for an image for the cache type.
 *
 * @param string       $cacheType The type of img cache to set.
 * @param string|array $imageKey The key of the image. A combination of the image data, dimensions and content. If a
 *                                 string key isn't provided then attempt to generate it.
 * @param mixed        $cacheValue The value to set in the cache against the cache type.
 * @return void.
 */
	protected function _setCache($cacheType, $imageKey, $cacheValue) {
		return ImgCaching::setCache($cacheType, $imageKey, $cacheValue);
	}

/**
 * Check if a cached value exists for a cache type of an image.
 *
 * @param string       $cacheType The type of img cache to set.
 * @param string|array $imageKey  The key of the image. A combination of the image data, dimensions and content. If a
 *                                string key isn't provided then attempt to generate it.
 * @return bool.
 */
	protected function _hasCache($cacheType, $imageKey) {
		return ImgCaching::hasCache($cacheType, $imageKey);
	}

/**
 * Get the cached value for an image with a cache type. Null is returned if no cache is found.
 *
 * @param string       $cacheType The type of img cache to set.
 * @param string|array $imageKey  The key of the image. A combination of the image data, dimensions and content. If a
 *                                string key isn't provided then attempt to generate it.
 * @return mixed.
 */
	protected function _getCache($cacheType, $imageKey) {
		return ImgCaching::getCache($cacheType, $imageKey);
	}

/**
 * Check if lazy loaded images can be used. They must be enabled in the config,
 * the image not be defined as an immediate image and the image must not be
 * being rendered for ajax or admin requests.
 *
 * @param array $options Img tag attributes.
 * @return bool.
 */
	protected function _canLazyLoad($options) {
		return Configure::read('EvImg.lazyload')
			&& empty($options['immediate'])
			&& !in_array(
					strtolower($this->theme),
					[
						'admin',
						'ajax',
					]
				);
	}

/**
 * Get the placeholder image for a lazy loaded image. The image can be provided
 * by the "placeholder" option or be set in the plugin config file. Otherwise
 * the placeholder defaults to a blank transparent image that fills the space
 * of the lazy loaded image.
 *
 * @param array $options Img tag attributes.
 * @return string.
 */
	protected function _getLazyLoadPlaceholder($options) {
		if (!empty($options['placeholder'])) {
			return $options['placeholder'];
		}

		if (Configure::read('EvImg.lazyLoadPlaceholder')) {
			return $this->assetUrl(
				Configure::read('EvImg.lazyLoadPlaceholder'),
				[
					'fullBase' => true,
				]
			);
		}

		return "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 11 14'%3E%3C/svg%3E";
	}

/**
 * Replace a resized image with a resized lazy loaded image. The resized path
 * becomes the placeholder that will be shown until the resized image is loaded.
 * The image element attributes are modified by adding the required lazy load
 * data attributes.
 *
 * For a resized image to be replaced by a lazy loaded image, lazy loading
 * needs to be enabled in the plugin config file.
 *
 * If a specific image must not be lazy loaded (e.g. header logo) then the
 * "immediate" option can be provided.
 *
 * By default the placeholder is an empty and transparent svg that will replace
 * the full size of the resized image. This placeholder can be customised per
 * project using the "lazyLoadPlaceholder" config value or by providing a full
 * asset url with the "placeholder" option.
 *
 * @param array|string &$image      Image to resize.
 * @param array        &$dimensions Dimensions for resized image.
 * @param array        &$options    Img tag attributes.
 * @param string       &$context    Image context (defines a directory to place image).
 * @param string       &$path       The path of the resized image.
 * @return void.
 */
	protected function _lazyLoadImage(&$image, &$dimensions, &$options, &$context, &$path) {
		$imageOptions = $options;

		//Remove the attributes from being added to the image element.
		unset($options['immediate']);
		unset($options['placeholder']);

		if (!$this->_canLazyLoad($imageOptions)) {
			//Do not turn the resized image into a lazy loaded image.
			return;
		}

		$options['data-src'] = $path;
		$options['data-lazy'] = true;
		$path = $this->_getLazyLoadPlaceholder($imageOptions);
	}

/**
 * Generate the webp image using a `<picture>` tag so that a fallback to the image's original format
 * can be defined for older browsers.
 *
 * @param array|string $image Image being resized
 * @param array $dimensions Dimensions for resized image
 * @param array $options img tag attributes
 * @param string $context Image context (defines a directory to place image)
 * @return string
 */
	protected function _generateWebpImage($image, $dimensions = [], $options = [], $context = null) {
		$fullBase = isset($options['fullBase']) ? $options['fullBase'] : $this->settings['fullBase'];
		$mainImagePath = $this->path($image, $dimensions, $context, $fullBase);
		$webpImagePath = $this->path($image, $dimensions, $context, $fullBase, true);
		unset($options['fullBase']);

		if (Configure::read('EvImg.forceImageSizeAttributes')) {
			$options = $this->_addImageSizeAttributes($mainImagePath, $options);
		}

		// When using the `<picture>` tag we can't use the JS lazy-loading solution used by the
		// plugin. Instead, we'll fallback to using the native browser functionality which is mostly
		// supported by the main browsers now.
		if ($this->_canLazyLoad($options)) {
			$options['loading'] = 'lazy';
		}
		unset($options['immediate']);
		unset($options['placeholder']);

		$mainImage = $this->Html->image($mainImagePath, $options);

		return $this->Html->tag(
			'picture',
			$this->Html->tag('source', null, ['srcset' => $webpImagePath, 'type' => 'image/webp'])
				. $mainImage
		);
	}

/**
 * Ensure width and height attrs are set in image tag
 *
 * @param string $imagePath The path to the rendered image
 * @param array $options The image options
 * @return array $options The (potentially modified) image options
 */
	protected function _addImageSizeAttributes($imagePath, $options) {
		if (isset($options['width']) && isset($options['height'])) {
			return $options;
		}

		$fullImagePath = APP . WEBROOT_DIR . str_replace('/', DS, strtok($imagePath, '?'));
		$imageSize = $this->getDimensions(['filepath' => $fullImagePath]);

		if (!isset($options['width'])) {
			$options['width'] = $imageSize['width'];
		}

		if (!isset($options['height'])) {
			$options['height'] = $imageSize['height'];
		}

		return $options;
	}
}
