<?php

class ImgCaching {

	protected static $_cachable = [];

	const CACHE_CONFIG = 'EvImg.ImgCaching';

/**
 * Check if a cache can be created for a specific cache type.
 *
 * @param string $cacheType The type of cache to find in the config.
 * @return bool.
 */
	public static function canCache($cacheType) {
		if (isset(self::$_cachable[$cacheType])) {
			return self::$_cachable[$cacheType];
		}

		$cacheConfig = Configure::read('EvImg.cache');

		if (!is_array($cacheConfig) || empty($cacheConfig) || !isset($cacheConfig[$cacheType])) {
			self::$_cachable[$cacheType] = false;
			return false;
		}

		$cachableType = $cacheConfig[$cacheType];

		if ($cachableType === false) {
			self::$_cachable[$cacheType] = false;
			return false;
		}

		if ($cachableType === true) {
			self::$_cachable[$cacheType] = true;
			return true;
		}

		if (!is_string($cachableType)) {
			if (empty($cachableType)) {
				self::$_cachable[$cacheType] = false;
				return false;
			} else {
				self::$_cachable[$cacheType] = true;
				return true;
			}
		}

		//Check that the environment matches
		if (strtoupper($cachableType) === Configure::read('app.environment')) {
			self::$_cachable[$cacheType] = true;
			return true;
		}

		self::$_cachable[$cacheType] = false;
		return false;
	}

/**
 * Set a cached value for an image for the cache type.
 *
 * @param string       $cacheType  The type of img cache to set.
 * @param string|array $imageKey   The key of the image. A combination of the image data, dimensions and content. If a
 *                                 string key isn't provided then attempt to generate it.
 * @param mixed        $cacheValue The value to set in the cache against the cache type.
 * @return void.
 */
	public static function setCache($cacheType, $imageKey, $cacheValue) {
		if (!self::canCache($cacheType)) {
			return;
		}

		$cacheKey = self::_generateCacheKey($cacheType, $imageKey);

		if (empty($cacheKey)) {
			return;
		}

		Cache::write($cacheKey, $cacheValue, self::CACHE_CONFIG);
	}

/**
 * Check if a cached value exists for a cache type of an image.
 *
 * @param string       $cacheType The type of img cache to set.
 * @param string|array $imageKey  The key of the image. A combination of the image data, dimensions and content. If a
 *                                string key isn't provided then attempt to generate it.
 * @return bool.
 */
	public static function hasCache($cacheType, $imageKey) {
		if (!self::canCache($cacheType)) {
			return false;
		}

		$cacheKey = self::_generateCacheKey($cacheType, $imageKey);

		if (empty($cacheKey)) {
			return false;
		}

		$cacheValue = Cache::read($cacheKey, self::CACHE_CONFIG);

		return !empty($cacheValue);
	}

/**
 * Get the cached value for an image with a cache type. Null is returned if no cache is found.
 *
 * @param string       $cacheType The type of img cache to set.
 * @param string|array $imageKey  The key of the image. A combination of the image data, dimensions and content. If a
 *                                string key isn't provided then attempt to generate it.
 * @return mixed.
 */
	public static function getCache($cacheType, $imageKey) {
		if (!self::hasCache($cacheType, $imageKey)) {
			return null;
		}

		$cacheKey = self::_generateCacheKey($cacheType, $imageKey);

		if (empty($cacheKey)) {
			return null;
		}

		$cacheValue = Cache::read($cacheKey, self::CACHE_CONFIG);

		return $cacheValue;
	}

/**
 * Generate an image cache key from an array of image key options. The options should contain:
 * - 'image' => The image data of the image being cached.
 * - 'dimensions' => The dimensions of the image being cached.
 * - 'context' => The context of the image being cached.
 *
 * @param array $imageKeyOptions An array of image key options.
 * @return string.
 */
	protected static function _generateImageCacheKey($imageKeyOptions) {
		if (empty($imageKeyOptions['dimensions'])) {
			return '';
		}

		if (empty($imageKeyOptions['image'])) {
			$imageKeyOptions['image'] = 'no-image.png';
		}

		if (empty($imageKeyOptions['context'])) {
			$imageKeyOptions['context'] = '';
		}

		$imageKeyOptions = Hash::flatten($imageKeyOptions);

		ksort($imageKeyOptions);

		return hash('sha256', http_build_query($imageKeyOptions));
	}

/**
 * Generate the key for a cached value based on the cache type and the image key.
 *
 * @param string       $cacheType The type of img cache to set.
 * @param string|array $imageKey  The key of the image. A combination of the image data, dimensions and content. If a
 *                                string key isn't provided then attempt to generate it.
 * @return string.
 */
	protected static function _generateCacheKey($cacheType, $imageKey) {
		if (!is_string($imageKey)) {
			$imageKey = self::_generateImageCacheKey($imageKey);
		}

		if (empty($imageKey)) {
			return '';
		}

		return $cacheType . '_' . $imageKey;
	}
}
