<?php

/**
 * Smart image helper
 */
class ImageHelper extends AppHelper {

	public $helpers = array(
		'Html'
	);


	public $settings = array();

	private $options = array();

/**
 * Helper constructor
 *
 * @param $View
 *
 * @param $settings		mixed		Array of configuration options
 * 									'uploadDirectory' : Root directory (from webroot) where source images are stored
 * 									'cacheDirectory' : Root directory to store cached images
 */
	public function __construct(View $View, $settings = array()) {

		if ($settings == null) {

			$settings = array();

		}

		parent::__construct($View, $settings);

		$defaults = array(
			'uploadDirectory' => 'files' . DS . 'image',
			'cacheDirectory' => 'files' . DS . 'img_cache',
			'memoryLimit' => '1024M',
			'upscale' => false // Don't upscale small images
		);

		$this->settings = array_merge($defaults, $settings);

		return;

	}

/**
 * Resizes an image and caches the result.
 *
 * Cache file created in $this->settings['cacheDirectory'] so ensure this directory is writable.
 *
 * @param mixed			$image		Either a regular $html->Image path or an Image model
 *
 * 									If Image model, source file must exist in $this->settings['uploadDirector']/$image['id']/
 *
 * @param array			$options	Image tag and resizing options.
 *
 * 									'width'		Width of resized image in pixels - defaults to original image width
 *
 * 									'height'	Height of resized image in pixels - defaults to original image height
 *
 * 									'crop'		boolean
 *
 * 												if true, resized image is cropped to fit width/height exactly. crop is taken from center of image
 *
 * 												if false and width is missing, image is resized to fit height maintaining aspect ratio
 * 												if false and height is missing, image is resized to fit width maintaining aspect ratio
 * 												if false and width and height are present, image is resized to fit within width/height maintaining aspect ratio
 *
 * 									'sharpen'	boolean
 *
 * 												if true, applies a smart sharpen filter to the resized image
 *
 * 									'quality'	integer
 *
 * 												0 (worst quality, smaller file) to 100 (best quality, biggest file)
 *
 * 									'transparent'	boolean
 *
 * 												Experimental! If true, preserves the transparancy of gif/png
 * 												Doesn't work so well with gif
 *
 * @return Fully qualified img tag
 */
	public function resize($image, $options = array()) {

		$this->options = $options;

		$cacheFilename = $this->getPath($image);

		if (isset($this->options['dimensions']) && $this->options['dimensions'] === false) {
			unset($this->options['height'], $this->options['width']);
		}

		unset($this->options['dimensions'], $this->options['sharpen'], $this->options['transparent'], $this->options['crop'], $this->options['quality']);

		// Return the fully qualified img tag.
		return $this->Html->image('/' . $cacheFilename, $this->options);

	}


	public function getPath($image, $options = false) {

		if ($options !== false) {
			$this->options = $options;
		}

		$defaults = array(
			'width' => null,
			'height' => null,
			'alt' => null,
			'crop' => false,
			'quality' => 100,
			'sharpen' => false,
			'transparent' => true,
			'title' => null,
			'dimensions' => true,
			'boundingbox' => false
		);

		$this->options = array_merge($defaults, $this->options);

		$directory_additon = '';

		if (is_array($image)) {

			// Image model
			$originalFilename = $image['filename'];
			$originalFilepath = $this->settings['uploadDirectory'] . DS . $image['dir'] . DS;
			$cacheFilepath = $this->settings['cacheDirectory'] . DS . $image['dir'] . DS;

			$directory_additon = $image['dir'];

			if (empty($this->options['alt']) && isset($image['name'])) {

				$this->options['alt'] = $image['name'];
			}

		} else {

			// image path
			$parts = pathinfo($image);

			$originalFilename = $parts['basename'];
			$originalFilepath = "img" . DS . $parts['dirname'] . DS;
			$cacheFilepath = $this->settings['cacheDirectory'] . DS;

			if (empty($this->options['alt'])) {

				$this->options['alt'] = $originalFilename;
			}
		}

		if (empty($originalFilename) || !file_exists(WWW_ROOT . $originalFilepath . $originalFilename)) {
			// setup no image
			$originalFilepath = 'img' . DS;
			$originalFilename = 'no-image.png';
		}

		// Build the cache filename.
		$cacheFileOptions = $this->options;
		foreach ($cacheFileOptions as $key => $value) {
			// Make sure that we don't use alt, title, or data attributes in
			// the cache filename (these potentially break things).
			if (!in_array($key, array('width', 'height', 'crop', 'sharpen', 'transparent'))) {
				unset($cacheFileOptions[$key]);
			}
		}
		$cacheFilename = $cacheFilepath . join("_", $cacheFileOptions) . $directory_additon . $originalFilename;

		if (!file_exists(WWW_ROOT . $cacheFilename)) {

			ini_set('memory_limit', $this->settings['memoryLimit']);
			$info = getimagesize(WWW_ROOT . $originalFilepath . $originalFilename);
			list($originalWidth, $originalHeight) = $info;

			if ($this->options['boundingbox']) {
				// resizes the image to fit within the requested proportions. If the width
				// if too wide then the image is resized by width, likewise with the height

				// calculate aspect ratios
				$requiredRratio = $originalWidth / $originalHeight;
				$imageRatio = $this->options['width'] / $this->options['height'];

				if ($requiredRratio < $imageRatio) {
					// force resize by width
					$this->options['width'] = null;
				} else {
					// forece resize by height
					$this->options['height'] = null;
				}
			}

			if ($this->options['crop']) {

				// Resize source image to max of width/height then crop to fit width/height
				// maintain aspec ratio
				// crop is taken from center of image ($x, $y)
				$this->options['width'] = $this->options['width'] == null ? $originalWidth : $this->options['width'];
				$this->options['height'] = $this->options['height'] == null ? $originalHeight : $this->options['height'];
				$ratio = max($this->options['width'] / $originalWidth, $this->options['height'] / $originalHeight);
				$y = ($originalHeight - $this->options['height'] / $ratio) / 2;
				$originalHeight = $this->options['height'] / $ratio;
				$x = ($originalWidth - $this->options['width'] / $ratio) / 2;
				$originalWidth = $this->options['width'] / $ratio;

			} else {

				// resize source to fit within width/height
				// maintain aspect ratio
				if ($this->options['width'] == null) {
					// missing width - scale to height
					if ($this->options['height'] == null) {
						$this->options['height'] = $originalHeight;
					}
					$ratio = $this->options['height'] / $originalHeight;
				} elseif ($this->options['height'] == null) {
					// missing height - scale to width
					if ($this->options['width'] == null) {
						$this->options['width'] = $originalWidth;
					}
					$ratio = $this->options['width'] / $originalWidth;
				} else {
					// scale to height or width depending on which is smaller
					$ratio = min($this->options['width'] / $originalWidth, $this->options['height'] / $originalHeight);
				}

				if (!$this->settings['upscale'] && $ratio>1) {

					$cacheFilename = $originalFilepath . $originalFilename;
					$this->options['width'] = $originalWidth;
					$this->options['height'] = $originalHeight;

				} else {

					//Set class wide for the img tag can read them
					$this->options['width'] = round($originalWidth * $ratio);
					$this->options['height'] = round($originalHeight * $ratio);
					$cacheFileOptions['width'] = $this->options['width'];
					$cacheFileOptions['height'] = $this->options['height'];
					$cacheFilename = $cacheFilepath . join("_", $cacheFileOptions) . $originalFilename;

				}

				$x = 0;
				$y = 0;

			}

			if (!file_exists(WWW_ROOT . $cacheFilename)) {

				// cached version of the scaled image does not exist - create it now

				switch ($info['mime']) {
					case 'image/jpeg':
						$img = imagecreatefromjpeg(WWW_ROOT . $originalFilepath . $originalFilename);
						break;
					case 'image/gif':
						$img = imagecreatefromgif(WWW_ROOT . $originalFilepath . $originalFilename);
						break;
					case 'image/png':
						$img = imagecreatefrompng(WWW_ROOT . $originalFilepath . $originalFilename);
						break;
					default:
						return 'Unsupported image format: ' . $info['mime'];
				}

				if (!file_exists(WWW_ROOT . $cacheFilepath)) {

					// cache directory doesn't exist - create it now
					mkdir(WWW_ROOT . $cacheFilepath, 0777, true);

				}

				// create resized image
				$new = imagecreatetruecolor($this->options['width'], $this->options['height']);

				// preserve transparency
				if ($this->options['transparent'] && in_array($info['mime'], array('image/png', 'image/gif'))) {

					$transparent = imagecolorallocatealpha($new, 0, 0, 0, 127);
					imagecolortransparent($new, $transparent);
					imagealphablending($new, false);
					imagesavealpha($new, true);

				}

				// resize image
				imagecopyresampled($new, $img, 0, 0, $x, $y, $this->options['width'], $this->options['height'], $originalWidth, $originalHeight);

				if ($this->options['sharpen']) {

					// 'smart' sharpen depending on amount we have scaled the image

					// calculate sharpness factor
					$final  = sqrt($this->options['width'] * $this->options['height']) * (750.0 / sqrt($originalWidth * $originalHeight));
					$a = 52;
					$b = -0.27810650887573124;
					$c = .00047337278106508946;

					$result = $a + $b * $final + $c * $final * $final;

					$sharpnessFactor = max(round($result), 0);

					// apply sharpen filter
					imageconvolution($new, array(array(-1, -2, -1), array(-2, $sharpnessFactor + 12, -2), array(-1, -2, -1)), $sharpnessFactor, 0);

				}

				// filter?

				if (isset($this->options['filter'])) {

					imagefilter($new, $this->options['filter']);
					unset($this->options['filter']);

				}

				// Save the resized image.
				switch ($info['mime']) {

					case 'image/jpeg':

						imagejpeg($new, WWW_ROOT . $cacheFilename, $this->options['quality']);
						break;

					case 'image/gif':

						imagegif($new, WWW_ROOT . $cacheFilename);
						break;

					case 'image/png':

						$quality = intval(($this->options['quality'] / 100) * 9);

						imagepng($new, WWW_ROOT . $cacheFilename, $quality);
						break;

				}

				unset($img, $new);

			}

		}

		// Unset any options we don't want to pass to $Html->image.
		unset($this->options['crop'], $this->options['sharpen'], $this->options['transparent']);

		// Return the image path (make sure we use forward and not back slashes
		// for the image, this is an issue when using a Windows server).
		return str_replace('\\', '/', $cacheFilename);

	}

}
