<?php
App::uses('Google', 'EvGoogleServices.Lib');

class Sheets extends Google {

	public $spreadsheetId = null;

	public $service;

/**
 * Sheets constructor.
 *
 * @param string $spreadsheetId The ID of the spreadsheet
 *
 * @throws Exception
 * @return void
 */
	public function __construct($spreadsheetId = null) {
		if (empty($spreadsheetId)) {
			throw new \Exception('No spreadsheet ID entered.');
		}

		// start setting up the google api
		parent::__construct();

		$this->client->addScope(Google_Service_Sheets::SPREADSHEETS);
		$this->client->setAccessType('offline');
		$this->client->setPrompt('select_account consent');

		$this->setSpreadsheetId($spreadsheetId);

		// complete the authentication process so we can start using this service
		$this->auth();

		$this->service = new Google_Service_Sheets($this->client);
	}

/**
 * Set the spreadsheet ID for the rest of the class.
 *
 * @param string $spreadsheetId The ID of the spreadsheet
 *
 * @return void
 */
	public function setSpreadsheetId($spreadsheetId) {
		$this->spreadsheetId = $spreadsheetId;
	}

/**
 * Load the spreadsheet
 *
 * @param string|null $range The range to load, if nothing is entered, it should pull in everything.
 *
 * @return array The spreadsheet data
 */
	public function loadSheet($range = null) {
		$range = !empty($range) ? $range : 'A1:ZZ5000000';

		$response = $this->service->spreadsheets_values->get($this->spreadsheetId, $range);

		return $response->getValues();
	}

/**
 * Clear the spreadsheet
 *
 * @param string|null $range The range to load, if nothing is entered, everything will be cleared.
 *
 * @return string Returns the cleared spreadsheet range (e.g Sheet1!A1:F1000)
 */
	public function clearSheet($range = null) {
		$range = !empty($range) ? $range : 'A1:ZZ5000000';

		$updateBody = new \Google_Service_Sheets_ClearValuesRequest([
			'range' => $range,
		]);

		$response = $this->service->spreadsheets_values->clear($this->spreadsheetId, $range, $updateBody);
		return $response->getClearedRange();
	}

/**
 * Update a single cell
 *
 * @param string $cell The cell to update
 * @param string $value The new cell value
 *
 * @return bool|Google_Service_Sheets_UpdateValuesResponse
 */
	public function updateCell($cell, $value) {
		if (empty($cell) || empty($value)) {
			return false;
		}

		$updateBody = new \Google_Service_Sheets_ValueRange([
			'range' => $cell,
			'majorDimension' => 'ROWS',
			'values' => ['values' => $value],
		]);

		return $this->service->spreadsheets_values->update(
			$this->spreadsheetId,
			$cell,
			$updateBody,
			['valueInputOption' => 'USER_ENTERED']
		);
	}

/**
 * A key/data array of cells ot update
 *
 * @param array $data The data to update a sheet with.
 *
 * @return bool
 */
	public function updateCells($data) {
		if (empty($data) || !is_array($data)) {
			return false;
		}

		foreach ($data as $cell => $value) {
			$this->updateCell($cell, $value);
		}

		return true;
	}

/**
 * Append rows to the bottom of the spreadsheet.
 *
 * @param array $row The row data.
 *
 * @return bool
 */
	public function appendRows($row = []) {
		if (empty($row)) {
			return false;
		}

		$updateBody = new \Google_Service_Sheets_ValueRange([
			'range' => 'A1',
			'majorDimension' => 'ROWS',
			'values' => $row,
		]);

		$this->service->spreadsheets_values->append(
			$this->spreadsheetId,
			'A1',
			$updateBody,
			['valueInputOption' => 'USER_ENTERED']
		);

		return true;
	}
}
