<?php
App::uses('Google', 'EvGoogleServices.Lib');

class Analytics extends Google {

	public $svc;

	public $profileId;

	public function __construct() {
		// start setting up the google api
		parent::__construct();

		// set the analytics profileId
		$this->profileId = Configure::read('EvGoogleServices.Analytics.profileId');

		// set the scope path and load up the service
		$this->client->addScope("https://www.googleapis.com/auth/analytics.readonly");
		$this->svc = new Google_Service_Analytics($this->client);

		// complete the authentication process so we can start using this service
		$this->auth();
	}

	/**
	 * gets the total page visits for a specific page/path between two dates. Second date can optionally
	 * be left out to only get one date.
	 *
	 * @param  string $path      path to get the page views for
	 * @param  string $startDate start date
	 * @param  string $endDate   optional end date
	 * @param  boolean $unique   optionally set to true to only get unique pageviews
	 * @return int            returns the total page views for the selected period
	 */
	public function getPageViews($path, $startDate, $endDate = false, $unique = false) {
		if (! $endDate) {
			$endDate = $startDate;
		}

		$metrics = 'ga:users';

		if ($unique) {
			$metrics = 'ga:uniquePageviews';
		}

		$result = $this->getData($startDate, $endDate, $metrics, array(
			'filters' => 'ga:pagePath==' . $path
		));

		return $this->printResults($result);
	}

	/**
	 * gets the total views for the entire site between two dates
	 *
	 * @param  string  $startDate   start date
	 * @param  string $endDate   optional end date
	 * @param  boolean $unique   optionally set to true to only get unique pageviews
	 * @return int         returns the tital views
	 */
	public function getTotalViews($startDate, $endDate = false, $unique = false) {
		$metrics = 'ga:users';

		if ($unique) {
			$metrics = 'ga:uniquePageviews';
		}

		$result = $this->getData($startDate, $endDate, $metrics);

		return $this->printResults($result);
	}

	/**
	 * runs a get request on the reporting api. Provided as a tidier/nicer method.
	 *
	 * @param  string $startDate start date
	 * @param  string $endDate   end date
	 * @param  string $metrics   the metrics string for the query (see https://developers.google.com/analytics/devguides/reporting/core/v3/reference#metrics)
	 * @param  array $params    optional parameters (see https://developers.google.com/analytics/devguides/reporting/core/v3/reference#q_details)
	 * @return object returns the api response object
	 */
	public function getData($startDate, $endDate, $metrics, $params = array()) {
		return $this->svc->data_ga->get($this->profileId, $startDate, $endDate, $metrics, $params);
	}

	/**
	 * print results
	 *
	 * used to format and return singular value results from analytics reports
	 *
	 * @param  object &$results results object
	 * @return mixed final value returned from the response
	 */
	public function printResults(&$results) {
		// Parses the response from the Core Reporting API and prints
		// the profile name and total sessions.
		if (count($results->getRows()) > 0) {

			// Get the profile name.
			$profileName = $results->getProfileInfo()->getProfileName();

			// Get the entry for the first entry in the first row.
			$rows = $results->getRows();
			$sessions = $rows[0][0];

			// Print the results.
			return $sessions;
		}
		return false;
	}
}
