<?php

App::uses('BoostCakeFormHelper', 'BoostCake.View/Helper');
App::uses('CakeText', 'Utility');
App::uses('InflectorExt', 'EvInflector.Lib');

class FormExtHelper extends BoostCakeFormHelper {

/**
 * An array list representing a stack of inputs being rendered. Useful to tracking options through nested inputs. The
 * current input being rendered input is at the bottom of the stack.
 *
 * @var array
 */
	protected $_inputStack = [];

	protected $_inputOptions = array(
		'inputDefaults' => array(
			'empty' => '',
			'div' => 'form-group',
			'label' => array(
				'class' => 'col-xs-12 col-sm-2 col-md-3 col-lg-2 control-label'
			),
			'class' => 'form-control',
			'wrapInput' => 'col-xs-12 col-sm-10 col-md-9 col-lg-10',
		),
		'type' => 'file',
		'class' => 'form-horizontal'
	);

	public function __construct(View $View, $settings = array()) {
		$this->helpers[] = 'Time';

		$this->_inputDefaults = $this->_inputOptions['inputDefaults'];

		$this->settings['domIdSuffixSeparator'] = '_';
		if (!empty($settings['domIdSuffixSeparator'])) {
			$this->settings['domIdSuffixSeparator'] = $settings['domIdSuffixSeparator'];
		}

		parent::__construct($View, $settings);
	}

/**
 * Creates a set of radio widgets. Used to override the boostcake radio
 * handler as it does not format data to bootstrap standards. Known issue
 * but as boostcake is not actively developed it look there's no update, so
 * this is here to work around it.
 *
 * Will create a legend and fieldset by default. Use $options to control this
 *
 * You can also customize each radio input element using an array of arrays:
 *
 * ```
 * $options = array(
 *  array('name' => 'United states', 'value' => 'US', 'title' => 'My title'),
 *  array('name' => 'Germany', 'value' => 'DE', 'class' => 'de-de', 'title' => 'Another title'),
 * );
 * ```
 *
 * ### Attributes:
 *
 * - `separator` - define the string in between the radio buttons (used to generate a wrapper div)
 * - `between` - the string between legend and input set or array of strings to insert
 *    strings between each input block
 * - `legend` - control whether or not the widget set has a fieldset & legend
 * - `value` - indicate a value that is should be checked
 * - `label` - boolean to indicate whether or not labels for widgets show be displayed
 * - `hiddenField` - boolean to indicate if you want the results of radio() to include
 *    a hidden input with a value of ''. This is useful for creating radio sets that non-continuous
 * - `disabled` - Set to `true` or `disabled` to disable all the radio buttons.
 * - `empty` - Set to `true` to create an input with the value '' as the first option. When `true`
 *   the radio label will be 'empty'. Set this option to a string to control the label value.
 * - `wrapInput` - Array containing 'tag' and 'class' values or a string containing the class to apply
 *   to the div that wraps individual input-label pairs
 * - `wrapInputs` - Class to apply to the div that wraps all the inputs
 *
 * @param string $fieldName Name of a field, like this "Modelname.fieldname"
 * @param array $radioOptions Radio button options array.
 * @param array $options Array of HTML attributes, and special attributes above.
 * @return string Completed radio widget set.
 */
	public function radio($fieldName, $radioOptions = array(), $options = array()) {
		// Set some radio defaults that can be overridden using $options.
		$defaultOptions = [
			'separator' => "\n",
			'legend' => [
				'class' => 'col-md-3 control-label'
			],
			'wrapInput' => 'radio',
			'wrapInputs' => 'col-md-9'
		];
		$options = Hash::merge($defaultOptions, $options);
		$afterData = false;

		// Pull the after option if it's provided
		if (isset($options['after']) && !empty($options['after'])) {
			$afterData = $options['after'];
			unset($options['after']);
		}

		// If the legend is being set as an array so that we can control the
		// tag attributes (e.g. `class`) we need to extract this here so that
		// we can manually apply it later.
		$legend = false;
		if (isset($options['legend']) && is_array($options['legend'])) {
			$legend = $options['legend'];
			// If the legend text hasn't been set we want to generate it.
			if (empty($legend['text'])) {
				$legend['text'] = __(InflectorExt::camelToPluralize($fieldName));
			}
		} elseif (!empty($options['legend'])) {
			$legend = ['text' => $options['legend']];
		}
		$options['legend'] = false;

		//Remove any non-html attributes from the input options so that they aren't included in the input html.
		$inputOptions = $options;
		unset($inputOptions['wrapInput']);
		unset($inputOptions['wrapInputs']);

		// Run through Cake's (NOT BOOTSTCAKES) radio handler
		$out = parent::radio($fieldName, $radioOptions, $inputOptions);

		// Now manipulate it to work with Bootstrap's default layout as
		// BoostCake seeminly doesn't do that. We're going to use the
		// radio input's separator to determine
		if (!empty($options['separator'])) {
			$out = explode($options['separator'], $out);
			// reset the key values on the radioOptions array and cache result
			$tmpRadioOptions = array_values($radioOptions);

			$i = 0;
			foreach ($out as &$_out) {
				// DEVTODO: not sure why we are stripping tags here, this is
				// perhaps unnecessary and can be removed?
				$input = strip_tags($_out, '<input><img><label><br>');
				if ($input) {
					$addWrapInput = false;
					$wrapInputTag = 'div';
					$wrapInputAttr = [];

					if (!empty($options['wrapInput'])) {
						$addWrapInput = true;

						/*
						 * If the wrapInput option is an array extract the tag and pass the rest of the array as
						 * attributes.
						 */
						if (is_array($options['wrapInput'])) {
							$wrapInputAttr = $options['wrapInput'];

							if (isset($wrapInputAttr['tag'])) {
								$wrapInputTag = $wrapInputAttr['tag'];
								unset($wrapInputAttr['tag']);
							}
						} else {
							// The wrapInput option is just the class
							$wrapInputAttr['class'] = $options['wrapInput'];
						}
					}

					// Check if the wrapInput option has been overridden for this specific radio input
					if (!empty($tmpRadioOptions[$i]['wrapInput'])) {
						$addWrapInput = true;

						if (is_array($tmpRadioOptions[$i]['wrapInput'])) {
							$wrapInputAttr = $tmpRadioOptions[$i]['wrapInput'];

							if (isset($wrapInputAttr['tag'])) {
								$wrapInputTag = $wrapInputAttr['tag'];
								unset($wrapInputAttr['tag']);
							}
						} else {
							$wrapInputAttr['class'] = $tmpRadioOptions[$i]['wrapInput'];
						}
					}

					if ($addWrapInput) {
						$_out = $this->Html->tag($wrapInputTag, $input, $wrapInputAttr);
					}
				}
				$i++;
			}

			// Put it all back together
			$out = implode($options['separator'], $out);
		}

		// Wrap the radio group
		if (!empty($options['wrapInputs'])) {
			$wrapInputsTag = 'div';
			$wrapInputsAttr = [];

			if (is_array($options['wrapInputs'])) {
				$wrapInputsAttr = $options['wrapInputs'];

				if (isset($wrapInputsAttr['tag'])) {
					$wrapInputsTag = $wrapInputsAttr['tag'];
					unset($wrapInputsAttr['tag']);
				}
			} else {
				$wrapInputsAttr['class'] = $options['wrapInputs'];
			}

			$out = $this->Html->tag($wrapInputsTag, $out, $wrapInputsAttr);
		}

		// Check if we need to include the legend and fieldset if this hasn't
		// already been applied.
		if (!empty($legend)) {
			$legendText = $legend['text'];
			unset($legend['text']);
			$out = $this->Html->tag('legend', $legendText, $legend) . $out;
			$out = $this->Html->tag('fieldset', $out);
		}

		// Apply displayInfo if provided
		if ($afterData) {
			$out .= $afterData;
		}

		return $out;
	}

/**
 * Returns a select element with yes and no options.
 * @param string $fieldName This should be "Modelname.fieldname"
 * @param array $options Input options, pass `reverseOptions` true to reverse order of the options
 * @return string
 */
	public function yesOrNo($fieldName, $options = array()) {
		$yesOrNoOptions = array(1 => __d('ev_form', 'Yes'), 0 => __d('ev_form', 'No'));
		if (!empty($options['reverseOptions'])) {
			$yesOrNoOptions = array_reverse($yesOrNoOptions);
			unset($options['reverseOptions']);
		}

		if (!empty($options['type']) && $options['type'] === 'radio') {
			unset($options['type']);
			return $this->radio($fieldName, $yesOrNoOptions, $options);
		}

		$options['options'] = $yesOrNoOptions;
		return $this->input($fieldName, $options);
	}

/**
 * Returns a textarea marked up for a WYSIWYG editor
 */
	public function wysiwyg($name, $options) {
		$defaultOptions = array(
			'type' => 'textarea',
			'rows' => 10,
			'cols' => 60
		);

		$options = array_merge($defaultOptions, $options);

		return $this->input($name, $options);
	}

/**
 * Creates an Inline Edit form field
 */
	public function inline($fieldName, $options = array()) {
		$options = $this->_initInputField($fieldName, $options);
		$value = null;

		if (array_key_exists('value', $options)) {
			$value = $options['value'];
			if (!array_key_exists('escape', $options) || $options['escape'] !== false) {
				$value = h($value);
			}
			unset($options['value']);
		}

		return $this->Html->div("inline-edit", $value, array('id' => $options['name'], 'escape' => false));
	}

/**
 * Return multicheck area
 *
 * @param string $name
 * @param array $options
 * @param array $attr
 * @return string
 */
	public function multicheck($name, $attr = array()) {
		// when a dot appears within the field name, explode and assign to separate variables
		if (strpos($name, '.') !== false) {
			list($name1, $name2) = explode('.', $name, 2);
		}

		// Try to find the results so we can prepopulate. We should be using the CakePHP conventions
		// of `ModelName.ModelName` for HABTM data, but need to have a fallback for where just
		// `ModelName` has been used.
		if (! empty($name1) && ! empty($name2) && empty($this->request->data[$name1][$name2]) && $name1 == $name2) {
			$results = Hash::get($this->request->data, $name2);
		} else {
			$results = Hash::get($this->request->data, $name);
		}

		$selected = array();
		if (isset($results['0']) && is_array($results['0'])) {

			$title = (array_key_exists('title', $results['0'])) ? 'title' : 'name';
			$selected = Hash::combine($results, '{n}.id', '{n}.' . $title);

		} elseif (isset($results) && is_array($results)) {

			$selected = array();
			foreach ($results as $row) {
				$selected[$row] = $row;
			}
		}

		// assign selected fields
		if (!empty($selected)) {
			$selected = array_keys($selected);
		}

		//Label is applied separately so extract it and make sure it isn't passed with the field attributes
		$label = isset($attr['label']) ? $attr['label'] : 'Tagged ' . Inflector::humanize(Inflector::pluralize(Inflector::underscore($name)));
		unset($attr['label']);

		if (isset($attr['selectAll']) && $attr['selectAll']) {
			if (! isset($attr['before'])) {
				$attr['before'] = '';
			}

			$attr['before'] .= '<div class="checkbox-option"><a href="#" class="btn btn-sm btn-primary multi-select-all">Select All</a> <a href="#" class="btn btn-sm btn-primary multi-unselect-all">Unselect All</a></div>';
			$this->Html->scriptBlock(
				"$(document).ready(function() {
					$('.multi-select-all').bind('click', function(e) {
						e.preventDefault();

						$(this).parent().parent().find('input[type=checkbox]').prop('checked', true);
					});

					$('.multi-unselect-all').bind('click', function(e) {
						e.preventDefault();

						$(this).parent().parent().find('input[type=checkbox]').prop('checked', false);
					});
				});",
				array('inline' => false)
			);
		}

		// for multichecks, we move the attr['after'] into its own variable and pass it in separately,
		// or otherwise the HTML gets messed up due to the $this->input()
		$after = '';
		if (!empty($attr['after'])) {
			$after = $attr['after'];
			unset($attr['after']);
		}

		//Apply custom attributes over default attributes.
		$attr = array_merge(
			array(
				'type' => 'select',
				'multiple' => 'checkbox',
				'div' => 'form-group form-group--multicheck',
				'label' => false,
				'selected' => $selected,
				'options' => (! empty($attr['options'])) ? $attr['options'] : null,
				'disabled' => (! empty($attr['disabled'])) ? $attr['disabled'] : null,
				'class' => 'radio col-xs-12 col-sm-6',
				'wrapInput' => false
			),
			$attr
		);

		// override the div class to assign later when we wrap the multicheck element
		if (! empty($attr['div'])) {
			$divClass = $attr['div'];

			$attr['div'] = false;
		}

		$input = $this->input($name, $attr);

		return $this->_wrapMultcheckboxes(
			$label,
			$input,
			$after,
			$divClass
		);
	}

/**
 * Helper function for multicheck method
 * Wraps the supplied input element in the necessary HTML markup
 *
 * @param string|array $label Label for checkboxes
 * @param string $input Checkboxes
 * @param string $after Markup to be included after the checkboxes
 * @param string $divClass Div class
 * @return string
 */
	protected function _wrapMultcheckboxes($label, $input, $after = '', $divClass = '') {
		if (is_string($label)) {
			$label = [
				'text' => $label,
				'class' => 'col-xs-12 col-sm-3 col-lg-2 control-label',
			];
		}
		$markup = $this->Html->tag('label', $label['text'], ['class' => $label['class']]);
		$markup .= '<div class="col-xs-12 col-sm-9 col-lg-10">';

		$markup .= $input;
		if (!empty($after)) {
			$markup .= $after;
		}
		$markup .= '</div>';

		// wrap the markup in a div when $div is provided
		if (! empty($divClass)) {
			$markup = $this->Html->div($divClass, $markup);
		}

		return $markup;
	}

/**
 * map fields
 *
 * Used to build google maps. Because of boostcake limitations this uses a ton of custom inline
 * html. If you choose to update/improve, please make sure it follows bootstrap (not cake or inuit)
 * coding standards!
 *
 * @param  string $field   name of the field
 * @param  array  $options all field options
 * @return string          html output of the entire map row
 */
	protected function _map($field, $options = array()) {
		// Default draggable to true if not specified
		if (!isset($options['draggable'])) {
			$options['draggable'] = true;
		}

		if (! isset($options['label']) || empty($options['label'])) {
			$options['label'] = 'Location';
		}

		$after = '';
		if (! empty($options['after'])) {
			$after = $options['after'];
		}

		$divClass = 'map ';
		$wrapClass = null;
		$labelClass = 'control-label';

		if (isset($this->_inputOptions['div'])) {
			$divClass .= $this->_inputOptions['div'];
		} elseif (isset($this->_inputOptions['inputDefaults']['div'])) {
			$divClass .= $this->_inputOptions['inputDefaults']['div'];
		}

		if (isset($this->_inputOptions['wrapInput'])) {
			$wrapClass = $this->_inputOptions['wrapInput'];
		} elseif (isset($this->_inputOptions['inputDefaults']['wrapInput'])) {
			$wrapClass .= $this->_inputOptions['inputDefaults']['wrapInput'];
		}

		if (is_array($options['label'])) {
			//Label has text and class set
			$labelText = $options['label']['text'];
			$labelClass = $options['label']['class'];
		} else {
			$labelText = $options['label'];
			if (isset($this->_inputOptions['label']['class'])) {
				$labelClass = $this->_inputOptions['label']['class'];
			} elseif (isset($this->_inputOptions['inputDefaults']['label']['class'])) {
				$labelClass = $this->_inputOptions['inputDefaults']['label']['class'];
			}
		}

		$out = '<div class="' . $divClass . '" data-draggable="' . (!empty($options['draggable']) ? '1' : '0') . '">';

			$out .= $this->Html->tag('label', $labelText, array('class' => $labelClass));

			$out .= '<div class="' . $wrapClass . '">';

				$out .= $this->input('google_address', array_merge(
					$options,
					array(
						'label' => false,
						'class' => 'map-address',
						'div' => false,
						'wrapClass' => false,
						'before' => '<a class="find-postcode btn btn-default btn-sm">Find</a><div class="adminmap"></div>',
						'type' => 'text',
						'after' => false

					)
				));

				$out .= $this->input($field,
					array(
						'type' => 'hidden',
						'class' => 'map_data',
						'div' => false,
						'after' => false
					)
				);

			$out .= '</div>';

			// Add in any after bits!
			if (isset($options['after'])) {
				$out .= $options['after'];
			}

		$out .= '</div>';
		return $out;
	}

	public function addField($field, $attr) {
		$displayId = false;
		if (isset($attr['displayId'])) {
			$displayId = $attr['displayId'];
			unset($attr['displayId']);
		}

		if ($this->_isFieldName($field, 'id') && !$displayId) {

			$attr['type'] = 'hidden';

		} elseif ($this->_isFieldName($field, 'slug')) {

			$attr['type'] = 'slug';

		} elseif ($this->_isFieldName($field, 'map') || $this->_isFieldName($field, 'map_data')) {

			$attr['type'] = 'map';

		} elseif (
			$this->_isFieldName($field, 'google_address') ||
			$this->_isFieldName($field, 'longitude') ||
			$this->_isFieldName($field, 'latitude')
		) {

			return;

		}

		// If display only handle and then return before the $this->input code
		if (isset($attr['displayonly']) && $attr['displayonly'] == 'displayonly') {
			return $this->_displayOnly($field, $attr);
		}

		// assign the elements tooltip text, if required
		if (isset($attr['info'])) {
			$helpText = '<div class="help" title="' . $attr['info'] . '">&nbsp;</div>';

			if (isset($attr['label'])) {
				if (isset($attr['label']['text'])) {
					$attr['label']['text'] = $attr['label']['text'] . $helpText;
				} else {
					$attr['label'] = $attr['label'] . $helpText;
				}
			}

			$this->Html->script(array('jquery.tipsy'), array('inline' => false));
		}

		// assign the elements displayable text
		if (! empty($attr['displayInfo'])) {

			if (! isset($attr['after'])) {
				$attr['after'] = '';
			}

			$attr['after'] .= '<p class="help-block"><span>' . $attr['displayInfo'] . '</span></p>';
		}

		if (! isset($attr['class'])) {
			$attr['class'] = null;
		}

		$attr['class'] .= ' form-control';

		return $this->_addFieldSwitchOnType($field, $attr);
	}

/**
 * Depending on the type of field that is being created, return a different html string. Defaults to returning input
 * with type of "text".
 * @param string $field The name of the field to add.
 * @param array  $attr  The input options to add the field with.
 */
	protected function _addFieldSwitchOnType($field, $attr) {
		switch ($attr['type']) {

			case 'hidden':

				return $this->input($field, array_merge($attr, array('type' => 'hidden')));
				break;

			case 'boolean':
				if (empty($attr['wrapInput'])) {
					$attr['wrapInput'] = 'col-xs-12 col-md-9 col-md-offset-3 col-lg-10 col-lg-offset-2';
				}

				return $this->input($field, array_merge($attr, array('type' => 'checkbox')));
				break;

			case 'color':
				return $this->input($field, array_merge($attr, array('type' => 'color')));
				break;

			case 'date':
			case 'daterange_from':
			case 'daterange_to':

				$out = null;

				$attr = $this->value($attr, $field);
				if (empty($attr['gmt']) || $attr['gmt'] === false) {
					// Dates are stored in the database in GMT so convert the date into the
					// user's timezone (remember to convert back to GMT before saving using
					// `CakeTime::toServer($date, Configure::read('Config.timezone'));`).
					$attr['value'] = !empty($attr['value']) ? $this->Time->format($attr['value'], '%Y-%m-%d') : null;
					// Include a hidden _gmt field to check for when modifying the date before
					// saving.
					$out .= $this->hidden($field . '_gmt', array('value' => true, 'disabled' => !empty($attr['disabled'])));
				}
				unset($attr['gmt']);

				$out .= $this->input($field, array_merge($attr, array('type' => 'text', 'data-type' => 'datepicker', 'default' => '')));
				return $out;
				break;

			case 'datetime':
				$out = null;

				$attr = $this->value($attr, $field);
				if (empty($attr['gmt']) || $attr['gmt'] === false) {
					// Dates are stored in the database in GMT so convert the date into the
					// user's timezone (remember to convert back to GMT before saving using
					// `CakeTime::toServer($date, Configure::read('Config.timezone'));`).
					$attr['value'] = !empty($attr['value']) ? $this->Time->format($attr['value'], '%Y-%m-%d %H:%M:%S') : null;
					// Include a hidden _gmt field to check for when modifying the date before
					// saving.
					$out .= $this->hidden($field . '_gmt', array('value' => true));
				}
				unset($attr['gmt']);

				$this->Html->script(array('jquery-ui-timepicker-addon'), array('inline' => false));
				$out .= $this->input($field, array_merge($attr, array('type' => 'text', 'data-type' => 'datetimepicker', 'default' => '')));
				return $out;
				break;

			case 'year':
				// assign a default min year value of 25 year in the past, if omitted
				if (empty($attr['minYear'])) {
					$attr['minYear'] = date('Y') - 25;
				}

				// assign a default max year value of 1 year in the future, if omitted
				if (empty($attr['maxYear'])) {
					$attr['maxYear'] = date('Y') + 1;
				}

				$defaultAttr = array(
					'dateFormat' => 'Y',
					'default' => date('Y'),
					'div' => 'input select select--small',
					'empty' => true,
					'type' => 'date',
				);

				// if a value is set then hack the value attribute to a full date
				// string as the FormHelper::_getDateTimeValue method cannot handle
				// a 4 digit year string
				$yearValue = Hash::get($this->request->data, $field);
				if (! empty($yearValue)) {
					$defaultAttr['value'] = Hash::insert($this->request->data, $field, $yearValue . '-01-01');
				}

				return $this->input($field, array_merge($attr, $defaultAttr));
				break;
			case 'time':

				$out = null;

				$attr = $this->value($attr, $field);

				$out .= $this->input($field, array_merge($attr, array('type' => 'text', 'data-type' => 'timepicker', 'default' => '')));
				return $out;
				break;

			case 'email':

				return $this->input($field, array_merge($attr, array('type' => 'email')));
				break;

			case 'password':

				return $this->input($field, array_merge($attr, array('type' => 'password')));
				break;

			case 'integer':
			case 'float':
			case 'decimal':

				unset($attr['length']);

				$displayAsInt = false;
				if (isset($attr['displayAsInt'])) {
					$displayAsInt = $attr['displayAsInt'];
					unset($attr['displayAsInt']);
				}

				if (substr($field, -3, 3) == '_id' && !$displayAsInt) {

					return $this->input(
						$field,
						array_merge(array('empty' => 'Please select…'), $attr, array('type' => 'select'))
					);

				} else {

					return $this->input($field, array_merge($attr, array('type' => 'number', 'size' => 6)));

				}
				break;

			case 'text':
			case 'html_lite':

				return $this->wysiwyg($field, array_merge($attr, array('class' => 'wysiwyg wysiwyg--lite')));
				break;

			case 'html':
				return $this->wysiwyg($field, array_merge($attr, array('class' => 'wysiwyg wysiwyg--full')));
				break;

			case 'html_xs':
				return $this->wysiwyg($field, array_merge($attr, array('class' => 'wysiwyg wysiwyg--xs')));
				break;

			case 'multiselect':

				return $this->input($field, array_merge($attr, array('type' => 'select', 'multiple' => true)));
				break;

			case 'select':

				return $this->input($field, array_merge(array('empty' => 'Please select…'), $attr, array('type' => 'select')));
				break;

			case 'multicheck':
				//Remove default class and type so as to not override the default attributes.
				$attr['class'] = trim(str_replace('form-control', '', $attr['class']));

				if (empty($attr['class'])) {
					unset($attr['class']);
				}

				unset($attr['type']); //Defaults to `select`

				return $this->multicheck($field, $attr);
				break;

			case 'text_plain':

				if (isset($attr['class']) && $attr['class'] != '') {
					$attr['class'] .= ' text-plain';
				} else {
					$attr['class'] = 'text-plain';
				}

				$attr['type'] = 'textarea';
				return $this->input($field, $attr);
				break;

			case 'slug':

				return $this->_routeSlug($field, $attr);
				break;

			case "route-alias":

				return $this->addField($field . '.slug', array_merge(
					$attr,
					array(
						'value' => (isset($attr['value'])) ? $attr['value'] : $this->value($field),
						'data-type' => ' route-alias',
						'class' => 'form-control route-alias-input input-sm'
					)
				)) . $this->input($field, array(
					'type' => 'hidden',
					'data-type' => 'route-alias'
				));

				break;

			case 'map':
				$googleMapsApiKey = $this->_mapApiKey();

				if (!empty($googleMapsApiKey)) {
					// echo out the admin based google api key to the dom
					echo $this->Html->scriptBlock('var adminGoogleMapsApiKey = \'' . $googleMapsApiKey . '\';');
				};

				return $this->_map($field, $attr);
				break;

			case 'radio':
				$options = array();

				if (! empty($attr['options'])) {
					$options = $attr['options'];
					unset($attr['options']);
				}

				return $this->radio($field, $options, array_merge($attr, array('type' => 'radio')));
				break;

			case 'file':
				return $this->input($field, array_merge($attr, array('type' => 'file')));
				break;

			case 'chosen_select':
				return $this->_chosenSelect($field, $attr);
				break;

			case 'chosen_multiselect':
				return $this->_chosenMultiselect($field, $attr);
				break;

			case 'recaptcha':
				return $this->recaptcha($field, $attr);
				break;

			case 'invisibleRecaptcha':
				return $this->invisibleRecaptcha($field, $attr);
				break;

			case 'fontawesome-picker':
				return $this->fontawesomePicker($field, $attr);
				break;

			case 'heading':
				return $this->_heading($field, $attr);
				break;

			case 'string':
			default:
				return $this->input($field, array_merge($attr, array('type' => 'text')));

		}
	}

/**
 * Generates a form field for slug aliases (for use with Routable).
 *
 * If using slug prefixing the prefix needs handling by the JS (i.e. do not
 * remove it here).
 *
 * @param string $fieldName
 * @param array $options
 * @return string
 */
	protected function _routeSlug($fieldName, $options = array()) {
		$displayUrl = $this->_getDisplayUrl($options);

		$routeDisplay = '<span class="non-edit base-url">' . $displayUrl . '</span>';
		$routeDisplay .= '<span class="non-edit route">';
		$routeDisplay .= '<span class="slug-display">';

		if (isset($options['value'])) {
			$value = $options['value'];
			unset($options['value']);
		} else {
			$value = $this->value($fieldName);
		}

		if (!empty($options['url_prefix'])) {
			$urlPrefix = $options['url_prefix'];
			$options['url_prefix'] = str_replace('/', '\/', $options['url_prefix']);
			$value = preg_replace("/^" . $options['url_prefix'] . "/", '', $value);
			unset($options['url_prefix']);
		}
		$value = trim($value, "/");

		$routeDisplay .= $value;

		$routeDisplay .= '</span> <a href="#" class="edit-route btn btn-primary btn-sm"><i class="fa fa-pencil"></i> Edit URL</a>';
		$routeDisplay .= '</span>';
		$routeDisplay .= '<div class="input-edit-slug">';

		$stopEditing = '<a href="#" class="stop-editing btn btn-success"><i class="fa fa-check"></i> Done</a></div>';

		$options['label'] = !empty($options['label']) ? $options['label'] : 'URL';

		return $this->input($fieldName, array_merge(
				array(
					'type' => 'string',
					'between' => $routeDisplay,
					'after' => $stopEditing,
					'class' => 'slug',
					'value' => $value,
					'data-url-prefix' => (isset($urlPrefix)) ? $urlPrefix : false
				),
				$options
			)
		);
	}

/**
 * Adds fields with the markup for just html no editable. The values can be turned into links by providing a "url"
 * attribute. If data needs to be added to the url then provide a data path as an attribute of the url like:
 * [
 *     'url' => [
 *         'dataPath' => 'ModelAlias.ModelAlias.id',
 *     ],
 * ]
 * The dataPath attribute can be an array if multiple pieces of data need to be added to the url.
 *
 * @param string $field Field name
 * @param array $attr Field attributes
 * @return string HTML markup for a display only field
 */
	protected function _displayOnly($field, $attr) {
		$output = '';

		// check wether the supplied label is a string, convert to Bootstrap
		// method of an array with 'text' / 'class' keys is this is the case
		if (isset($attr['label']) && ! is_array($attr['label'])) {
			$labelText = $attr['label'];

			$attr['label'] = array(
				'class' => null,
				'text' => $labelText
			);
		}

		// assign the input default to the label
		if (empty($attr['label']['class']) && ! empty($this->_inputDefaults['label']['class'])) {
			$attr['label']['class'] = $this->_inputDefaults['label']['class'];
		}

		$label = null;
		if (isset($attr['label']) && $attr['type'] !== 'radio') {
			$label = $attr['label'];
			unset($attr['label']);
		}

		// get the model / field name
		if (// checking for just model name in field
			preg_match('/[A-Z]/', substr($field, 0, 1)) === 1 &&
			strpos($field, '.') === false // checking for just model name in field
		) {
			$fieldBits = array(
				'0' => $field
			);
			$fieldPath = $field;
		} elseif (strpos($field, '.') !== false) { // check for a model.field name entry

			$fieldBits = explode('.', $field);
			$fieldPath = $field;
		} else {
			// it's just the field name, get the defaultModel from the form helper
			$fieldBits = array(
				'0' => $this->defaultModel,
				'1' => $field
			);
			$fieldPath = $this->defaultModel . "." . $field;
		}
		// Make sure we only have the primary model and field, otherwise this method can't handle
		// fields belonging to associated models more than one level deep
		// (e.g. `User.Address.country_id`).
		$fieldBits = array_slice($fieldBits, -2, 2);

		$label = $this->_inputLabel($field, $label, $attr);

		if (// check to see if it's a select
			$attr['type'] == 'multiselect' ||
			$attr['type'] == 'multicheck' ||
			$attr['type'] == 'select' ||
			$attr['type'] == 'radio' ||
			substr($field, -3, 3) == "_id"
		) {

			// check if it's multiselect
			if ($attr['type'] === 'multiselect' || $attr['type'] === 'multicheck') {

				if (!empty($this->request->data[$fieldBits['0']])) {

					$results = array();

					// loop the options
					foreach ($this->request->data[$fieldBits['0']] as $key => $item) {
						// Stops from adding all results to the array, we only want this field
						if ($key != $fieldBits['1']) {
							continue;
						}

						// check if our option exists, add to array if so
						if (isset($item['name'])) {
							$results[] = $item['name'];
						} elseif (isset($item['title'])) {
							$results[] = $item['title'];
						} elseif (is_array($item)) {
							// If $item is a list of results without a name/title then add them to the array
							$results = array_merge($results, $item);
						}

					}

					$value = CakeText::toList($results);

				} else {
					$value = '-';
				}

			} else {

				// check if options have been passed, if not get them
				if (!isset($attr['options'])) {

					$varName = strtolower((isset($fieldBits['1'])) ? $fieldBits['1'] : $fieldBits['0']);
					$varName = Inflector::variable(
						Inflector::pluralize(preg_replace('/_id$/', '', $varName))
					);
					$varOptions = $this->_View->getVar($varName);
					if (is_array($varOptions)) {
						$attr['options'] = $varOptions;
					}
				}

				// get the id number of the option we are showing
				// $id = (!empty($this->request->data[$fieldBits['0']][$fieldBits['1']])) ? $this->request->data[$fieldBits['0']][$fieldBits['1']] : null;
				$id = Hash::get($this->request->data, $fieldPath);
				// if it doesn't exist show a dash, if it does get the options
				if (!is_null($id)) {

					// check if our option exists, display if so. dash if not
					if (!empty($attr['options'][intval($id)])) {

						$value = $attr['options'][intval($id)];

					} else {
						if (!empty($attr['display_id'])) {
							$value = $id;
						} else {
							$value = '-';
						}
					}

				} else {
					$value = '-';
				}
			}

		} elseif (empty($attr['value'])) {

			$value = Hash::get($this->request->data, $fieldPath);

			if (!is_null($value)) {

				if (in_array($attr['type'], array('text', 'text_plain'))) {
					if (empty($value)) {
						$value = '-';
					} elseif (is_array($value)) {
						$value = implode(', ', $value);
					} else {
						if (!isset($attr['escape']) || $attr['escape'] !== false) {
							$value = h($value);
						}

						$value = nl2br($value);
					}

				} elseif ($attr['type'] === 'boolean') {

					$value = Hash::get($this->request->data, $fieldPath);
					$value = ($value == '1') ? 'Yes' : 'No';

				} elseif ($attr['type'] === 'datetime') {

					$value = Hash::get($this->request->data, $fieldPath);
					$value = !empty($value) && $value != '0000-00-00 00:00:00' ? $this->Time->format($value, '%d/%m/%Y %H:%M') : '&nbsp;';

				} else {
					$value = Hash::get($this->request->data, $fieldPath);
					if (!isset($attr['escape']) || $attr['escape'] !== false) {
						$value = h($value);
					}
				}

			} else {

				$value = '-';
			}
		} else {
			$value = $attr['value'];
		}

		if (!empty($attr['url'])) {
			if (!empty($attr['url']['dataPath'])) {
				if (!is_array($attr['url']['dataPath'])) {
					$attr['url']['dataPath'] = [$attr['url']['dataPath']];
				}

				foreach ($attr['url']['dataPath'] as $dataPath) {
					$attr['url'][] = Hash::get($this->request->data, $dataPath);
				}

				unset($attr['url']['dataPath']);
			}

			$value = $this->Html->link(
				$value,
				$attr['url'],
				[
					'escape' => false,
				]
			);
		}

		$class = !empty($attr['div']) && is_string($attr['div']) ? $attr['div'] : null;
		$output .= $this->Html->tag(
			'div',
			null,
			array(
				'class' => $class . ''
			)
		);
			$output .= $this->Html->tag(
			'div',
			null,
			array(
				'class' => 'form-group'
			)
		);
			$output .= $label;

			$wrapField = ! empty($attr['wrapInput']) ? $attr['wrapInput'] : 'col-xs-12 col-sm-10 col-md-8 form-control-readonly';
			$output .= $this->Html->tag('div', $value, array('class' => $wrapField));

			$output .= $this->Html->tag('/div');
		$output .= $this->Html->tag('/div');

		return $output;
	}

/**
 * Create - helper method to add in the novalidate option as default to
 * prevent customer browser inline form validate
 */
	public function create($model = null, $options = array()) {
		// If _SimpleCsrf token is set add the hidden attribute to the form
		$append = '';
		if (!empty($this->request->params['_SimpleCsrf'])) {
			$append .= $this->hidden('_SimpleCsrf.key', array(
				'value' => $this->request->params['_SimpleCsrf']['token'], 'id' => 'Token',
				'secure' => static::SECURE_SKIP
			));
			$append = $this->Html->useTag('hiddenblock', $append);
		}
		return parent::create($model, array_merge(array('novalidate' => true, 'role' => 'form'), $options)) . $append;
	}

/**
 * Generates a display URL for a given prefix/slug along with the current host URL
 */
	protected function _getDisplayUrl($attr) {
		$displayUrl = 'http://' . $_SERVER['HTTP_HOST'] . '/';

		// Optionally override the url that shows before the path (useful for multi-domain sites).
		if (isset($attr['custom_url'])) {
			$displayUrl = $attr['custom_url'];
		}

		if (! empty($attr['url_prefix'])) {
			$displayUrl .= $attr['url_prefix'];

			if (substr($attr['url_prefix'], -1) != '/') {
				$displayUrl .= '/';
			}
		}

		return $displayUrl;
	}

	protected function _isFieldName($fieldname, $name) {
		$position = - (strlen($name) + 1);

		return substr($fieldname, $position) == ".$name" || $fieldname == $name;
	}

/**
 * Generate div options for input
 *
 * @param array $options Options list.
 * @return array
 */
	protected function _divOptions($options) {
		// We need to grab the 'type' from $options before calling the parent
		// as Cake will strip this out.
		$type = !empty($options['type']) ? $options['type'] : null;
		$options = array_merge(parent::_divOptions($options), $options);
		if (!empty($options['type']) && $options['type'] === 'hidden') {
			return array();
		}
		$div = $this->_extractOption('div', $options, true);
		if (!$div) {
			return array();
		}

		$divOptions = array('class' => 'input');

		if ($type === 'select' && isset($options['multiple']) && $options['multiple']) {
			$type .= ' multiselect';
		}

		$divOptions = $this->addClass($divOptions, $type);
		if (is_string($div)) {
			$divOptions['class'] = $div;
		} elseif (is_array($div)) {
			$divOptions = array_merge($divOptions, $div);
		}
		if ($this->_extractOption('required', $options) !== false &&
			$this->_introspectModel($this->model(), 'validates', $this->field())
		) {
			$divOptions = $this->addClass($divOptions, 'required');
		}
		if (!isset($divOptions['tag'])) {
			$divOptions['tag'] = 'div';
		}

		return $divOptions;
	}

/**
 * Instansiates select instance of the chosen jquery plugin on the requested input
 *
 * @param string $field The name of the form field
 * @param array $attr The options to pass to the form field
 * @return string Contains HTML for the chosen multiselect element
 */
	protected function _chosenSelect($field, $attr) {
		$this->Html->script(array('chosen/chosen.jquery'), array('inline' => false));
		$this->Html->css(array('../js/chosen/chosen'), array('inline' => false));

		// Check for options
		if (!empty($attr['chosen_options'])) {
			$chosenOptions = json_encode($attr['chosen_options']);
			unset($attr['chosen_options']);
		} else {
			$chosenOptions = '';
		}

		// Give this a unique ID to only apply options to this instance
		$chosenSelectId = 'chosen-' . uniqid();

		$this->Html->scriptBlock(
			"$('[data-chosen-id=\"$chosenSelectId\"]').chosen($chosenOptions);",
			array('inline' => false)
		);

		$attr['class'] = ! empty($attr['class']) ? $attr['class'] . ' chosen-select form-control' : 'chosen-select form-control';
		$attr['data-chosen-id'] = $chosenSelectId;

		return $this->input($field, array_merge($attr, array('type' => 'select')));
	}

/**
 * Instansiates multiselect instance of the chosen jquery plugin on the requested input
 *
 * @param string $field The name of the form field
 * @param array $attr The options to pass to the form field
 * @return string Contains HTML for the chosen multiselect element
 */
	protected function _chosenMultiselect($field, $attr) {
		// allow multiple options for selection
		return $this->_chosenSelect($field, array_merge($attr, array('multiple' => true)));
	}

/**
 * Allows multiple fields to be checked for errors at once and returns true if one is failing.
 * @param array|string $fieldNames Array of field names
 * @return bool True if one of the fields has an active validation error
 */
	public function isFieldsError($fieldNames) {
		$errors = false;
		foreach ((array)$fieldNames as $fieldName) {
			$errors = $errors ?: $this->isFieldError($fieldName);
		}
		return $errors;
	}

/**
 * Returns a reCAPTCHA ready form element and uses the form input defaults as a
 * basis for the reCAPTCHA element output with the aim of the field laying out
 * seamlessly on the template with further form fields
 *
 * @param string $fieldName The name of the form field to render
 * @param array $attr Contains the defaults for the element output
 * @return string Contains the reCAPTCHA element HTML
 */
	public function recaptcha($fieldName, array $attr = []) {
		$validRecaptchaTypes = [
			'V2Checkbox',
			'V3Invisible',
		];

		$recaptchaType = 'V2Checkbox';
		if (!empty($attr['recaptchaType'])) {
			$recaptchaType = $attr['recaptchaType'];
			unset($attr['recaptchaType']);
		}

		$recaptchaType = ucfirst($recaptchaType);

		$recaptchaSiteKey = $this->_getRecaptchaSiteKey($attr);

		if (!in_array($recaptchaType, $validRecaptchaTypes) || empty($recaptchaSiteKey)) {
			return '';
		}

		return $this->{'_recaptcha' . $recaptchaType}($fieldName, $attr);
	}

/**
 * Generate a v2 checkbox google recaptcha.
 *
 * @param string $fieldName  The name of the form field to render.
 * @param array  $attributes Contains the defaults for the element output.
 * @return string Contains the reCAPTCHA element HTML.
 */
	protected function _recaptchaV2Checkbox($fieldName, array $attributes = []) {
		$fieldIdentifier = 'recaptcha_';

		// Need to keep a count of the recaptchas as they all need to be unique
		$this->recaptchaCount = !empty($this->recaptchaCount) ? $this->recaptchaCount + 1 : 1;
		$fieldIdentifier .= $this->recaptchaCount;

		$attributes = array_merge_recursive($this->_inputDefaults, $attributes);

		$output = '';

		if (! empty($attributes['div'])) {
			$output .= $this->Html->tag('div', null, ['class' => $attributes['div']]);
		}

		if (! empty($attributes['label'])) {
			if (is_array($attributes['label'])) {
				$output .= $this->Html->tag(
					'label',
					! empty($attributes['label']['text']) ? $attributes['label']['text'] : '',
					['class' => ! empty($attributes['label']['class']) ? $attributes['label']['class'] : false]
				);
			} else {
				$output .= $this->Html->tag('label', $attributes['label']);
			}
		}

		if (! empty($attributes['wrapInput'])) {
			$output .= $this->Html->tag('div', null, ['class' => $attributes['wrapInput']]);
		}

		if (empty($attributes['data-sitekey'])) {
			$attributes['data-sitekey'] = Configure::read('EvCore.userReCaptcha.siteKey');
		}

		// attach the Google reCAPTCHA Javascript
		echo $this->Html->script(['https://www.google.com/recaptcha/api.js?onload=CaptchaCallback&render=explicit'], ['defer', 'async']);
		echo $this->Html->scriptBlock("
			var CaptchaCallback = function(){
				var recaptchas = document.getElementsByClassName('g-recaptcha');

				for (x = 0; x < recaptchas.length; x++) {
					grecaptcha.render(recaptchas[x], {'sitekey' : recaptchas[x].getAttribute('data-siteKey')});
				}
			};
		");

		$output .= $this->Html->tag('div', '', [
			'id' => $fieldIdentifier,
			'class' => 'g-recaptcha',
			'data-sitekey' => $attributes['data-sitekey']
		]);

		if (! empty($attributes['wrapInput'])) {
			$output .= $this->Html->tag('/div');
		}

		if (! empty($attributes['div'])) {
			$output .= $this->Html->tag('/div');
		}

		return $output;
	}

/**
 * Generate a v3 invisible Google recaptcha.
 *
 * @param string $fieldName       The name of the form field to render.
 * @param array  $fieldAttributes Contains the defaults for the element output.
 * @return string Contains the reCAPTCHA element HTML.
 */
	protected function _recaptchaV3Invisible($fieldName, array $fieldAttributes = []) {
		$siteKey = $this->_getRecaptchaSiteKey($fieldAttributes);

		return $this->_View->element(
			'EvForm.Inputs/Recaptcha/v3-invisible',
			[
				'siteKey' => $siteKey,
				'fieldName' => $fieldName,
				'fieldAttributes' => $fieldAttributes,
			]
		);
	}

/**
 * Get the google recaptcha site key. By default it takes the site key from the config setting in the EvCore plugin or
 * if the site key has been passed as a field attribute then it is taken from there instead.
 *
 * @param array $fieldAttributes Attributes for the recaptcha field.
 * @return string.
 */
	protected function _getRecaptchaSiteKey(array $fieldAttributes = []) {
		return !empty($fieldAttributes['data-sitekey'])
			? $fieldAttributes['data-sitekey']
			: Configure::read('EvCore.userReCaptcha.siteKey');
	}

/**
 * Creates a font awesome icom picker select box that outputs a font awesome icon class
 */
	public function fontawesomePicker($fieldName, $attr = []) {
		$attr = array_merge_recursive($this->_inputDefaults, $attr);
		$flattenedData = Hash::flatten($this->request->data);

		$value = null;
		if (!empty($flattenedData[$fieldName])) {
			$value = $flattenedData[$fieldName];
		} elseif (!empty($attr['value'])) {
			$value = $attr['value'];
		} elseif (!empty($attr['default'])) {
			$value = $attr['default'];
		}

		$output = '';

		$output .= $this->hidden($fieldName, array_merge_recursive( $attr, ['data-field-name' => $fieldName] ));

		if (! empty($attr['div'])) {
			$output .= $this->Html->tag('div', null, ['class' => 'bs ' . $attr['div']]);
		}

		if (! empty($attr['label'])) {
			if (is_array($attr['label'])) {
				$output .= $this->Html->tag(
					'label',
					! empty($attr['label']['text']) ? $attr['label']['text'] : 'Icon',
					['class' => ! empty($attr['label']['class']) ? $attr['label']['class'] : false]
				);
			} else {
				$output .= $this->Html->tag('label', $attr['label']);
			}
		}

		if (! empty($attr['wrapInput'])) {
			$output .= $this->Html->tag('div', null, ['class' => $attr['wrapInput']]);
		}

		$output .= $this->Html->tag('div', null, ['class' => 'btn-group']);

		$output .= $this->Html->tag('button', '<i class="fa fa-fw ' . $value . '"></i>', [
			'class' => 'icp-icon-preview iconpicker-component',
			'type' => 'button'
		]);

		$output .= $this->Html->tag('button', '<span class="caret"></span><span class="sr-only">Toggle Dropdown</span>', [
			'class' => 'icp icp-dd dropdown-toggle',
			'data-toggle' => 'dropdown',
			'data-selected' => $value,
			'type' => 'button'
		]);

		$output .= $this->Html->tag('div', '<input class="form-control iconpicker-search" data-field="' . $fieldName . '" type="text" placeholder="Type to search..."/>', [
			'class' => 'dropdown-menu',
		]);

		$output .= $this->Html->tag('/div'); // btn-group

		if (! empty($attr['wrapInput'])) {
			$output .= $this->Html->tag('/div');
		}

		if (! empty($attr['div'])) {
			$output .= $this->Html->tag('/div');
		}

		$script = '$(document).ready(function() {';
		$script .= '	$(".icp-dd").iconpicker({inputSearch: true});';
		$script .= '	$(".icp").on("iconpickerSelected", function(e) {';
		$script .= "		$('.iconpicker-search').each(function(e) {";
		$script .= "			var fieldName = $(this).data('field');";
		$script .= "			var value = $(this).val();";
		$script .= "			$(\"[data-field-name='\" + fieldName + \"']\").val(value);";
		$script .= "		});";
		$script .= '	});';
		$script .= '});';
		echo $this->Html->scriptBlock( $script, ['inline' => false, 'once' => true] );
		return $output;
	}

	/**
	 *
	 * @param string $fieldName The name of the form field to render
	 * @param array $attr Contains the defaults for the element ouput
	 * @return string
	 */
	protected function _heading($fieldName, $attr = []) {
		$output = '<div class="row">';
			$output .= '<div class="col-xs-12">';
				$output .= $this->input($fieldName, array_merge($attr, array('type' => 'hidden')));
				$output .= '<h2 class="faux-legend">' . $attr['label']['text'] . '</h2>';
			$output .= '</div>';
		$output .= '</div>';

		return $output;
	}

/**
 * Generates a form input element complete with label and wrapper div
 *
 * @param string $fieldName This should be "Modelname.fieldname"
 * @param array $options Each type of input takes different options.
 * @return string Completed form widget.
 * @link http://book.cakephp.org/2.0/en/core-libraries/helpers/form.html#creating-form-elements
 */
	public function input($fieldName, $options = []) {
		$this->_addToInputStack($fieldName, $options);

		if (empty($options['type']) || $options['type'] != 'checkbox' || empty($options['noBoost'])) {
			$input = parent::input($fieldName, $options);
			$this->_removeFromInputStack();
			return $input;
		}

		unset($options['noBoost']);

		$boostedCheckbox = parent::input($fieldName, $options);
		$unboostedCheckbox = $this->unboostCheckbox($boostedCheckbox);

		$this->_removeFromInputStack();
		return $unboostedCheckbox;
	}

/**
 * Returns a formatted SELECT element.
 *
 * @param string $fieldName Name attribute of the SELECT
 * @param array $options Array of the OPTION elements (as 'value'=>'Text' pairs) to be used in the
 *	SELECT element
 * @param array $attributes The HTML attributes of the select element.
 * @return string Formatted SELECT element
 * @link https://book.cakephp.org/2.0/en/core-libraries/helpers/form.html#options-for-select-checkbox-and-radio-inputs
 */
	public function select($fieldName, $options = [], $attributes = []) {
		if (!empty($attributes['multiple']) && $attributes['multiple'] === 'checkbox' && !empty($attributes['noBoost'])) {
			//We need to noBoost each option individually
			foreach ($options as $value => &$title) {
				if (is_array($title) && !isset($title['noBoost'])) {
					$title['noBoost'] = true;
					continue;
				}

				$title = [
					'name' => $title,
					'value' => $value,
					'noBoost' => true,
				];
			}
		}

		return parent::select($fieldName, $options, $attributes);
	}

/**
 * Returns an array of formatted OPTION/OPTGROUP elements
 *
 * @param array $elements Elements to format.
 * @param array $parents Parents for OPTGROUP.
 * @param bool $showParents Whether to show parents.
 * @param array $attributes HTML attributes.
 * @return array
 */
	protected function _selectOptions($elements = [], $parents = [], $showParents = null, $attributes = []) {
		$selectOptions = parent::_selectOptions($elements, $parents, $showParents, $attributes);

		if ($attributes['style'] === 'checkbox') {
			//The elements are passed in reverse order so get the index in reverse order.
			$optionIndex = count($selectOptions) - 1;
			foreach ($elements as $value => $title) {
				if (!is_array($title) || empty($title['noBoost']) || empty($selectOptions[$optionIndex])) {
					$optionIndex--;
					continue;
				}

				//Remove the boosting from the associated selectOptions
				$selectOptions[$optionIndex] = $this->unboostCheckbox($selectOptions[$optionIndex]);

				$optionIndex--;
			}
		}

		return $selectOptions;
	}

	public function unboostCheckbox($boostedCheckbox) {
		$unboostedCheckbox = '';

		//Check if the current input has a checkbox we can move.
		$inputRegex = '/(<input type="checkbox".*?)(.*?\/>)/';
		if (!preg_match($inputRegex, $boostedCheckbox, $checkbox)) {
			//The current input doesn't match the expected layout so return the boosted checkbox.
			return $boostedCheckbox;
		}

		//Remove the current checkbox input.
		$unboostedCheckbox = preg_replace($inputRegex, '', $boostedCheckbox);

		//Insert the new checkbox input before the label.
		$labelString = '<label';
		$startOfLabel = strpos($unboostedCheckbox, $labelString);
		$unboostedCheckbox = substr_replace($unboostedCheckbox, $checkbox, $startOfLabel, 0);

		return $unboostedCheckbox;
	}

/**
 * Add an input to the input stack when it is beginning to be rendered.
 *
 * @param string $fieldName The name of the field.
 * @param array  $options   An array of input attributes.
 * @return void.
 */
	protected function _addToInputStack($fieldName, $options) {
		$this->_inputStack[] = [
			'fieldName' => $fieldName,
			'options' => $options,
		];
	}

/**
 * Get the index of the input on the stack that is currently being rendered.
 *
 * @return array Input data, fieldName and fieldAttributes.
 */
	protected function _currentIndexOnStack() {
		end($this->_inputStack);
		$currentIndex = key($this->_inputStack);
		reset($this->_inputStack);

		return $currentIndex;
	}

/**
 * Get the input on the stack that is currently being rendered.
 *
 * @return array Input data, fieldName and fieldAttributes.
 */
	protected function _currentInputOnStack() {
		return $this->_inputStack[$this->_currentIndexOnStack()];
	}

/**
 * Remove an input to the input stack when it has finished being rendered.
 *
 * @return void.
 */
	protected function _removeFromInputStack() {
		unset($this->_inputStack[$this->_currentIndexOnStack()]);
	}

/**
 * Draw an invisible recaptcha. Use this instead of a submit
 *
 * @param string $fieldName The field name
 * @param array $options Configurable parameters
 * @return string The HTML code
 */
	public function invisibleRecaptcha($fieldName, $options = []) {
		$this->recaptchaCount = !empty($this->recaptchaCount) ? $this->recaptchaCount + 1 : 1;

		if (empty($options['data-sitekey'])) {
			$options['data-sitekey'] = Configure::read('EvCore.userReCaptcha.siteKey');
		}
		$options['data-callback'] = 'onRecaptcha' . $this->recaptchaCount . 'Submit';
		$options['class'] = empty($options['class']) ? 'g-recaptcha' : $options['class'] . ' g-recaptcha';
		$options['id'] = 'recaptcha_' . $this->recaptchaCount;

		// Add the callback function that submits the form
		$output = $this->Html->scriptBlock('var ' . $options['data-callback'] . ' = function(token) {' .
			'document.getElementById("' . $options['id'] . '").form.submit();' .
		'}');

		$output .= $this->button($fieldName, $options);
		$this->Html->script(['https://www.google.com/recaptcha/api.js'], ['inline' => false, 'once' => true]);
		return $output;
	}

/**
 * Generates a valid DOM ID suffix from a string.
 * Also avoids collisions when multiple values are coverted to the same suffix by
 * appending a numeric value.
 *
 * For pre-HTML5 IDs only characters like a-z 0-9 - _ are valid. HTML5 doesn't have that
 * limitation, but to avoid layout issues it still filters out some sensitive chars.
 *
 * OVERRIDDEN
 * The suffixes generated by this method append numerical ids directly without a separator so generated suffixes can
 * collide with real ids. Adding a separator between the suffix itself and the duplicate count.
 *
 * @param string $value The value that should be transferred into a DOM ID suffix.
 * @param string $type Doctype to use. Defaults to html4.
 * @return string DOM ID
 */
	public function domIdSuffix($value, $type = 'html4') {
		$suffixSeparator = $this->settings['domIdSuffixSeparator'];

		if ($type === 'html5') {
			$value = str_replace(array('@', '<', '>', ' ', '"', '\''), '_', $value);
		} else {
			$value = Inflector::camelize(Inflector::slug($value));
		}
		$value = Inflector::camelize($value);
		$count = 1;
		$suffix = $value;

		//Add a separator between the suffix and the field name.
		$suffix = $suffixSeparator . $suffix;

		while (in_array($suffix, $this->_domIdSuffixes)) {
			$suffix = $suffixSeparator . $value . $suffixSeparator . $count++;
		}

		$this->_domIdSuffixes[] = $suffix;
		return $suffix;
	}

/**
 * Get an google api key for a map field. If the site is on a production or UAT environment then check to see if a
 * client API is available otherwise use the default admin api key.
 *
 * @return string Api key.
 */
	protected function _mapApiKey() {
		if (!empty(Configure::read('SiteSetting.general.maps_api_key'))) {
			return Configure::read('SiteSetting.general.maps_api_key');
		} elseif (!empty(Configure::read('adminGoogleMapsApiKey'))) {
			return Configure::read('adminGoogleMapsApiKey');
		}

		return '';
	}
}
