<?php

App::uses('AppComponent', 'Controller/Component');

class SimpleCsrfComponent extends Component {

/**
 * The controller method that will be called if this request is black-hole'd
 *
 * @var string
 */
	public $blackHoleCallback = null;

/**
 * Actions to exclude from CSRF and POST validation checks.
 * Other checks like requireAuth(), requireSecure(),
 * requirePost(), requireGet() etc. will still be applied.
 *
 * @var array
 */
	public $unlockedActions = array();

/**
 * Whether to use CSRF protected forms. Set to false to disable CSRF protection on forms.
 *
 * @var bool
 * @see http://www.owasp.org/index.php/Cross-Site_Request_Forgery_(CSRF)
 * @see SecurityComponent::$csrfExpires
 */
	public $csrfCheck = true;

/**
 * The duration from when a CSRF token is created that it will expire on.
 * Each form/page request will generate a new token that can only be submitted once unless
 * it expires. Can be any value compatible with strtotime()
 *
 * @var string
 */
	public $csrfExpires = '+30 minutes';

/**
 * Controls whether or not CSRF tokens are use and burn. Set to false to not generate
 * new tokens on each request. One token will be reused until it expires. This reduces
 * the chances of users getting invalid requests because of token consumption.
 * It has the side effect of making CSRF less secure, as tokens are reusable.
 *
 * @var bool
 */
	public $csrfUseOnce = true;

/**
 * Control the number of tokens a user can keep open.
 * This is most useful with one-time use tokens. Since new tokens
 * are created on each request, having a hard limit on the number of open tokens
 * can be useful in controlling the size of the session file.
 *
 * When tokens are evicted, the oldest ones will be removed, as they are the most likely
 * to be dead/expired.
 *
 * @var int
 */
	public $csrfLimit = 100;

/**
 * Other components used by the Security component
 *
 * @var array
 */
	public $components = array('Session');

/**
 * Holds the current action of the controller
 *
 * @var string
 */
	protected $_action = null;

/**
 * Request object
 *
 * @var CakeRequest
 */
	public $request;

/**
 * Component startup. All security checking happens here.
 *
 * @param Controller $controller Instantiating controller
 * @return void
 */
	public function startup(Controller $controller) {
		$this->request = $controller->request;
		$this->_action = $controller->request->params['action'];

		$hasData = !empty($this->request->data);
		$isNotRequestAction = (
			!isset($controller->request->params['requested']) ||
			$controller->request->params['requested'] != 1
		);

		if ($this->_action === $this->blackHoleCallback) {
			return $this->blackHole($controller, 'auth');
		}

		if (!in_array($this->_action, (array)$this->unlockedActions) && $hasData && $isNotRequestAction) {
			if ($this->csrfCheck && $this->_validateCsrf($controller) === false) {
				return $this->blackHole($controller, 'csrf');
			}
		}
		$this->generateToken($controller->request);
		if ($hasData && is_array($controller->request->data)) {
			unset($controller->request->data['_Token']);
		}
	}

/**
 * Generates a basic CSRF token
 */
	public function generateToken($request) {
		// Generate a random token with timeout
		$token = Security::generateAuthKey();
		$request->params['_SimpleCsrf']['token'] = $token;
		$this->Session->write('_SimpleCsrf.' . $token, strtotime('+1 hour'));
	}

/**
 * Black-hole an invalid request with a 400 error or custom callback. If SecurityComponent::$blackHoleCallback
 * is specified, it will use this callback by executing the method indicated in $error
 *
 * @param Controller $controller Instantiating controller
 * @param string $error Error method
 * @return mixed If specified, controller blackHoleCallback's response, or no return otherwise
 * @see SecurityComponent::$blackHoleCallback
 * @link http://book.cakephp.org/2.0/en/core-libraries/components/security-component.html#handling-blackhole-callbacks
 * @throws BadRequestException
 */
	public function blackHole(Controller $controller, $error = '') {
		if (!$this->blackHoleCallback) {
			throw new BadRequestException(__d('cake_dev', 'The request has been black-holed'));
		}
		return $this->_callback($controller, $this->blackHoleCallback, array($error));
	}

	protected function _validateCsrf($controller) {
		if (!$controller->request->is('ajax') && $controller->request->is('post') || $controller->request->is('put')) {
			if (!empty($controller->request->data['_SimpleCsrf']['key'])) {
				$csrfToken = $controller->request->data['_SimpleCsrf']['key'];
				// Check the token is valid based on session
				if ($this->Session->check('_SimpleCsrf.' . $csrfToken)) {
					$tokenExpiry = $this->Session->read('_SimpleCsrf.' . $csrfToken);
					if ($tokenExpiry > time()) {
						$this->Session->delete('_SimpleCsrf.' . $csrfToken);
						return true;
					} else {
						// Token had expired
						$this->Session->delete('_SimpleCsrf.' . $csrfToken);
						return false;
					}
				} else {
					// no session token
					return false;
				}
			} else {
				// Failed verification
				// no token
				return false;
			}
			return false;
		}
		return true;
	}

/**
 * Expire CSRF nonces and remove them from the valid tokens.
 * Uses a simple timeout to expire the tokens.
 *
 * @param array $tokens An array of nonce => expires.
 * @return array An array of nonce => expires.
 */
	protected function _expireTokens($tokens) {
		$now = time();
		foreach ($tokens as $nonce => $expires) {
			if ($expires < $now) {
				unset($tokens[$nonce]);
			}
		}
		$overflow = count($tokens) - $this->csrfLimit;
		if ($overflow > 0) {
			$tokens = array_slice($tokens, $overflow + 1, null, true);
		}
		return $tokens;
	}

}
