<?php

App::uses('FormHelper', 'View/Helper');
App::uses('CakeText', 'Utility');

class FormExtHelper extends FormHelper {

	public function __construct(View $View, $settings = array()) {
		$this->helpers[] = 'Time';

		parent::__construct($View, $settings);
	}

	public function radio($fieldName, $options = array(), $attributes = array()) {
		$attributes = $this->_initInputField($fieldName, $attributes);

		$hiddenField = isset($attributes['hiddenField']) ? $attributes['hiddenField'] : true;
		$attributes['hiddenField'] = false;

		$out = parent::radio($fieldName, $options, $attributes);

		$out = str_replace('<fieldset>', "<fieldset class='radios'>", $out);
		$out = str_replace('</legend>', "</legend><div class='radios__wrapper'>", $out);
		$out = str_replace('</fieldset>', "</div></fieldset>", $out);

		$out = str_replace("<input ", "<div class='radio-wrapper'><input ", $out);
		$out = str_replace("</label>", "</label></div>", $out);

		$value = null;
		if (isset($attributes['value'])) {
			$value = $attributes['value'];
		} else {
			$value = $this->value($fieldName);
		}

		if ($hiddenField) {

			if (!isset($value) || $value === '') {

				$hidden = $this->hidden($fieldName, array(
					'id' => $attributes['id'] . '_', 'value' => '', 'name' => $attributes['name']
				));

				$out = $hidden . $out;

			}

		}

		return $out;
	}

	/**
	 * Returns a textarea marked up for a WYSIWYG editor
	 */
	public function wysiwyg($name, $options) {
		$defaultOptions = array(
			'type' => 'textarea',
			'rows' => 10,
			'cols' => 60,
			'div' => array('class' => 'input textarea wysiwyg')
		);

		$options = Hash::merge($defaultOptions, $options);

		$this->Html->script(array('tinymce/js/tinymce/tinymce.min'), array('inline' => false));

		return $this->input($name, $options);
	}

	/**
	 * Creates an Inline Edit form field
	 */
	public function inline($fieldName, $options = array()) {
		$options = $this->_initInputField($fieldName, $options);
		$value = null;

		if (array_key_exists('value', $options)) {
			$value = $options['value'];
			if (!array_key_exists('escape', $options) || $options['escape'] !== false) {
				$value = h($value);
			}
			unset($options['value']);
		}

		return $this->Html->div("inline-edit", $value, array('id' => $options['name'], 'escape' => false));
	}

	/**
	 * Return multicheck area
	 *
	 * @param string $name
	 * @param array $options
	 * @param array $attr
	 * @return string
	 */
	public function multicheck($name, $options = array(), $attr = array()) {
		list($tmp, $field) = split("\.", $name);

		$selected = array();

		if (isset($this->request['data'][$field]) && !empty($this->request['data'][$field])) {
			$results = $this->request['data'][$field];
		}

		if (isset($results[0])) {
			$title = (array_key_exists('title', $results[0])) ? 'title' : 'name';
			$selected = Hash::combine($results, '{n}.id', '{n}.' . $title);
		} elseif (isset($results[$field]) && is_array($results[$field])) {
			$selected = array();
			foreach ($results[$field] as $row) {
				$selected[$row] = $row;
			}
		}

		// assign selected fields
		if (!empty($selected)) {
			$selected = array_keys($selected);
		}

		$label = isset($attr['label']) ? $attr['label'] : 'Tagged ' . Inflector::humanize(Inflector::pluralize(Inflector::underscore($field)));

		return $this->_wrapMultcheckboxes(
			$label,
			$this->input(
				$name,
				array(
					'multiple' => 'checkbox',
					'div' => false,
					'label' => false,
					'selected' => $selected,
					'options' => $options
				)
			)
		);
	}

	/**
	 * Helper function for multicheck method
	 * Wraps the supplied input element in the necessary HTML markup
	 */
	protected function _wrapMultcheckboxes($label, $input) {
		$markup = '<div class="input multicheck">';
		$markup .= '<label>' . $label . '</label>';
		$markup .= '<div class="holder">';
		$markup .= '<div class="checkboxes-holder">';
		$markup .= '<div class="checkboxes">';
		$markup .= $input;
		$markup .= '</div>';
		$markup .= '</div>';
		$markup .= '</div>';
		$markup .= '</div><!-- .multicheck -->';

		return $markup;
	}

	/**
	 * @todo extend for multiple map fields using JSON stored field.
	 */
	protected function _map($field, $options = array()) {
		if (! isset($options['label']) || empty($options['label'])) {
			$options['label'] = 'Location';
		}

		$out = $this->input('google_address', array_merge(
			$options,
			array(
				'label' => $options['label'],
				'class' => 'map-address',
				'div' => 'input text map',
				'after' => '<button class="find-postcode">Find</button><div class="adminmap"></div>',
				'type' => 'text'
			)
		));

		$out .= $this->input($field,
			array(
				'type' => 'hidden',
				'class' => 'map_data'
			)
		);

		$this->Html->script(array('map_location'), array('inline' => false));

		return '<div class="map-wrap">' . $out . '</div>';
	}

	public function addField($field, $attr) {
		if ($this->__isFieldName($field, 'id')) {

			$attr['type'] = 'hidden';

		} elseif ($this->__isFieldName($field, 'is_protected')) {

			$attr['type'] = 'hidden';

		} elseif ($this->__isFieldName($field, 'slug')) {

			$attr['type'] = 'slug';

		} elseif ($this->__isFieldName($field, 'map') || $this->__isFieldName($field, 'map_data')) {

			$attr['type'] = 'map';

		} elseif ($this->__isFieldName($field, 'google_address') || $this->__isFieldName($field, 'longitude')) {

			return;

		}

		// If display only handle and then return before the $this->input code
		if (isset($attr['displayonly']) && $attr['displayonly'] == 'displayonly') {
			return $this->_displayOnly($field, $attr);
		}

		// assign the elements tooltip text, if required
		if (isset($attr['info'])) {
			$tooltip = array('between' => '<div class="help" title="' . $attr['info'] . '">&nbsp;</div>');
			$attr = array_merge($attr, $tooltip);

			$this->Html->script(array('jquery.tipsy'), array('inline' => false));
		}

		switch ($attr['type']) {

			case 'hidden':

				return $this->input($field, array_merge($attr, array('type' => 'hidden')));
				break;

			case 'boolean':

				return $this->input($field, array_merge($attr, array('type' => 'checkbox')));
				break;

			case 'date':

				$out = null;

				$attr = $this->value($attr, $field);
				if (empty($attr['gmt']) || $attr['gmt'] === false) {
					// Dates are stored in the database in GMT so convert the date into the
					// user's timezone (remember to convert back to GMT before saving using
					// `CakeTime::toServer($date, Configure::read('Config.timezone'));`).
					$attr['value'] = !empty($attr['value']) ? $this->Time->format($attr['value'], '%Y-%m-%d') : null;
					// Include a hidden _gmt field to check for when modifying the date before
					// saving.
					$out .= $this->hidden($field . '_gmt', array('value' => true));
				}
				unset($attr['gmt']);

				$out .= $this->input($field, array_merge($attr, array('type' => 'text', 'div' => 'input datepicker', 'default' => '')));
				return $out;
				break;

			case 'datetime':

				$out = null;

				$attr = $this->value($attr, $field);
				if (empty($attr['gmt']) || $attr['gmt'] === false) {
					// Dates are stored in the database in GMT so convert the date into the
					// user's timezone (remember to convert back to GMT before saving using
					// `CakeTime::toServer($date, Configure::read('Config.timezone'));`).
					$attr['value'] = !empty($attr['value']) ? $this->Time->format($attr['value'], '%Y-%m-%d %H:%M:%S') : null;
					// Include a hidden _gmt field to check for when modifying the date before
					// saving.
					$out .= $this->hidden($field . '_gmt', array('value' => true));
				}
				unset($attr['gmt']);

				$this->Html->script(array('jquery-ui-timepicker-addon'), array('inline' => false));
				$out .= $this->input($field, array_merge($attr, array('type' => 'text', 'div' => 'input datetimepicker', 'default' => '')));
				return $out;
				break;

			case 'email':

				return $this->input($field, array_merge($attr, array('type' => 'email')));
				break;

			case 'password':

				return $this->input($field, array_merge($attr, array('type' => 'password')));
				break;

			case 'integer':
			case 'float':
			case 'decimal':

				unset($attr['length']);

				if (substr($field, -3, 3) == '_id') {

					return $this->input(
						$field,
						array_merge(array('empty' => 'Please select…'), $attr, array('type' => 'select'))
					);

				} else {

					return $this->input($field, array_merge($attr, array('type' => 'number', 'size' => 6)));

				}
				break;

			case 'text':
			case 'html_lite':

				return $this->wysiwyg($field, array_merge(array('class' => 'wysiwyg wysiwyg--lite'), $attr));
				break;

			case 'html':

				return $this->wysiwyg($field, array_merge(array('class' => 'wysiwyg'), $attr));
				break;

			case 'multiselect':

				return $this->input($field, array_merge($attr, array('type' => 'select', 'multiple' => true)));
				break;

			case 'select':

				return $this->input($field, array_merge(array('empty' => 'Please select…'), $attr, array('type' => 'select')));
				break;

			case 'multicheck':

				return $this->multicheck($field, null, $attr);
				break;

			case 'text_plain':

				if (isset($attr['class']) && $attr['class'] != '') {
					$attr['class'] .= ' text-plain';
				} else {
					$attr['class'] = 'text-plain';
				}

				$attr['type'] = 'textarea';
				return $this->input($field, $attr);
				break;

			case 'slug':

				return $this->_routeSlug($field, $attr);
				break;

			case "route-alias":

				return $this->addField($field . '.slug', array_merge(
					$attr,
					array(
						'value' => (isset($attr['value'])) ? $attr['value'] : $this->value($field),
						'div' => 'input route-alias'
					)
				)) . $this->input($field, array(
					'type' => 'hidden',
					'class' => 'route-alias'
				));

				break;

			case 'map':

				return $this->_map($field, $attr);
				break;

			case 'string':
			default:

				return $this->input($field, array_merge($attr, array('type' => 'text')));

		}
	}

	/**
	 * Generates a form field for slug aliases (for use with Routable).
	 *
	 * If using slug prefixing the prefix needs handling by the JS (i.e. do not
	 * remove it here).
	 *
	 * @param string $fieldName
	 * @param array $options
	 * @return string
	 */
	protected function _routeSlug($fieldName, $options = array()) {
		$displayUrl = $this->_getDisplayUrl($options);

		$routeDisplay = '<span class="non-edit base-url">' . $displayUrl . '</span>';
		$routeDisplay .= '<span class="non-edit route">';
		$routeDisplay .= '<span class="slug-display">';

		if (isset($options['value'])) {
			$value = $options['value'];
			unset($options['value']);
		} else {
			$value = $this->value($fieldName);
		}

		$urlPrefix = '';
		if (!empty($options['url_prefix'])) {
			$urlPrefix = $options['url_prefix'];
			unset($options['url_prefix']);
		}
		$value = trim($value, "/");

		$routeDisplay .= $value;

		$routeDisplay .= '</span><a href="#" class="edit-route">Edit URL</a>';
		$routeDisplay .= '</span>';
		$routeDisplay .= '<div class="input-edit-slug">';

		$stopEditing = '<a href="#" class="stop-editing">Stop Editing</a></div>';

		$options['label'] = !empty($options['label']) ? $options['label'] : 'URL';

		return $this->input($fieldName, array_merge(
				array(
					'type' => 'string',
					'between' => $routeDisplay,
					'after' => $stopEditing,
					'class' => 'slug',
					'value' => $value,
					'data-url-prefix' => (isset($urlPrefix)) ? $urlPrefix : false
				),
				$options
			)
		);
	}

	/**
	 * Adds fields with the markup for just html no editable
	 */
	protected function _displayOnly($field, $attr) {
		$output = '';

		$label = null;
		if (isset($attr['label']) && $attr['type'] !== 'radio') {
			$label = $attr['label'];
			unset($attr['label']);
		}

		// get the model / field name
		if (
			preg_match('/[A-Z]/', substr($field, 0, 1)) === 1 &&
			strpos($field, '.') === false // checking for just model name in field
		) {

			$fieldBits = array(
				'0' => $field
			);
			$fieldPath = $field;
		} elseif (strpos($field, '.') !== false) { // check for a model.field name entry

			$fieldBits = explode('.', $field);
			$fieldPath = $field;
		} else {
			// it's just the field name, get the defaultModel from the form helper
			$fieldBits = array(
				'0' => $this->defaultModel,
				'1' => $field
			);
			$fieldPath = $this->defaultModel . "." . $field;
		}

		$label = $this->_inputLabel($field, $label, $attr);

		// check to see if it's a select
		if (
				$attr['type'] == 'multiselect' ||
				$attr['type'] == 'select' ||
				$attr['type'] == 'radio' ||
				substr($field, -3, 3) == "_id"
		) {

			// check if it's multiselect
			if ($attr['type'] === 'multiselect') {

				if (!empty($this->request->data[$fieldBits['0']])) {

					$results = array();

					// loop the options
					foreach ($this->request->data[$fieldBits['0']] as $item) {

						// check if our option exists, add to array if so
						if (isset($item['name'])) {
							$results[] = $item['name'];
						} elseif (isset($item['title'])) {
							$results[] = $item['title'];
						}
					}

					$value = CakeText::toList($results);

				} else {
					$value = '-';
				}

			} else {

				// check if options have been passed, if not get them
				if (!isset($attr['options'])) {

					$varName = strtolower((isset($fieldBits['1'])) ? $fieldBits['1'] : $fieldBits['0']);
					$varName = Inflector::variable(
						Inflector::pluralize(preg_replace('/_id$/', '', $varName))
					);
					$varOptions = $this->_View->getVar($varName);
					if (is_array($varOptions)) {
						$attr['options'] = $varOptions;
					}
				}

				// get the id number of the option we are showing
				// $id = (!empty($this->request->data[$fieldBits['0']][$fieldBits['1']])) ? $this->request->data[$fieldBits['0']][$fieldBits['1']] : null;
				$id = Hash::get($this->request->data, $fieldPath);
				// if it doesn't exist show a dash, if it does get the options
				if (!is_null($id)) {

					// check if our option exists, display if so. dash if not
					if (!empty($attr['options'][intval($id)])) {

						$value = $attr['options'][intval($id)];

					} else {
						$value = '-';
					}

				} else {
					$value = '-';
				}
			}

		} else {

			$value = Hash::get($this->request->data, $fieldPath);

			if (!is_null($value)) {

				if (in_array($attr['type'], array('text', 'text_plain'))) {

					$value = nl2br(h(Hash::get($this->request->data, $fieldPath)));

				} elseif ($attr['type'] === 'boolean') {

					$value = Hash::get($this->request->data, $fieldPath);
					$value = ($value == '1') ? 'Yes' : 'No';

				} elseif ($attr['type'] === 'datetime') {

					$value = Hash::get($this->request->data, $fieldPath);
					$value = !empty($value) && $value != '0000-00-00 00:00:00' ? $this->Time->format($value, '%d/%m/%Y %H:%M') : '&nbsp;';

				} else {

					$value = h(Hash::get($this->request->data, $fieldPath));
				}

			} else {

				$value = '-';
			}
		}

		$class = !empty($attr['div']) && is_string($attr['div']) ? $attr['div'] : 'field';
		$output .= $this->Html->tag(
			'div',
			null,
			array(
				'class' => $class
			)
		);
		$output .= $this->Html->tag('div', $label, array('class' => 'field__label'));
		$output .= $this->Html->tag('div', $value, array('class' => 'field__value'));
		$output .= $this->Html->tag('/div');

		return $output;
	}

	/**
	 * Create - helper method to add in the novalidate option as default to
	 * prevent customer browser inline form validate
	 */
	public function create($model = null, $options = array()) {
		return parent::create($model, array_merge($options, array('novalidate' => true)));
	}

	/**
	 * Generates a display URL for a given slug along with the current host URL
	 */
	protected function _getDisplayUrl($attr) {
		$displayUrl = 'http://' . $_SERVER['HTTP_HOST'] . '/';

		return $displayUrl;
	}

	private function __isFieldName($fieldname, $name) {
		$position = - (strlen($name) + 1);

		return substr($fieldname, $position) == ".$name" || $fieldname == $name;
	}

}
