<?php

App::uses('EvFormBuilderAppModel', 'EvFormBuilder.Model');

class Question extends EvFormBuilderAppModel {

	public $belongsTo = [
		'QuestionType' => [
			'className' => 'EvFormBuilder.QuestionType',
		],
	];

	public $hasMany = [
		'QuestionResponse' => [
			'className' => 'EvFormBuilder.QuestionResponse',
		],
	];

	public $hasAndBelongsToMany = [
		'QuestionBlock' => [
			'className' => 'EvFormBuilder.QuestionBlock',
			'foreignKey' => 'question_id',
			'associationForeignKey' => 'question_block_id',
			'with' => 'EvFormBuilder.QuestionBlocksQuestion',
			'order' => 'QuestionBlocksQuestion.sequence',
			'unique' => 'keepExisting',
		],
	];

	public $validate = [
		'name' => [
			'required' => [
				'rule' => ['notBlank'],
				'message' => 'The question name must not be blank.',
			],
		],
		'question_type_id' => [
			'required' => [
				'rule' => ['notBlank'],
				'message' => 'A question must have a question type.',
			],
		],
		'answers' => [
			'answersRequired' => [
				'rule' => ['answersRequired'],
				'message' => 'Answers are required for this question type'
			],
		],
	];

/**
 * Custom validation method that checks whether the question being saved has a question type that can have answers and
 * if it does then at least one answer must be provided. If the question type has default options then answers are not
 * required.
 *
 * @param array $check The fields to check.
 * @return bool True if answers are required and have been provided. False otherwise.
 */
	public function answersRequired($check) {
		if (empty($this->data[$this->alias]['question_type_id'])) {
			return true;
		}

		$questionType = $this->QuestionType->findById($this->data[$this->alias]['question_type_id']);

		if (!empty($questionType['QuestionType']['can_have_answers']) && empty($questionType['QuestionType']['default_options'])) {
			if (!empty($this->data[$this->alias]['answers'])) {
				return true;
			}
		}

		return false;
	}

/**
 * Get the list of questions for a drop down menu.
 *
 * @param array $query Array of query params.
 * @return array List of questions.
 */
	public function getForDropDown($query = array()) {
		$virtualFields = $this->virtualFields;

		$query = $this->addDropDownQueryParams($query);

		$dropdownResults = parent::getForDropDown($query);

		$this->virtualFields = $virtualFields;

		return $dropdownResults;
	}

/**
 * Query used to retrieve a record ready for edit.
 *
 * @param int   $id     ID of row to edit.
 * @param array $params The db query array.
 * @return array The found item.
 */
	public function readForEdit($id, $params = array()) {
		$params['contain'][] = 'QuestionType';

		return parent::readForEdit($id, $params);
	}
}
