<?php

App::uses('EvFormBuilderAppModel', 'EvFormBuilder.Model');

class FormResponse extends EvFormBuilderAppModel {

	public $belongsTo = [
		'Form' => [
			'className' => 'EvFormBuilder.Form',
		],
		'User' => [
			'className' => 'EvCore.User',
		]
	];

	public $hasMany = [
		'QuestionBlockResponse' => [
			'className' => 'EvFormBuilder.QuestionBlockResponse',
		],
		'FormResponseAssociation' => [
			'className' => 'EvFormBuilder.FormResponseAssociation',
		],
	];

/**
 * Query used to retrieve a record ready for edit
 *
 * @param int   $id ID of row to edit
 * @param array $params The db query array - can be used to pass in additional parameters such as contain
 * @return array
 */
	public function readForEdit($id, $params = array()) {
		$params['contain'][] = 'Form';
		$params['contain'][] = 'User';

		$params['contain'][] = 'FormResponseAssociation';

		$params['contain']['QuestionBlockResponse'] = [
			'QuestionBlock',
			'QuestionResponse' => [
				'Question' => [
					'QuestionType',
				],
			]
		];

		return parent::readForEdit($id, $params);
	}

/**
 * Add the provided form associations to the responses.
 *
 * @param array $responses    The responses that have been submitted.
 * @param array $associations The associations in [model => model_id] format.
 * @return array The responses with associations added.
 */
	public function addFormResponseAssociations($responses, $associations) {
		if (empty($associations)) {
			return $responses;
		}

		//Check we can associate and get className
		foreach ($associations as $model => $modelId) {
			//If coming from the request query then they need to be camelised.
			$modelAlias = InflectorExt::camelize($model);

			if (empty(Configure::read('EvFormBuilder.associations.' . $modelAlias))) {
				unset($associations[$model]);
				continue;
			}

			$association = Configure::read('EvFormBuilder.associations.' . $modelAlias);

			foreach ($responses as $responseIndex => $response) {
				$responses[$responseIndex]['FormResponseAssociation'][] = [
					'model' => $association['className'],
					'model_id' => $modelId,
				];
			}
		}

		return $responses;
	}

/**
 * Format a form response so that it can be saved.
 *
 * @param array $response The responses that have been submitted.
 * @return array The formatted responses.
 */
	public function formatResponseData($response) {
		return $response;
	}

/**
 * Check if a user can respond to a form. Check if the user has already responded, if they are authorised to respond to
 * the form or if the form is currently available to respond to, etc.
 *
 * @param int   $id           The id of the form to check.
 * @param int   $userId       The id of the user to check.
 * @param array $associations Array of form response associations to check for, in the structure [{modelClass} => {modelId}].
 * @return bool True if the form can be responded to by the user, false otherwise.
 */
	public function canRespondToForm($id, $userId, $associations) {
		if (Configure::check('EvFormBuilder.responseLimit') && Configure::read('EvFormBuilder.responseLimit') != -1) {
			if ($this->totalUserFormReponses($id, $userId, $associations) >= Configure::read('EvFormBuilder.responseLimit')) {
				return false;
			}
		}

		return true;
	}

/**
 * Get the total count of responses for a form by a user. If associations have been provided then only count responses
 * with those associations.
 *
 * @param int   $id           The id of the form to count responses to.
 * @param int   $userId       The id of the user to get responses for.
 * @param array $associations Array of form response associations to check for, in the structure [{modelClass} => {modelId}].
 * @return int|bool The total number of repsonses found, false otherwise.
 */
	public function totalUserFormReponses($id, $userId, $associations) {
		$query = [
			'conditions' => [
				$this->alias . '.form_id' => $id,
				$this->alias . '.user_id' => $userId,
			]
		];

		if (!empty($associations)) {
			foreach ($associations as $model => $modelId) {
				$modelAlias = 'FormResponseAssociation' . InflectorExt::slug($model);

				$query['joins'][] = [
					'table' => $this->tablePrefix . 'form_response_associations',
					'alias' => $modelAlias,
					'conditions' => [
						$modelAlias . '.form_response_id = ' . $this->alias . '.id',
						$modelAlias . '.model' => $model,
						$modelAlias . '.model_id' => $modelId,
					],
				];
			}
		}

		return $this->find('count', $query);
	}
}
