<?php

App::uses('EvFormBuilderAppModel', 'EvFormBuilder.Model');

class Form extends EvFormBuilderAppModel {

	public $hasMany = [
		'FormResponse' => [
			'className' => 'EvFormBuilder.FormResponse',
		]
	];

	public $hasAndBelongsToMany = [
		'QuestionBlock' => [
			'className' => 'EvFormBuilder.QuestionBlock',
			'foreignKey' => 'form_id',
			'associationForeignKey' => 'question_block_id',
			'with' => 'EvFormBuilder.FormsQuestionBlock',
			'order' => 'FormsQuestionBlock.sequence',
			'unique' => 'keepExisting',
		],
	];

	public $validate = [
		'name' => [
			'required' => [
				'rule' => ['notBlank'],
				'message' => 'The form name must not be blank.',
			]
		]
	];

/**
 * Get the list of forms for a drop down menu.
 *
 * @param array $query Array of query params.
 * @return array List of forms.
 */
	public function getForDropDown($query = array()) {
		$virtualFields = $this->virtualFields;

		$query = $this->addDropDownQueryParams($query);

		$dropdownResults = parent::getForDropDown($query);

		$this->virtualFields = $virtualFields;

		return $dropdownResults;
	}

/**
 * Query used to retrieve a record ready for edit.
 *
 * @param int   $id     ID of row to edit.
 * @param array $params The db query array.
 * @return array The found item.
 */
	public function readForEdit($id, $params = array()) {
		$params['contain'][] = 'QuestionBlock';

		return parent::readForEdit($id, $params);
	}

/**
 * Query used to retrieve a form ready for submission. It is and extension of readForView().
 *
 * @param int   $id     Id of the form to submit.
 * @param array $params The db query array.
 * @return array The found item.
 */
	public function readForSubmit($id, $params = []) {
		//If the current form parameters are not containing the question blocks and the questions then add them as
		//default contains.
		if (
			!isset($params['contain']['QuestionBlock']['Question']['QuestionType']) ||
			!in_array('QuestionType', $params['contain']['QuestionBlock']['Question'])
		) {
			$params['contain']['QuestionBlock']['Question']['QuestionType'] = [];
		}

		//Contain the images for the question blocks.
		if (!empty(Configure::read('EvFormBuilder.imageSlots.QuestionBlock'))) {
			$questionBlockImageSlots = Configure::read('EvFormBuilder.imageSlots.QuestionBlock');
			foreach ($questionBlockImageSlots as $slotName => $slot) {
				$slotName = InflectorExt::camelize($slotName);

				$params['contain']['QuestionBlock'][$slotName . 'Image'] = [];
			}
		}

		//Contain the images for the questions.
		if (!empty(Configure::read('EvFormBuilder.imageSlots.Question'))) {
			$questionBlockImageSlots = Configure::read('EvFormBuilder.imageSlots.Question');
			foreach ($questionBlockImageSlots as $slotName => $slot) {
				$slotName = InflectorExt::camelize($slotName);

				$params['contain']['QuestionBlock']['Question'][$slotName . 'Image'] = [];
			}
		}

		$params['readForSubmit'] = true;

		return $this->readForView($id, $params);
	}

/**
 * Retrieve a record for submission or throw a not found exception.
 *
 * @param int   $id     ID of row to edit.
 * @param array $params The db query array.
 * @return array The found item.
 *
 * @throws NotFoundException Thrown when data can't be found.
 */
	public function readForSubmitOrFail($id, $params = []) {
		$data = $this->readForSubmit($id, $params);
		if (empty($data)) {
			throw new NotFoundException(InflectorExt::camelToHumanize($this->name) . ' not found');
		}

		return $data;
	}
}
