<?php

App::uses('AppBehavior', 'Model/Behavior');

class FormsBehavior extends AppBehavior {

	public $settings = [];

/**
 * Initialise the behavior - store the settings against the model
 *
 * @param Model $Model    The model the behavior is attached to.
 * @param array $settings The behavior settings.
 * @return void.
 * @see ModelBehavior::setup()
 */
	public function setup(Model $Model, $settings = []) {
		parent::setup($Model, $settings);

		$this->settings[$Model->alias] = $settings;

		$Model->bindModel(
			[
				'hasAndBelongsToMany' => array(
					'Form' => array(
						'className' => 'EvFormBuilder.Form',
						'foreignKey' => 'model_id',
						'associationForeignKey' => 'form_id',
						'with' => 'EvFormBuilder.FormAssociation',
						'joinTable' => 'ev_form_builder_form_associations',
						'conditions' => array(
							'FormAssociation.model' => EvClassRegistry::getNameFromModel($Model)
						),
						'type' => 'LEFT',
					)
				)
			],
			false
		);
	}

/**
 * beforeBeforeSave callback, in before saveAll is called, allowing you to modify the whole data array before
 * save processes.
 *
 * @param Model &$Model The model being saved.
 * @param array $data The data that is going to be saved.
 * @param array $options The save options being used.
 * @return array|bool Bool false to stop save, return modified data array to proceed
 */
	public function beforeBeforeSave(&$Model, $data, $options = array()) {
		$modelName = EvClassRegistry::getNameFromModel($Model);

		$formModelAlias = $Model->Form->alias;

		//Only want to modify the data if it is part of a list. E.g from a chosen multiselect.
		if (!empty($data[$formModelAlias][$formModelAlias])) {
			foreach ($data[$formModelAlias][$formModelAlias] as $formIndex => $formId) {
				if (!is_array($formId)) {
					$data[$formModelAlias][$formModelAlias][$formIndex] = [
						'form_id' => $formId,
						'model' => $modelName,
					];
				}
			}
		}

		return $data;
	}

/**
 * Called before each find operation. Return false if you want to halt the find
 * call, otherwise return the (modified) query data.
 *
 * @param Model $Model The model being searched for.
 * @param array $query Data used to execute this query, i.e. conditions, order, etc.
 * @return mixed true if the operation should continue, false if it should abort; or, modified
 *  $query to continue with new $query
 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#beforefind
 */
	public function beforeFind(Model $Model, $query) {
		if (!empty($query['readForEdit']) && !empty($this->settings[$Model->alias]['formInject'])) {
			$query['contain'][] = 'Form';
		}

		$parentQuery = parent::beforeFind($Model, $query);

		if ($parentQuery === true) {
			return $query;
		} elseif ($parentQuery) {
			return Hash::merge($parentQuery, $query);
		}

		return false;
	}
}
