<?php

App::uses('EvFormBuilderAppController', 'EvFormBuilder.Controller');

class QuestionBlocksController extends EvFormBuilderAppController {

/**
 * Constructor.
 *
 * @param CakeRequest $request Request object for this controller. Can be null for testing,
 *  but expect that features that use the request parameters will not work.
 * @param CakeResponse $response Response object for this controller.
 */
	public function __construct($request = null, $response = null) {
		$this->adminActions[] = 'admin_reorder_questions';

		parent::__construct($request, $response);
	}

/**
 * Pagination settings for admin_index.
 *
 * @return array Pagination settings.
 */
	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};

		$paginate = parent::_adminIndexPaginate();

		$paginate['joins'][] = [
			'type' => 'left',
			'table' => 'ev_form_builder_question_blocks_questions',
			'alias' => 'QuestionCount',
			'conditions' => [
				'QuestionCount.question_block_id = ' . $Model->alias . '.id',
			]
		];

		$paginate['group'] = $Model->alias . '.id';

		$Model->virtualFields['question_count'] = 'COUNT(' . $Model->alias . '.id' . ')';

		return $paginate;
	}

/**
 * Defines which actions are available to each row of the admin_index table results.
 *
 * @return array The index actions.
 */
	protected function _adminIndexActions() {
		$actions = parent::_adminIndexActions();

		$preActions = [
			'Reorder' => [
				'cell' => [
					'class' => 'reorder',
				],
				'link' => [
					'url' => [
						'action' => 'reorder_questions',
					],
					'text' => '<i class="s s--listing"></i>',
					'options' => [
						'escape' => false,
					],
				],
			],
		];

		$actions = $preActions + $actions;

		return $actions;
	}

/**
 * Defines the fields displayed in a filter form. Defaults to id, display name, is_active, created and modified fields.
 *
 * @return array The fields to use in the admin filter.
 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = parent::_adminFilterFields();

		$preActiveFilterFields = [
			$modelAlias . '.internal_name' => [
				'label' => 'Internal Name',
				'type' => 'string',
				'compare' => [
					$modelAlias . '.internal_name LIKE' => '%%%s%%',
				],
			],
		];

		$fields = ArrayUtil::addBefore($fields, $modelAlias . '.is_active', $preActiveFilterFields);

		return $fields;
	}

/**
 * Used to populate form drop down selects
 *
 * @return void.
 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		$this->set('questions', $Model->Question->getForDropDown());
	}

/**
 * Defines the columns displayed in the admin_index results table
 * using a columns whitelist.
 *
 * @return array admin_index columns.
 */
	protected function _adminIndexColumns() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = parent::_adminIndexColumns();

		$preActiveColumns = [
			$modelAlias . '.internal_name' => [
				'type' => 'string',
				'label' => 'Internal Name',
			],
			$modelAlias . '.question_count' => [
				'type' => 'integer',
				'label' => 'Number of Questions',
			],
		];

		$columns = ArrayUtil::addBefore($columns, $modelAlias . '.is_active', $preActiveColumns);

		return $columns;
	}

/**
 * Defines the fields displayed in an admin_form for this model. Defaults to all fields in the db table.
 *
 * @return array The fields for the question admin form.
 */
	protected function _adminFormFields() {
		$modelAlias = $this->{$this->modelClass}->alias;

		$fields = parent::_adminFormFields();

		$fields = ArrayUtil::addAfter($fields, $modelAlias . '.introduction', [
			$modelAlias . '.Question.Question' => [
				'label' => 'Questions',
				'data-placeholder' => 'Select Questions',
				'type' => 'chosen_multiselect'
			],
		]);

		return $fields;
	}

/**
 * Defines the buttons in the toolbar displayed on an admin_form.
 *
 * @param int $id The id of the question block being edited.
 * @return array Toolbar actions.
 */
	protected function _adminFormToolbar($id = null) {
		$toolbar = parent::_adminFormToolbar($id);

		$preToolbar = [
			'Reorder Questions' => [
				'url' => [
					'action' => 'reorder_questions',
					$id,
				],
				'class' => 'toolbar__btn--listing',
			],
		];

		$toolbar = array_merge($preToolbar, $toolbar);

		return $toolbar;
	}

/**
 * Reorder the questions in a question block.
 *
 * @param int $id The id of the question block to reorder.
 * @return void.
 * @throws NotFoundException If the question block can't be found.
 */
	public function admin_reorder_questions($id) {
		$Model = $this->{$this->modelClass};

		$questionBlock = $Model->find('first', [
			'contain' => [
				'Question'
			],
			'conditions' => [
				$Model->alias . '.id' => $id
			]
		]);

		if (empty($questionBlock)) {
			throw new NotFoundException('Category not found');
		}

		if ($this->request->is('post') || $this->request->is('put')) {
			if ($Model->QuestionBlocksQuestion->saveMany($this->request->data['reorder'])) {
				$this->Flash->success(
					[
						'title' => 'Order Saved',
						'description' => 'Reordering has been saved successfully!',
					]
				);
			} else {
				$this->Flash->fail(
					[
						'title' => 'Reordering Failed',
						'description' => 'There was an error saving the data',
					]
				);
			}

			return $this->redirect(
				[
					'action' => 'edit',
					$questionBlock['QuestionBlock']['id']
				]
			);
		}

		$this->set(compact('questionBlock'));
		$this->set('orderKey', 'sequence');
		$this->set('maxDepth', 1);
		$this->set('toolbar', $this->_adminReorderToolbar());
	}
}
