<?php

App::uses('EvFormBuilderAppController', 'EvFormBuilder.Controller');

class FormsController extends EvFormBuilderAppController {

	public $components = [
		'EvFormBuilder.Forms',
	];

	public $helpers = [
		'EvFormBuilder.FormBuilder',
	];

/**
 * Constructor.
 *
 * @param CakeRequest $request Request object for this controller. Can be null for testing,
 *  but expect that features that use the request parameters will not work.
 * @param CakeResponse $response Response object for this controller.
 */
	public function __construct($request = null, $response = null) {
		$this->adminActions[] = 'admin_reorder_question_blocks';

		parent::__construct($request, $response);
	}

/**
 * Defines which actions are available to each row of the admin_index table results.
 *
 * @return array The index actions.
 */
	protected function _adminIndexActions() {
		$actions = parent::_adminIndexActions();

		$preActions = [
			'Reorder' => [
				'cell' => [
					'class' => 'reorder',
				],
				'link' => [
					'url' => [
						'action' => 'reorder_question_blocks',
					],
					'text' => '<i class="s s--listing"></i>',
					'options' => [
						'escape' => false,
					],
				],
			],
		];

		$actions = $preActions + $actions;

		return $actions;
	}

/**
 * Defines the fields displayed in a filter form. Defaults to id, display name, is_active, created and modified fields.
 *
 * @return array The fields to use in the admin filter.
 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = parent::_adminFilterFields();

		$preActiveFilterFields = [
			$modelAlias . '.internal_name' => [
				'label' => 'Internal Name',
				'type' => 'string',
				'compare' => [
					$modelAlias . '.internal_name LIKE' => '%%%s%%',
				],
			],
		];

		$fields = ArrayUtil::addBefore($fields, $modelAlias . '.is_active', $preActiveFilterFields);

		return $fields;
	}

/**
 * Used to populate form drop down selects
 *
 * @return void.
 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		$this->set('questionBlocks', $Model->QuestionBlock->getForDropDown());
	}

/**
 * Defines the columns displayed in the admin_index results table
 * using a columns whitelist.
 *
 * @return array admin_index columns.
 */
	protected function _adminIndexColumns() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = parent::_adminIndexColumns();

		$preActiveColumns = [
			$modelAlias . '.internal_name' => [
				'type' => 'string',
				'label' => 'Internal Name',
			],
		];

		$columns = ArrayUtil::addBefore($columns, $modelAlias . '.is_active', $preActiveColumns);

		return $columns;
	}

/**
 * Defines the fields displayed in an admin_form for this model. Defaults to all fields in the db table.
 *
 * @return array The fields for the question admin form.
 */
	protected function _adminFormFields() {
		$modelAlias = $this->{$this->modelClass}->alias;

		$fields = parent::_adminFormFields();

		$fields = ArrayUtil::addAfter($fields, $modelAlias . '.body', [
			$modelAlias . '.QuestionBlock.QuestionBlock' => [
				'label' => 'Question Blocks',
				'data-placeholder' => 'Select Question Blocks',
				'type' => 'chosen_multiselect'
			],
		]);

		return $fields;
	}

/**
 * Defines the buttons in the toolbar displayed on an admin_form.
 *
 * @param int $id The id of the question block being edited.
 * @return array Toolbar actions.
 */
	protected function _adminFormToolbar($id = null) {
		$toolbar = parent::_adminFormToolbar($id);

		$preToolbar = [
			'Reorder Question Blocks' => [
				'url' => [
					'action' => 'reorder_question_blocks',
					$id,
				],
				'class' => 'toolbar__btn--listing',
			],
		];

		$toolbar = array_merge($preToolbar, $toolbar);

		return $toolbar;
	}

/**
 * Reorder the question blocks on a form.
 *
 * @param int $id The id of the form to reorder.
 * @return void.
 * @throws NotFoundException If the form can't be found.
 */
	public function admin_reorder_question_blocks($id) {
		$Model = $this->{$this->modelClass};

		$form = $Model->find('first', [
			'contain' => [
				'QuestionBlock'
			],
			'conditions' => [
				$Model->alias . '.id' => $id
			]
		]);

		if (empty($form)) {
			throw new NotFoundException('Category not found');
		}

		if ($this->request->is('post') || $this->request->is('put')) {
			if ($Model->FormsQuestionBlock->saveMany($this->request->data['reorder'])) {
				$this->Flash->success(
					[
						'title' => 'Order Saved',
						'description' => 'Reordering has been saved successfully!',
					]
				);
			} else {
				$this->Flash->fail(
					[
						'title' => 'Reordering Failed',
						'description' => 'There was an error saving the data',
					]
				);
			}

			return $this->redirect(
				[
					'action' => 'edit',
					$form['Form']['id']
				]
			);
		}

		$this->set(compact('form'));
		$this->set('orderKey', 'sequence');
		$this->set('maxDepth', 1);
		$this->set('toolbar', $this->_adminReorderToolbar());
	}

/**
 * Submit a form and record the submission.
 *
 * @param int $id The id of the form to submit.
 * @return void.
 */
	public function submit($id) {
		$Model = $this->{$this->modelClass};

		$form = $Model->readForSubmitOrFail($id);

		$requestAssociations = [];

		/*
		 * Loop through the query parameters. They should be provided as a model => modelId
		 * array. For each one load the model and use readForView to get the data.
		 * If model data is found then assign it to the view.
		 */
		if (!empty($this->request->query)) {
			foreach ($this->request->query as $modelAlias => $modelId) {
				$modelAlias = InflectorExt::camelize($modelAlias);
				if (!empty(Configure::read('EvFormBuilder.associations.' . $modelAlias . '.className'))) {
					$modelClass = Configure::read('EvFormBuilder.associations.' . $modelAlias . '.className');
					$requestAssociations[$modelClass] = $modelId;

					$this->loadModel($modelClass);

					$this->Forms->loadFormAssociationData($modelAlias, $modelId);
				}
			}
		}

		$canRespond = $Model->FormResponse->canRespondToForm(
			$id,
			$this->Forms->getUserIdForResponse(),
			$requestAssociations
		);
		if (!$canRespond) {
			$this->Forms->responseSuccess($id, $this->request->query);
		}

		$this->set('form', $form);

		if ($this->request->is('post') || $this->request->is('put')) {
			if (!empty($this->request->data['FormResponse'])) {
				$formResponses = $this->request->data['FormResponse'];

				if (!empty($this->request->query)) {
					$formResponses = $this->Forms->addFormResponseAssociations($formResponses, $this->request->query);
				}

				$formResponses = $this->Forms->formatFormResponseData($formResponses);

				$saved = $Model->FormResponse->saveMany(
					$formResponses,
					[
						'deep' => true,
					]
				);

				if ($saved) {
					$this->Forms->responseSuccess($id, $this->request->query);
				} else {
					$this->Forms->responseError($id, $this->request->query);
				}
			}
		}
	}

/**
 * Success page for when a form is submitted successfully.
 *
 * @return void.
 */
	public function submit_success() {
		$pageData = $this->assignPage('Form Response Success');

		if (!empty($pageData)) {
			$template = $this->Tpl->getTemplate($pageData, 'Page');
			if (!empty($template)) {
				$this->view = $template;
			}
		}
	}
}
