<?php

App::uses('AppComponent', 'Controller/Component');

class FormsComponent extends AppComponent {

/**
 * A list of associations for the current request. In the format `modelClass => modelId`.
 *
 * @var array
 */
	public $requestAssociations = [];

/**
 * Run the form injection. For custom fields this add type / validation arrays to the template for fields
 *
 * @param array    $data  Data from $this->request->data.
 * @param obj      $Model Model Object for the primary model we are dealing with.
 * @param int|null $id    The ID if on edit or null of add.
 * @return void.
 */
	public function injectAdminForm($data, $Model, $id) {
		$Model = $this->_controller->{$this->_controller->modelClass};

		if (isset($this->_controller->viewVars['fields'])) {
			$formModelAlias = $Model->Form->alias;

			$chosenOptions = [];
			if (empty(Configure::read('EvFormBuilder.associationsHaveMultipleForms'))) {
				$chosenOptions['max_selected_options'] = 1;
			}

			$formFields = [
				$formModelAlias . '.' . $formModelAlias => [
					'type' => 'chosen_multiselect',
					'label' => 'Form',
					'data-placeholder' => 'Select forms',
					'chosen_options' => $chosenOptions,
				]
			];

			$fields = $this->_controller->viewVars['fields'];

			$addBefore = 'is_active';
			if (!$Model->hasField('is_active')) {
				if ($Model->hasField('created')) {
					$addBefore = 'created';
				} else {
					$addBefore = null;
				}
			}

			if (!empty($addBefore)) {
				$fields = ArrayUtil::addBefore($fields, $Model->alias . '.is_active', $formFields);
			} else {
				$fields = Hash::merge($fields, $formFields);
			}

			$this->_controller->viewVars['fields'] = $fields;
			$this->_controller->set('forms', $Model->Form->getForDropDown());
		}
	}

/**
 * Add the provided form associations to the responses.
 *
 * @param array $responses    The responses that have been submitted.
 * @param array $associations The associations in [model => model_id] format.
 * @return array The responses with associations added.
 */
	public function addFormResponseAssociations($responses, $associations) {
		if (empty($this->_controller->Form)) {
			$this->_controller->loadModel('EvFormBuilder.Form');
		}

		return $this->_controller->Form->FormResponse->addFormResponseAssociations($responses, $associations);
	}

/**
 * Format the form responses so that they can be saved. Include the user who has submitted the response. The expected
 * format is an array of multiple readForEdit results.
 *
 * @param array $responses The responses that have been submitted.
 * @return array The formatted responses.
 */
	public function formatFormResponseData($responses) {
		if (empty($this->_controller->Form)) {
			$this->_controller->loadModel('EvFormBuilder.Form');
		}

		foreach ($responses as &$response) {
			$response['user_id'] = $this->_getUserIdForResponse();

			$response = $this->_controller->Form->FormResponse->formatResponseData($response);
		}

		return $responses;
	}

/**
 * Format the form responses so that they can be displayed or used to populate a frontend form.
 *
 * @param array $responses The responses that have been submitted.
 * @return array The formatted responses.
 */
	public function formatFormResponsesDisplay($responses) {
		$formattedResponses = [];

		foreach ($responses as $response) {
			$formId = $response['FormResponse']['form_id'];

			$formattedResponses['FormResponse'][$formId] = $this->_formatFormResponseDisplay($response);
		}

		return $formattedResponses;
	}

/**
 * Format a individual response so that it can be displayed.
 *
 * @param array $response A single response.
 * @return array Formatted response.
 */
	protected function _formatFormResponseDisplay($response) {
		$formattedReponse = [];

		$formattedReponse['form_id'] = $response['FormResponse']['form_id'];

		if (!empty($response['QuestionBlockResponse'])) {
			foreach ($response['QuestionBlockResponse'] as $questionBlockResponse) {
				$questionBlockId = $questionBlockResponse['question_block_id'];

				$formattedReponse['QuestionBlockResponse'][$questionBlockId] = $this->_formatQuestionBlockResponseDisplay(
					$questionBlockResponse
				);
			}
		}

		return $formattedReponse;
	}

/**
 * Format an individual question block response so that it can be displayed.
 *
 * @param array $questionBlockResponse A single question block response.
 * @return array Formatted question block response.
 */
	protected function _formatQuestionBlockResponseDisplay($questionBlockResponse) {
		$formattedQuestionBlockResponse = [];

		$formattedQuestionBlockResponse['question_block_id'] = $questionBlockResponse['question_block_id'];

		if (!empty($questionBlockResponse['QuestionResponse'])) {
			foreach ($questionBlockResponse['QuestionResponse'] as $questionResponse) {
				$questionId = $questionResponse['question_id'];

				$formattedQuestionBlockResponse['QuestionResponse'][$questionId] = $this->_formatQuestionResponseDisplay(
					$questionResponse
				);
			}
		}

		return $formattedQuestionBlockResponse;
	}

/**
 * Format an individual question response so that it can be displayed.
 *
 * @param array $questionResponse A single question response.
 * @return array Formatted question response.
 */
	protected function _formatQuestionResponseDisplay($questionResponse) {
		$formattedQuestionResponse = [];

		$formattedQuestionResponse['question_id'] = $questionResponse['question_id'];
		$formattedQuestionResponse['response'] = $questionResponse['response'];

		return $formattedQuestionResponse;
	}

/**
 * Get the id of the current user so that they can be associated with the response they have made.
 *
 * @return int The id of the user submitting the form response.
 */
	protected function _getUserIdForResponse() {
		return $this->_controller->Auth->user('User.id');
	}

/**
 * Same as above but publicly accessible
 * Only override the above function if extending this class
 *
 * @return int The user ID
 */
	public function getUserIdForResponse() {
		return $this->_getUserIdForResponse();
	}

/**
 * A user has responded to a form and it has saved successfully, redirect them to a success page and provide
 * a flash success message.
 *
 * @param int   $formId       The id of the form submitted.
 * @param array $requestQuery The query data from the request.
 * @return void.
 */
	public function responseSuccess($formId, $requestQuery) {
		$this->_controller->Flash->success('Your response has been saved');

		$redirect = [
			'action' => 'submit_success',
		];

		if (!empty(Configure::read('EvFormBuilder.redirects.responseSuccess'))) {
			$redirect = Configure::read('EvFormBuilder.redirects.responseSuccess');
		}

		$this->_controller->redirect($redirect);
	}

/**
 * A user has responded to a form and it has failed to save, redirect them to an error page and provide
 * a flash fail message.
 *
 * @param int   $formId       The id of the form submitted.
 * @param array $requestQuery The query data from the request.
 * @return void.
 */
	public function responseError($formId, $requestQuery) {
		$this->_controller->Flash->fail(
			[
				'title' => 'Your response failed to save.',
				'list' => $this->_controller->Form->FormResponse->validationErrors,
			]
		);

		if (!empty(Configure::read('EvFormBuilder.redirects.responseError'))) {
			$this->_controller->redirect(
				Configure::read('EvFormBuilder.redirects.responseError')
			);
		}
	}

/**
 * Check the query data to see if any associations have been set and if they have load the model and the associated
 * data.
 *
 * @param bool $refresh True to force the requestAssociations to be rebuilt, default false.
 * @return array A list of associations, in the format `modelClass => modelId`.
 */
	public function loadAssocationsFromRequestQuery($refresh = false) {
		//If we aren't refreshing and requestAssociations have already been built.
		if ($refresh && !empty($this->requestAssociations)) {
			return $this->requestAssociations;
		}

		//If request query data has not been set.
		if (empty($this->_controller->request->query)) {
			return $this->requestAssociations;
		}

		foreach ($this->_controller->request->query as $modelAlias => $modelId) {
			$modelAlias = InflectorExt::camelize($modelAlias);

			if (!empty(Configure::read('EvFormBuilder.associations.' . $modelAlias . '.className'))) {
				$modelClass = Configure::read('EvFormBuilder.associations.' . $modelAlias . '.className');
				$this->requestAssociations[$modelClass] = $modelId;

				$this->_controller->loadModel($modelClass);

				$this->loadFormAssociationData($modelAlias, $modelId);
			}
		}

		return $this->requestAssociations;
	}

/**
 * Load the association data from the request using the readForView method on each model. Format the data and
 * set it to the current view. The model should have already been loaded on the model.
 *
 * @param string $modelAlias The alias of the current model to load.
 * @param int    $modelId    The id of the model to get data of.
 * @return array The data that was loaded.
 */
	public function loadFormAssociationData($modelAlias, $modelId) {
		//If the model hasn't previously loaded then load it now. Requires config association to have been setup.
		if (empty($this->_controller->{$modelAlias})) {
			if (empty(Configure::read('EvFormBuilder.associations.' . $modelAlias . '.className'))) {
				return [];
			}
			$modelClass = Configure::read('EvFormBuilder.associations.' . $modelAlias . '.className');
			$this->_controller->loadModel($modelClass);
		}

		$data = $this->_controller->{$modelAlias}->readForView(
			$modelId,
			$this->_formAssociationDataQuery(
				$this->_controller->{$modelAlias},
				$modelId
			)
		);

		if (empty($data)) {
			return [];
		}

		$this->_formatFormAssociationData($modelAlias, $modelId, $data);

		$this->_controller->set(InflectorExt::snake($modelAlias), $data);

		return $data;
	}

/**
 * Modify the query that gets form association data. Checks in the config to see if any contains have been
 * provided as a basis of query.
 *
 * @param Model $Model   The current model data is being loaded from.
 * @param int   $modelId The id of the records being loaded.
 * @return array Query array passed into readForView().
 */
	protected function _formAssociationDataQuery($Model, $modelId) {
		$query = [];

		if (Configure::check('EvFormBuilder.associations.' . $Model->alias . '.contain')) {
			$query['contain'] = Configure::read('EvFormBuilder.associations.' . $Model->alias . '.contain');
		}

		return $query;
	}

/**
 * Format association data before it is set to the view. Currently this method is for extension only.
 *
 * @param string $modelAlias The alias of the model of the current model that data is being formatted.
 * @param int    $modelId    The id of the model data being formatted.
 * @param array  $data       The data being formatted.
 * @return array The formatted data.
 */
	protected function _formatFormAssociationData($modelAlias, $modelId, $data) {
		return $data;
	}
}
