<?php

App::uses('AppHelper', 'View/Helper');

App::uses('InflectorExt', 'EvInflector.Lib');

class FilterFormsHelper extends AppHelper {

/**
 * Default configuration settings for a filter form. These are set up to provide a functioning form with
 * basic responsive elements. It is not in anyway pretty but it is functional.
 *
 * @var array.
 */
	public $filterFormSettings = [
		'filterWrapper' => [
			'tag' => 'div',
			'class' => 'menu',
		],
		'filterHeading' => [
			'link' => '#menu-collapse-content-',
			'class' => 'menu__heading',
			'data-toggle' => 'collapse',
		],
		'filterHeader' => [
			'tag' => 'div',
			'class' => 'menu__header',
		],
		'contentWrapper' => [
			'tag' => 'div',
			'aria-expanded' => false,
			'class' => 'menu__content collapse',
			'id' => 'menu-collapse-content-',
		],
		'form' => [
			'inputDefaults' => [
				'div' => 'form-group',
				'label' => [
					'class' => 'control-label'
				],
				'wrapInput' => '',
				'class' => 'form-control'
			],
			'id' => 'Filter',
			'class' => ''
		],
		'sectionWrapper' => [
			'tag' => 'div',
			'class' => 'menu__section menu__section--filter-',
		],
		'sectionHeader' => [
			'tag' => 'div',
			'class' => 'menu__sub-heading collapsed',
			'data-toggle' => 'collapse',
			'data-target' => '#menu-collapse-content-sub-',
		],
		'tooltip' => [
			'tag' => 'i',
			'content' => '',
			'class' => 'fa fa-info-circle',
			'data-toggle' => 'tooltip',
			'data-target' => 'right',
		],
		'sectionContentInput' => [
			'type' => 'checkbox',
			'class' => 'js-auto-submit',
			'label' => false,
			'div' => false,
			'checkboxDiv' => false,
		],
		'sectionContentLabel' => [
			'tag' => 'label',
		],
		'sectionContentInputWrapper' => [
			'tag' => 'div',
			'class' => 'checkbox',
		],
		'sectionContentWrapper' => [
			'tag' => 'div',
			'class' => 'menu__section__content collapse',
			'id' => 'menu-collapse-content-sub-',
			'aria-expanded' => false,
		],
		'groupSectionContentInput' => [
			'class' => 'required js-auto-submit',
		],
		'groupSectionContentWrapper' => [
			'tag' => 'div',
			'class' => 'truefilters hidden',
		],
		'groupSectionContentGroupInput' => [
			'type' => 'checkbox',
			'checkboxDiv' => false,
			'class' => 'group-filter-check',
			'value' => '',
		],
		'groupSectionContentGroupLabel' => [
			'tag' => 'label',
		],
		'groupSectionContentGroupWrapper' => [
			'tag' => 'div',
			'class' => 'checkbox',
		],
		'clearLink' => [
			'position' => 'bottom',
			'name' => 'Clear All Filters',
			'class' => 'link--clear',
			'escape' => false,
		],
		'clearLinkWrapper' => [
			'tag' => 'div',
			'class' => 'menu__section menu__section--clear-filters',
		]
	];

/**
 * Default Constructor.
 *
 * @param View $View The View this helper is being attached to.
 * @param array $settings Configuration settings for the helper.
 * @return void.
 */
	public function __construct(View $View, $settings = array()) {
		$this->helpers[] = 'Number';

		$this->helpers['Form'] = array(
			'className' => 'EvForm.FormExt'
		);

		$this->helpers['Html'] = array(
			'className' => 'HtmlExt'
		);

		if (!empty($settings['attr'])) {
			$this->filterFormSettings = Hash::merge($this->filterFormSettings, $settings['attr']);
		}

		parent::__construct($View, $settings);
	}

/**
 * Checks the supplied values and returns a boolean value based on whether the filter field should be marked
 * as checked / selected
 *
 * @param string $productOptionGroupName The name of the option group to check for in the request.
 * @param string $productOptionName The option name to check for in the request.
 * @return bool
 */
	protected function _isFilterSelected(
		$productOptionGroupName = null,
		$productOptionName = null
	) {
		$selected = false;

		//Check if this group has any selected filters
		if (in_array($productOptionGroupName, $this->request->params['named'])) {
			//change name to url safe name
			$urlSafeProductOptionName = InflectorExt::slugUrlSafePreserveChars($productOptionGroupName . '-' . $productOptionName, true);
			$selected = array_key_exists(
				$urlSafeProductOptionName,
				$this->request->params['named']
			);
		}

		return $selected;
	}

/**
 * An extension to @_isFilterSelected that checks that an entire filter group has been selected.
 * To be used when filtering with filter groups. Will only return true when every option within
 * the group is selected.
 *
 * @param string $productOptionGroupName The name of the option group to check for in the request.
 * @param string $filterGroup            The filtergroup to check each option for.
 * @return bool
 */
	protected function _isGroupFilterSelected(
		$productOptionGroupName = null,
		$filterGroup = null
	) {
		$selected = false;

		foreach ($filterGroup['Option'] as $productOption) {
			if (
				$this->_isFilterSelected(
					$productOptionGroupName,
					$productOption['name']
				)
			) {
				$selected[] = true;
			}
		}

		//Check that every option in the group has been selected.
		if (!empty($selected) && count($selected) == count($filterGroup['Option'])) {
			$selected = true;
		}

		return $selected;
	}

/**
 * Build a filter form that includes the options split into section by their option groups. Can be used with
 * filter groups. Without changing any of the configurations, it will output html that provides a basic
 * collapse structure with each option being a checkbox. Configurations can be set: when initialising the
 * helper, when called from a template, for each option group, for each filtergroup and for each option.
 *
 * @param array  $options    An array of available options.
 * @param string $filterName The name of the filter. Useful for multiple filter forms.
 * @param array  $attr       The configuration of the filter form.
 * @return string Filter form html string.
 */
	public function buildFilterForm($options, $filterName = 'filter', $attr = []) {
		$filterHtml = '';

		$attr = Hash::merge($this->filterFormSettings, $attr);

		$filterHtml .= $this->_buildFilterWrapper($filterName, $attr);

		$filterHtml .= $this->_buildFilterHeader($filterName, $attr);

		$filterHtml .= $this->_buildFilterContentWrapper($filterName, $attr);

		$attr['form']['id'] .= $filterName;
		$filterHtml .= $this->Form->create('EvFilter' . $filterName, $attr['form']);

		$filterHtml .= $this->_buildFilterContent($filterName, $attr, $options);

		$filterHtml .= $this->Form->end();

		$filterHtml .= $this->_buildFilterContentWrapper($filterName, $attr, true);

		$filterHtml .= $this->_buildFilterWrapper($filterName, $attr, true);

		return $filterHtml;
	}

/**
 * Build the html string for the wrapper that contains the entire filter.
 *
 * @param string $filterName  The name of the filter.
 * @param array  $attr        The configuration of the filter form.
 * @param bool   $close       True if we are closing the wrapper tag. Defaults to false.
 * @param string $wrapperHtml Used for extending to prepend wrapper html.
 * @return string Filter wrapper html string.
 */
	protected function _buildFilterWrapper($filterName, $attr, $close = false, $wrapperHtml = '') {
		$tag = $attr['filterWrapper']['tag'];
		unset($attr['filterWrapper']['tag']);

		if ($close) {
			$wrapperHtml .= $this->Html->tag('/' . $tag);
			return $wrapperHtml;
		}

		$wrapperHtml .= $this->Html->tag($tag, null, $attr['filterWrapper']);
		return $wrapperHtml;
	}

/**
 * Build the html string for the filter header. The header is separate from the filter sections. The header
 * isn't added if a filterHeading title isn't supplied. By default this would hide make the filter hidden
 * as the heading is used to collapse the sections of the filter form.
 *
 * @param string $filterName The name of the filter.
 * @param array  $attr       The configuration of the filter form.
 * @param string $headerHtml Used for extending to prepend header html.
 * @return string Filter header html string.
 */
	protected function _buildFilterHeader($filterName, $attr, $headerHtml = '') {
		$tag = $attr['filterHeader']['tag'];
		unset($attr['filterHeader']['tag']);

		if (!empty($attr['filterHeading']['title'])) {
			$headerHtml .= $this->Html->tag(
				$tag,
				$this->_buildFilterHeading($filterName, $attr),
				$attr['filterHeader']
			);
		}

		return $headerHtml;
	}

/**
 * Build the html string for the filter heading. This is the content inside the filter header. Will always
 * provide the filterHeading title as a link. A clear link can be added by providing a clearLink url and
 * setting the clearLink position to top or both.
 *
 * @param string $filterName  The name of the filter.
 * @param array  $attr        The configuration of the filter form.
 * @param string $headingHtml Used for extending to prepend heading html.
 * @return string Filter heading html string.
 */
	protected function _buildFilterHeading($filterName, $attr, $headingHtml = '') {
		$title = $attr['filterHeading']['title'];
		$link = $attr['filterHeading']['link'] . $filterName;

		unset($attr['filterHeading']['title']);
		unset($attr['filterHeading']['link']);

		$headingHtml .= $this->Html->link(
			$title,
			$link,
			$attr['filterHeading']
		);

		if (!empty($attr['clearLink']['url'])) {
			if ($attr['clearLink']['position'] == 'top' || $attr['clearLink']['position'] == 'both') {
				$headingHtml .= $this->_buildFilterClearLink($filterName, $attr, true);
			}
		}

		return $headingHtml;
	}

/**
 * Build the html string for the wrapper that contains all the filter sections and the form itself.
 *
 * @param string $filterName         The name of the filter.
 * @param array  $attr               The configuration of the filter form.
 * @param bool   $close              True if we are closing the wrapper tag. Defaults to false.
 * @param string $contentWrapperHtml Used for extending to prepend content wrapper html.
 * @return string Filter content wrapper html string.
 */
	protected function _buildFilterContentWrapper($filterName, $attr, $close = false, $contentWrapperHtml = '') {
		if (empty($attr['contentWrapper'])) {
			return $contentWrapperHtml;
		}

		$tag = $attr['contentWrapper']['tag'];
		unset($attr['contentWrapper']['tag']);

		if ($close) {
			$contentWrapperHtml .= $this->Html->tag('/' . $tag);
			return $contentWrapperHtml;
		}

		$attr['contentWrapper']['id'] .= $filterName;
		$contentWrapperHtml .= $this->Html->tag($tag, null, $attr['contentWrapper']);

		return $contentWrapperHtml;
	}

/**
 * Build the html string for the content of the filter form. The content contains a section for each option
 * group. A clear link can be added by providing a clearLink url and setting the clearLink position to bottom
 * or both. If configuration is available for each option group then it will be merged with the current
 * filter form configuration.
 *
 * @param string $filterName  The name of the filter.
 * @param array  $attr        The configuration of the filter form.
 * @param array  $options     An array of available options.
 * @param string $contentHtml Used for extending to prepend content html.
 * @return string Filter content html string.
 */
	protected function _buildFilterContent($filterName, $attr, $options, $contentHtml = '') {
		if (! empty($options)) {
			foreach ($options as $optionGroup) {
				if (!empty($optionGroup['Option'])) {
					$contentHtml .= $this->_buildFilterContentInnerHtml($filterName, $attr, $optionGroup);
				}
			}

			if (!empty($attr['clearLink']['url'])) {
				if ($attr['clearLink']['position'] == 'bottom' || $attr['clearLink']['position'] == 'both') {
					$contentHtml .= $this->_buildFilterClearLink($filterName, $attr);
				}
			}
		}
		return $contentHtml;
	}

/**
 * Build the header/options segment of the filter HTML for a given option
 *
 * @param string $filterName  The name of the filter.
 * @param array  $attr        The configuration of the filter form.
 * @param array  $optionGroup The option group being used.
 * @return string Filter content html string.
 */
	protected function _buildFilterContentInnerHtml($filterName, $attr, $optionGroup) {
		//Merge any optionGroup specific attributes into the attributes array.
		if (!empty($optionGroup['attr'])) {
			$attr = Hash::merge($attr, $optionGroup['attr']);
		}

		// Group container open
		$contentHtml = $this->_buildFilterSectionWrapper($filterName, $attr, $optionGroup);

		$contentHtml .= $this->_buildFilterSectionHeader($filterName, $attr, $optionGroup);

		if (isset($optionGroup['FilterGroup'])) {
			$contentHtml .= $this->_buildFilterSectionGroupContent(
				$filterName,
				$attr,
				$optionGroup
			);
		} else {
			$contentHtml .= $this->_buildFilterSectionContent($filterName, $attr, $optionGroup);
		}

		// Group container close
		$contentHtml .= $this->_buildFilterSectionWrapper($filterName, $attr, $optionGroup, true);

		return $contentHtml;
	}

/**
 * Build the html string for the section wrapper. This a wrapper that contains a section.
 *
 * @param string $filterName         The name of the filter.
 * @param array  $attr               The configuration of the filter form.
 * @param array  $optionGroup        The option group of the current section.
 * @param bool   $close              True if we are closing the wrapper tag. Defaults to false.
 * @param string $sectionWrapperHtml Used for extending to section wrapper html.
 * @return string Filter section wrapper html string.
 */
	protected function _buildFilterSectionWrapper($filterName, $attr, $optionGroup, $close = false, $sectionWrapperHtml = '') {
		$tag = $attr['sectionWrapper']['tag'];
		unset($attr['sectionWrapper']['tag']);

		if ($close) {
			$sectionWrapperHtml .= $this->Html->tag('/' . $tag);
			return $sectionWrapperHtml;
		}

		$attr['sectionWrapper']['class'] .= InflectorExt::slug(strtolower(h($optionGroup['name'])));

		$sectionWrapperHtml .= $this->Html->tag($tag, null, $attr['sectionWrapper']);
		return $sectionWrapperHtml;
	}

/**
 * Build the html string for the section header. The header is separate from the section content itself.
 *
 * @param string $filterName        The name of the filter.
 * @param array  $attr              The configuration of the filter form.
 * @param array  $optionGroup       The option group of the current section.
 * @param string $sectionHeaderHtml Used for extending to section header html.
 * @return string Filter section header html string.
 */
	protected function _buildFilterSectionHeader($filterName, $attr, $optionGroup, $sectionHeaderHtml = '') {
		if (empty($attr['sectionHeader'])) {
			return $sectionHeaderHtml;
		}

		$tag = $attr['sectionHeader']['tag'];
		unset($attr['sectionHeader']['tag']);

		$attr['sectionHeader']['data-target'] .= $filterName . '-' . InflectorExt::slug(strtolower(h($optionGroup['name'])));

		$sectionHeading = $this->_buildFilterSectionHeading($filterName, $attr, $optionGroup);

		$sectionHeaderHtml .= $this->Html->tag($tag, $sectionHeading, $attr['sectionHeader']);

		return $sectionHeaderHtml;
	}

/**
 * Build the html string for the section heading. This is the actual content of the section header. It
 * contains the optiong group name and, if provided, a tooltip.
 *
 * @param string $filterName         The name of the filter.
 * @param array  $attr               The configuration of the filter form.
 * @param array  $optionGroup        The option group of the current section.
 * @param string $sectionHeadingHtml Used for extending to section heading html.
 * @return string Filter section heading html string.
 */
	protected function _buildFilterSectionHeading($filterName, $attr, $optionGroup, $sectionHeadingHtml = '') {
		$sectionHeadingText = $optionGroup['name'];
		if (!empty($optionGroup['label'])) {
			$sectionHeadingText = $optionGroup['label'];
		}

		$escape = true;
		if (isset($attr['sectionHeading']['escape']) && $attr['sectionHeading']['escape'] === false) {
			$escape = false;
		}

		if (isset($optionGroup['escape']) && $optionGroup['escape'] === false) {
			$escape = false;
		}

		if ($escape) {
			$sectionHeadingText = h($sectionHeadingText);
		}

		$sectionHeadingHtml .= $sectionHeadingText;

		$sectionHeadingHtml .= $this->_buildFilterSectionTooltip($filterName, $attr, $optionGroup);

		return $sectionHeadingHtml;
	}

/**
 * Build the html string for the section tooltip. If a tooltip isn't provided then it will return the passed
 * html string.
 *
 * @param string $filterName         The name of the filter.
 * @param array  $attr               The configuration of the filter form.
 * @param array  $optionGroup        The option group of the current section.
 * @param string $sectionTooltipHtml Used for extending to section tooltip html.
 * @return string Filter section tooltip html string.
 */
	protected function _buildFilterSectionTooltip($filterName, $attr, $optionGroup, $sectionTooltipHtml = '') {
		if (!empty($optionGroup['tooltip'])) {
			$attr['tooltip']['title'] = $optionGroup['tooltip'];

			$tag = $attr['tooltip']['tag'];
			unset($attr['tooltip']['tag']);

			$sectionTooltipHtml .= ' ' . $this->Html->tag($tag, $attr['tooltip']['content'], $attr['tooltip']);
		}

		return $sectionTooltipHtml;
	}

/**
 * Build the html string for the section content. The content contains a wrapper for each option group, an
 * input wrapper for each option, an input for each option and a label for each option. If an option
 * provides a separate configuration then it will merged with the current filter form configuration.
 *
 * @param string $filterName       The name of the filter.
 * @param array  $attr             The configuration of the filter form.
 * @param array  $optionGroup      The option group of the current section.
 * @param string $filterFieldsHtml Used for extending to filter fields html.
 * @return string Filter fields html string.
 */
	protected function _buildFilterSectionContent($filterName, $attr, $optionGroup, $filterFieldsHtml = '') {
		$filterFieldsHtml .= $this->_buildSectionContentWrapper($filterName, $attr, $optionGroup);

		foreach ($optionGroup['Option'] as $option) {
			if (!empty($option['attr'])) {
				$attr = Hash::merge($attr, $option['attr']);
			}

			$optionSafeName = InflectorExt::slugUrlSafePreserveChars(
				$option['name'],
				true
			);

			$optionGroupSafeName = InflectorExt::slugUrlSafePreserveChars(
				$optionGroup['name'],
				true
			);

			$attr['sectionContentInput']['id'] = $optionGroup['name'] . $optionSafeName . $filterName;
			$attr['sectionContentInput']['value'] = $optionGroupSafeName . '-' . $optionSafeName;

			$filterFieldsHtml .= $this->_buildSectionContentInputWrapper($filterName, $attr, $optionGroup);

			$filterFieldsHtml .= $this->_buildSectionContentInput($filterName, $attr, $optionGroup, $option, $optionSafeName, $optionGroupSafeName);

			$filterFieldsHtml .= $this->_buildSectionContentLabel($filterName, $attr, $optionGroup, $option, $optionSafeName);

			$filterFieldsHtml .= $this->_buildSectionContentInputWrapper($filterName, $attr, $optionGroup, true);
		}

		$filterFieldsHtml .= $this->_buildSectionContentWrapper($filterName, $attr, $optionGroup, true);

		return $filterFieldsHtml;
	}

/**
 * Build the html string for the section content input. By default it produces a checkbox. The request is
 * checked to see if the option has been selected by the user and if so the checked attribute is added.
 *
 * @param string $filterName              The name of the filter.
 * @param array  $attr                    The configuration of the filter form.
 * @param array  $optionGroup             The option group of the current section.
 * @param array  $option                  The option of the current input.
 * @param string $optionSafeName          The name of the option with dangerous/invalid characters replaced.
 * @param string $optionGroupSafeName     The name of the option group with dangerous/invlaid characters replaced.
 * @param string $sectionContentInputHtml Used for extending to section content input html.
 * @return string Section content input html string.
 */
	protected function _buildSectionContentInput($filterName, $attr, $optionGroup, $option, $optionSafeName, $optionGroupSafeName, $sectionContentInputHtml = '') {
		$attr['sectionContentInput']['checked'] = $this->_isFilterSelected(
			$optionGroupSafeName,
			$optionSafeName
		);

		$sectionContentInputHtml .= $this->Form->input(
			'EvFilter.' . $optionGroupSafeName . '.' . $optionSafeName,
			$attr['sectionContentInput']
		);

		return $sectionContentInputHtml;
	}

/**
 * Build the html string for the section content label.
 *
 * @param string $filterName              The name of the filter.
 * @param array  $attr                    The configuration of the filter form.
 * @param array  $optionGroup             The option group of the current section.
 * @param array  $option                  The option of the current label.
 * @param string $optionSafeName          The name of the option with dangerous/invalid characters replaced.
 * @param string $sectionContentLabelHtml Used for extending to section content label html.
 * @return string Section content input html string.
 */
	protected function _buildSectionContentLabel($filterName, $attr, $optionGroup, $option, $optionSafeName, $sectionContentLabelHtml = '') {
		$attr['sectionContentLabel']['for'] = $attr['sectionContentInput']['id'];

		$text = $option['name'];

		if (!empty($option['label'])) {
			$text = $option['label'];
		}

		$escape = true;
		if (isset($attr['sectionContentLabel']['escape']) && $attr['sectionContentLabel']['escape'] === false) {
			$escape = false;
		}

		if (isset($option['escape']) && $option['escape'] === false) {
			$escape = false;
		}

		if ($escape) {
			$text = h($text);
		}

		$tag = $attr['sectionContentLabel']['tag'];
		unset($attr['sectionContentLabel']['tag']);
		$sectionContentLabelHtml .= $this->Html->tag($tag, $text, $attr['sectionContentLabel']);

		return $sectionContentLabelHtml;
	}

/**
 * Build the html string for the section content input wrapper. If the configuration for the section content
 * input wrapper is empty then it will return the passed html string, default blank.
 *
 * @param string $filterName                     The name of the filter.
 * @param array  $attr                           The configuration of the filter form.
 * @param array  $optionGroup                    The option group of the current section.
 * @param bool   $close                          True if we are closing the wrapper tag. Defaults to false.
 * @param string $sectionContentInputWrapperHtml Used for extending to section content input wrapper html.
 * @return string Section content input wrapper html string.
 */
	protected function _buildSectionContentInputWrapper($filterName, $attr, $optionGroup, $close = false, $sectionContentInputWrapperHtml = '') {
		if (empty($attr['sectionContentInputWrapper'])) {
			return $sectionContentInputWrapperHtml;
		}

		$tag = $attr['sectionContentInputWrapper']['tag'];
		unset($attr['sectionContentInputWrapper']['tag']);

		if ($close) {
			$sectionContentInputWrapperHtml .= $this->Html->tag('/' . $tag);
			return $sectionContentInputWrapperHtml;
		}

		$sectionContentInputWrapperHtml .= $this->Html->tag($tag, null, $attr['sectionContentInputWrapper']);

		return $sectionContentInputWrapperHtml;
	}

/**
 * Buidl the html string for the section content wrapper. If the configuration for the section content
 * wrapper is empty then it will return the passed html string, default blank. The filterName is
 * appended to the section content wrapper id so that the collapse works with multiple filter forms on
 * a page.
 *
 * @param string $filterName                The name of the filter.
 * @param array  $attr                      The configuration of the filter form.
 * @param array  $optionGroup               The option group of the current section.
 * @param bool   $close                     True if we are closing the wrapper tag. Defaults to false.
 * @param string $sectionContentWrapperHtml Used for extending to section content wrapper html.
 * @return string Section content wrapper html string.
 */
	protected function _buildSectionContentWrapper($filterName, $attr, $optionGroup, $close = false, $sectionContentWrapperHtml = '') {
		if (empty($attr['sectionContentWrapper'])) {
			return $sectionContentWrapperHtml;
		}

		$tag = $attr['sectionContentWrapper']['tag'];
		unset($attr['sectionContentWrapper']['tag']);

		if ($close) {
			$sectionContentWrapperHtml .= $this->Html->tag('/' . $tag);
			return $sectionContentWrapperHtml;
		}

		$attr['sectionContentWrapper']['id'] .= $filterName . '-' . InflectorExt::slug(strtolower(h($optionGroup['name'])));
		$sectionContentWrapperHtml .= $this->Html->tag($tag, null, $attr['sectionContentWrapper']);

		return $sectionContentWrapperHtml;
	}

/**
 * Build the html string for the filter section group content. For each filter group options, the content
 * contains: a wrapper, section content for the options of the filter group, a group input and a group
 * label. If a configuration is set for a filter group option then that will be merged with the current
 * filter form configuration. Each group is checked to see if all the options below it have been selected
 * and if they have then the group input will be checked as selected.
 *
 * @param string $filterName            The name of the filter.
 * @param array  $attr                  The configuration of the filter form.
 * @param array  $optionGroup           The option group of the current section.
 * @param string $filterGroupFieldsHtml Used for extending to filter section group content html.
 * @return string Filter section group content html string.
 */
	protected function _buildFilterSectionGroupContent($filterName, $attr, $optionGroup, $filterGroupFieldsHtml = '') {
		foreach ($optionGroup['FilterGroupOption'] as $filterOptionItem) {
			if (!empty($filterOptionItem['attr'])) {
				$attr = Hash::merge($attr, $filterOptionItem);
			}

			$groupSelected = $this->_isGroupFilterSelected(
				$optionGroup['name'],
				$filterOptionItem
			);

			$filterGroupFieldsHtml .= $this->_buildGroupSectionContentWrapper($filterName, $attr, $optionGroup, $filterOptionItem);

			$filterGroupHtml = $this->_buildGroupSectionContent($filterName, $attr, $optionGroup, $filterOptionItem);

			$filterGroupHtml .= $this->_buildGroupSectionContentInput($filterName, $attr, $optionGroup, $filterOptionItem, $groupSelected);

			$filterGroupHtml .= $this->_buildGroupSectionContentLabel($filterName, $attr, $optionGroup, $filterOptionItem);

			$filterGroupFieldsHtml .= $this->_buildGroupSectionContentWrapper($filterName, $attr, $optionGroup, $filterOptionItem, true);
		}

		return $filterGroupFieldsHtml;
	}

/**
 * Build the html string for group section content. This extends @_buildFilterSectionContent, replacing the
 * section content configurations with group section content configurations. It also removes the section
 * content input wrapper.
 *
 * @param string $filterName              The name of the filter.
 * @param array  $attr                    The configuration of the filter form.
 * @param array  $optionGroup             The option group of the current section.
 * @param array  $filterOptionItem        The filter group option of the current section.
 * @param string $groupSectionContentHtml Used for extending to group section content html.
 * @return string Group section content html string.
 */
	protected function _buildGroupSectionContent($filterName, $attr, $optionGroup, $filterOptionItem, $groupSectionContentHtml = '') {
		if (isset($attr['sectionContentInputWrapper'])) {
			unset($attr['sectionContentInputWrapper']);
		}

		$sectionContentWrapperAttr = $attr;
		if (!empty($attr['groupSectionContentWrapper'])) {
			$sectionContentWrapperAttr['sectionContentWrapper'] = $attr['groupSectionContentWrapper'];
		}

		if (!empty($attr['groupSectionContentInput'])) {
			$sectionContentWrapperAttr['sectionContentInput'] = $attr['groupSectionContentInput'];
		}
		$groupSectionContentHtml .= $this->_buildFilterSectionContent($filterName, $sectionContentWrapperAttr, $optionGroup);

		return $groupSectionContentHtml;
	}

/**
 * Build the html string for the group section content input. This is the input that will be selecting all
 * the options below it.
 *
 * @param string $filterName                   The name of the filter.
 * @param array  $attr                         The configuration of the filter form.
 * @param array  $optionGroup                  The option group of the current section.
 * @param array  $filterOptionItem             The filter group option of the current section.
 * @param bool   $groupSelected                True if the group has been selected. False otherwise.
 * @param string $groupSectionContentInputHtml Used for extending to group section content input html.
 * @return string Group section content input html string.
 */
	protected function _buildGroupSectionContentInput($filterName, $attr, $optionGroup, $filterOptionItem, $groupSelected, $groupSectionContentInputHtml = '') {
		$attr['groupSectionContentGroupInput']['id'] = 'filterGf' . $filterOptionItem['id'];
		$attr['groupSectionContentGroupInput']['checked'] = $groupSelected;

		$groupSectionContentInputHtml .= $this->Form->input(false, $attr['groupSectionContentGroupInput']);

		return $groupSectionContentInputHtml;
	}

/**
 * Build the html string for the group section content label.
 *
 * @param string $filterName                        The name of the filter.
 * @param array  $attr                              The configuration of the filter form.
 * @param array  $optionGroup                       The option group of the current section.
 * @param array  $filterOptionItem                  The filter group option of the current section.
 * @param string $groupSectionContentGroupLabelHtml Used for extending to group section content label html.
 * @return string Group section content label html string.
 */
	protected function _buildGroupSectionContentLabel($filterName, $attr, $optionGroup, $filterOptionItem, $groupSectionContentGroupLabelHtml = '') {
		$attr['groupSectionContentGroupLabel']['for'] = 'filterGf' . $filterOptionItem['id'];

		$tag = $attr['groupSectionContentGroupLabel']['tag'];
		unset($attr['groupSectionContentGroupLabel']['tag']);

		$groupSectionContentGroupLabelHtml .= $this->Html->tag($tag, $filterOptionItem['name'], $attr['groupSectionContentGroupLabel']);

		return $groupSectionContentGroupLabelHtml;
	}

/**
 * Build the group section content wrapper html string. This is the wrapper that contains all the group
 * inputs in the current section.
 *
 * @param string $filterName                     The name of the filter.
 * @param array  $attr                           The configuration of the filter form.
 * @param array  $optionGroup                    The option group of the current section.
 * @param array  $filterOptionItem               The filter group option of the current section.
 * @param bool   $close                          True if we are closing the wrapper tag. Defaults to false.
 * @param string $groupSectionContentWrapperHtml Used for extending to group section content wrapper html.
 * @return string Group section content wrapper html string.
 */
	protected function _buildGroupSectionContentWrapper($filterName, $attr, $optionGroup, $filterOptionItem, $close = false, $groupSectionContentWrapperHtml = '') {
		$tag = $attr['groupSectionContentGroupWrapper']['tag'];
		unset($attr['groupSectionContentGroupWrapper']['tag']);

		if ($close) {
			$groupSectionContentWrapperHtml .= $this->Html->tag('/' . $tag);
			return $groupSectionContentWrapperHtml;
		}

		$groupSectionContentWrapperHtml .= $this->Html->tag(
			$tag,
			$filterGroupHtml,
			$attr['groupSectionContentGroupWrapper']
		);

		return $groupSectionContentWrapperHtml;
	}

/**
 * Build the filter clear link. This can be displayed at top of the filter form in the filter header or at the
 * bottom of the filter form in it's own section. The url needs to be provided. Normally it would link back to
 * the current page without any of the filters included.
 *
 * @param string $filterName    The name of the filter.
 * @param array  $attr          The configuration of the filter form.
 * @param string $clearLinkHtml Used for extending to clear link html.
 * @return string Filter clear link html string.
 */
	protected function _buildFilterClearLink($filterName, $attr, $top = false, $clearLinkHtml = '') {
		$tag = $attr['clearLinkWrapper']['tag'];
		unset($attr['clearLinkWrapper']['tag']);

		$clearFilterLink = $this->Html->link(
			$attr['clearLink']['name'],
			$attr['clearLink']['url'],
			$attr['clearLink']
		);

		if ($top) {
			return $clearFilterLink;
		}

		$clearLinkHtml .= $this->Html->tag(
			$tag,
			$clearFilterLink,
			$attr['clearLinkWrapper']
		);

		return $clearLinkHtml;
	}
}
