<?php

App::uses('AppComponent', 'Controller/Component');

class FiltersComponent extends AppComponent {

	public $keepFilters = [
		'limit',
		'order',
		'page',
	];

/**
 * Processes the supplied filter parameters and returns a route array, allowing
 * us to redirect to a bookmarkable URL
 *
 * @param array $request Array containing the current request parameters ($this->request)
 * @return string Containing the route to redirect to
 */
	public function getRedirectFromFilters($request = []) {
		// Filters have been applied (on the index page),
		// redirect to include the named parameters
		$newRoute = $request->params;

		$newRoute = $this->_addPassedParamsToRoute($newRoute, $request);

		$newRoute = $this->_addNamedParamsToRoute($newRoute, $request);

		$newRoute = $this->_addQueryParamsToRoute($newRoute, $request);

		$addedFilter = false;

		//Take out the passed in filter data and assign it to the new route
		if (! empty($request->data['EvFilter'])) {
			foreach ($request->data['EvFilter'] as $filterGroup => $filterOption) {
				if (is_array($filterOption)) {
					$newRoute = $this->_addMultiOptionFilterToRoute($newRoute, $filterGroup, $filterOption, $addedFilter);
				} else {
					$newRoute = $this->_addSingleOptionFilterToRoute($newRoute, $filterGroup, $filterOption, $addedFilter);
				}
			}
		}

		$newRoute = $this->_checkToResetPage($newRoute, $addedFilter);

		return Router::url($newRoute);
	}

/**
 * Take any passed request parameters and assign them to the new route. If any passed parameters are set in
 * the new route then they are removed.
 *
 * @param array $newRoute A CakePhp route array.
 * @param array $request  A CakePhp request array.
 * @return array The route array with passed params added.
 */
	protected function _addPassedParamsToRoute($newRoute, $request) {
		if (!empty($request->params['pass'])) {
			//Take out the passedData and assign it to the new route
			foreach ($request->params['pass'] as $passedData) {
				$newRoute[] = $passedData;
			}

			if (isset($newRoute['pass'])) {
				unset($newRoute['pass']);
			}
		}

		return $newRoute;
	}

/**
 * Take any named request parameters and assign them to the new route. If any named parameters are set in
 * the new route then they are removed.
 *
 * @param array $newRoute A CakePhp route array.
 * @param array $request  A CakePhp request array.
 * @return array The route array with named params added.
 */
	protected function _addNamedParamsToRoute($newRoute, $request) {
		if (!empty($request->params['named'])) {
			//Take out the named parameters and assign them to the new route
			foreach ($request->params['named'] as $namedKey => $namedData) {
				$newRoute[$namedKey] = $namedData;
			}

			if (isset($newRoute['named'])) {
				unset($newRoute['named']);
			}
		}

		return $newRoute;
	}

/**
 * Take any query request parameters and assign them to the new route. If any query parameters are set in
 * the new route then they are removed.
 *
 * @param array $newRoute A CakePhp route array.
 * @param array $request  A CakePhp request array.
 * @return array The route array with query params added.
 */
	protected function _addQueryParamsToRoute($newRoute, $request) {
		if (!empty($request->query)) {
			//Take out the query data and add it to the new route
			foreach ($request->query as $queryKey => $queryData) {
				$newRoute['?'][$queryKey] = $queryData;
			}

			if (isset($newRoute['named'])) {
				unset($newRoute['named']);
			}
		}

		return $newRoute;
	}

/**
 * Add an option group with a single option to the new route. This is also used to add the standard CakePhp
 * filters.
 *
 * @param array  $newRoute     A CakePhp route array.
 * @param string $filterGroup  The name of the filter option group to add.
 * @param string $filterOption The name of the filter option to add.
 * @param bool   &$addedFilter True if we have added a filter. False otherwise.
 * @return array The route array with single options added.
 */
	protected function _addSingleOptionFilterToRoute($newRoute, $filterGroup, $filterOption, &$addedFilter) {
		if (empty($filterOption)) {
			if (isset($newRoute[$filterGroup])) {
				unset($newRoute[$filterGroup]);
			}
			return $newRoute;
		}

		$filterOption = InflectorExt::slugUrlSafePreserveChars(
			$filterOption,
			true
		);

		$newRoute[$filterGroup] = $filterOption;
		$addedFilter = true;

		return $newRoute;
	}

/**
 * Add an option group with a multiple options to the new route.
 *
 * @param array  $newRoute     A CakePhp route array.
 * @param string $filterGroup  The name of the filter option group to add.
 * @param string $filterOption The name of the filter option to add.
 * @param bool   &$addedFilter True if we have added a filter. False otherwise.
 * @return array The route array with single options added.
 */
	protected function _addMultiOptionFilterToRoute($newRoute, $filterGroup, $filterOption, &$addedFilter) {
		foreach ($filterOption as $optionName => $selected) {
			$urlFilterGroup = $filterGroup;
			$urlOptionName = $urlFilterGroup . '-' . $optionName;

			if ($selected != '0') {
				$newRoute[$urlOptionName] = $urlFilterGroup;
				$addedFilter = true;
			} else {
				if (isset($newRoute[$urlOptionName])) {
					unset($newRoute[$urlOptionName]);
				}
			}
		}

		return $newRoute;
	}

/**
 * When creating a new route, check to see if the route should reset the paging on the current filter. If a
 * filter has been added that may effect the number of total results then the page needs to return the first
 * page.
 *
 * @param array $newRoute    A CakePhp route array.
 * @param bool  $addedFilter True if the page needs resetting. False otherwise.
 * @return array The route array with page parameter removed or not.
 */
	protected function _checkToResetPage($newRoute, $addedFilter) {
		if ($addedFilter && isset($newRoute['page'])) {
			unset($newRoute['page']);
		}

		return $newRoute;
	}

/**
 * Loops through the supplied named parameters and humanizes strings and
 * reinstates decimal points where required
 *
 * @param array $namedParams Contains the named request parameters to clean
 * @return array Containing a sanatized collection of named request params
 */
	public function sanatizeParams($namedParams = []) {
		$newParamArray = [];

		if (! empty($namedParams)) {
			foreach ($namedParams as $paramName => $paramValue) {
				if (in_array($paramName, $this->keepFilters)) {
					$newParamArray = $this->_sanitizeSingleOptionFilter($newParamArray, $paramName, $paramValue);
				} else {
					$newParamArray = $this->_sanitizeMultiOptionFilter($newParamArray, $paramName, $paramValue);
				}
			}
		}

		return $newParamArray;
	}

/**
 * Sanitise a single option filter. It can be read as it has been provided once it has had it's characters
 * returned from being encoded.
 *
 * @param array  $params     The sanitised array of parameters.
 * @param string $paramName  The parameter name to sanitise.
 * @param string $paramValue The parameter value to sanitise.
 * @return array The sanitised array of parameters.
 */
	protected function _sanitizeSingleOptionFilter($params, $paramName, $paramValue) {
		$paramName = InflectorExt::slugUrlSafePreserveChars(
			$paramName,
			false
		);

		$paramValue = InflectorExt::slugUrlSafePreserveChars(
			$paramValue,
			false
		);

		$params[$paramName] = $paramValue;

		return $params;
	}

/**
 * Sanitise a mutliple option filter. The option group was appended to the request parameter so that multiple
 * values could be added to the request. The option group needs to be removed after it's characters have been
 * returned from being encoded. If the option isn't in the correct format for multiple options then it is
 * treated as a single option.
 *
 * @param array  $params     The sanitised array of parameters.
 * @param string $paramName  The parameter name to sanitise.
 * @param string $paramValue The parameter value to sanitise.
 * @return array The sanitised array of parameters.
 */
	protected function _sanitizeMultiOptionFilter($params, $paramName, $paramValue) {
		$paramName = InflectorExt::slugUrlSafePreserveChars($paramName, false);
		$paramValue = InflectorExt::slugUrlSafePreserveChars($paramValue, false);

		// Filter options are presented as Group-Option => Group
		// Ignore any that are not in this format
		if (strpos($paramName, $paramValue . '-') === 0) {
			$paramName = substr( $paramName, strlen($paramValue) + 1 );
			$params[$paramValue][] = $paramName;
		} else {
			$params = $this->_sanitizeSingleOptionFilter($params, $paramName, $paramValue);
		}

		return $params;
	}
}
