<?php

App::uses('FaqsAppModel', 'Faqs.Model');

class Faq extends FaqsAppModel {

	public $displayField = 'question';

	public $belongsTo = array(
		'FaqCategory' => array('className' => 'Faqs.FaqCategory')
	);

	public $validate = array(
		'question' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Please add a question'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 250),
				'message' => 'No more than 250 characters long'
			)
		),
		'answer' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Please add an answer'
			)
		),
		'sequence' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Please add a sequence'
			),
			'integer' => array(
				'rule' => array('naturalNumber', true),
				'message' => 'Integer values only'
			)
		)
	);

	public $order = array(
		'Faq.sequence' => 'asc',
		'Faq.question' => 'asc'
	);

/**
 * Gets the FAQs grouped by the FAQ categories in sequential order.
 *
 * @return array
 */
	public function getFaqs() {
		return $this->getAllFaqs();
	}

/**
 * Get all FAQs. Provide query parameters to customise the results.
 *
 * @param array $params Custom query parameters.
 * @return array The found FAQs.
 */
	public function getAllFaqs(array $params = []) {
		if (Configure::read('Faqs.has_categories') === true) {
			$params = $this->_getAllFaqsCategoryParams($params);
		}

		$params = $this->_getAllFaqsDefaultParams($params);

		$data = $this->find('all', $params);

		$data = $this->_formatAllFaqs($data);

		return $data;
	}

/**
 * Add the category query parameterss to a query parameter array so that the results can be grouped by FAQ category.
 *
 * @param array $params The current query parameters.
 * @return array The modified query parameters.
 */
	protected function _getAllFaqsCategoryParams(array $params) {
		$params['contain']['FaqCategory']['conditions']['FaqCategory.is_active'] = true;

		$params['order']['FaqCategory.sequence'] = 'ASC';
		$params['order']['FaqCategory.name'] = 'ASC';

		return $params;
	}

/**
 * Add the default parameters for getting all FAQs to a query parameter array.
 *
 * @param array $params The current query parameters.
 * @return array The modified query parameters.
 */
	protected function _getAllFaqsDefaultParams(array $params) {
		$params['conditions'][$this->alias . '.is_active'] = true;

		$params['order'][$this->alias . '.sequence'] = 'ASC';
		$params['order'][$this->alias . '.question'] = 'ASC';

		return $params;
	}

/**
 * Format the data returned from _getAllFaqs. Put the FAQs into the category they belong to, if categories aren't being
 * used then a 0-keyed element is created with the FAQs in there.
 *
 * @param array $data The data returned from _getAllFAQs find.
 * @return array The formatted data.
 */
	protected function _formatAllFaqs(array $data) {
		$formattedData = [];

		foreach ($data as $faq) {
			if (Configure::read('Faqs.has_categories') === true && !empty($faq['FaqCategory']['id'])) {
				$id = $faq['FaqCategory']['id'];

				if (empty($formattedData[$id])) {
					$formattedData[$id]['FaqCategory'] = $faq['FaqCategory'];
				}
			} else {
				$id = 0;
			}

			$formattedData[$id]['Faq'][] = $faq['Faq'];
		}

		return $formattedData;
	}
}
