<?php
App::uses('ErrorHandler', 'Error');

use Airbrake\Instance;
use Airbrake\Notifier;

class Errbit extends ErrorHandler {

	private static $__errbit;

	private static $__notifier;

	/**
	 * Init the airbrake/errbit api instance (intentionally not in the constructor)
	 *
	 * @return void
	 */
	public static function init() {
		if (
			empty(Configure::read('EvErrbit.projectKey')) ||
			empty(Configure::read('EvErrbit.projectId'))) {
			// If the project key or project id are missing, then the plugin
			// has not been configured so we dont want to use it.
			return false;
		}

		// We don't need to know when a developer does something wrong locally
		if (
			Configure::read('EvErrbit.alwaysReport') == false &&
			Configure::read('app.environment') == 'DEVELOPMENT'
		) {
			return false;
		}

		self::$__notifier = new Airbrake\Notifier([
			'projectId' => Configure::read('EvErrbit.projectId'),
			'projectKey' => Configure::read('EvErrbit.projectKey'),
			'host' => Configure::read('EvErrbit.host'),
			'rootDirectory' => '/',
			'httpClient' => 'curl',
			'appVersion' => '1',
			'environment' => Configure::read('app.environment')

		]);

		Airbrake\Instance::set(self::$__notifier);

		self::$__errbit = new Airbrake\ErrorHandler(self::$__notifier);
		self::$__errbit->register();

		return true;
	}

	/**
	 * Catch errors and run them through Errbit before passing back to cake
	 */
	public static function handleError($code, $description, $file = null, $line = null, $context = null) {
		if (self::init()) {
			self::onError($code, $description, $file, $line);
		}
		return parent::handleError($code, $description, $file, $line, $context);
	}

	/**
	 * Catch exceptions and run them through Errbit before passing back to cake
	 */
	public static function handleException($exception) {
		if (self::init()) {
			if (self::filterErrbitExceptions($exception)) {
				Airbrake\Instance::notify($exception);
			}
		}

		return parent::handleException($exception);
	}

	/**
	 * Filter out certain exceptions, we don't want to report 404 errors to Errbit
	 * @param $exception Application reported exception
	 */
	public static function filterErrbitExceptions($exception) {
		$ignoredErrors = Configure::read('EvErrbit.ignoredCodes');
		if (in_array($exception->getCode(), $ignoredErrors)) {
			return false;
		}
		return true;
	}

	/**
	 * Catch exceptions and run them through Errbit but don't pass it back to cake.
	 * Used to just create a errbit notification
	 */
	public static function notify($exception) {
		if (self::init()) {
			self::_filterData($_REQUEST['data'], 25);

			Airbrake\Instance::notify($exception);
		}

		return true;
	}

	/**
	 * Handles different error types, pasing them to Airbrake/Errbit's different
	 * handlers
	 */
	public static function onError($code, $message, $file, $line) {
		switch ($code) {
			case E_NOTICE:
			case E_USER_NOTICE:
				$exc = new Airbrake\Errors\Notice($message, debug_backtrace());
				break;
			case E_WARNING:
			case E_USER_WARNING:
				$exc = new Airbrake\Errors\Warning($message, debug_backtrace());
				break;
			case E_ERROR:
			case E_CORE_ERROR:
			case E_RECOVERABLE_ERROR:
				$exc = new Airbrake\Errors\Fatal($message, debug_backtrace());
				break;
			case E_USER_ERROR:
			default:
				$exc = new Airbrake\Errors\Error($message, debug_backtrace());
		}
		self::$__notifier->notify($exc);
	}

/**
 * Filter data so that certain fields aren't displayed in the error report. Extend the filters
 * using the filters array. The key being the field you want to filter and the value being what
 * you want to replace the value with. Recursive call to replace any fields.
 * @param  array  &$data        The data to check and replace matching fields.
 * @param  int    $maxDepth     The maximum level to recurse to.
 * @param  int    $currentDepth The current level that is being recursed.
 * @return array                The data as passed in with any changes made.
 */
	protected static function _filterData(&$data = null, $maxDepth = 1, $currentDepth = 0) {
		if (!empty($data)) {
			if ($maxDepth > $currentDepth) {
				$filters = Configure::read('EvErrbit.filters');

				if (!empty($filters)) {
					$currentKeys = array_keys($data);
					$filterKeys = array_keys($filters);

					foreach ($currentKeys as $key) {
						if (in_array($key, $filterKeys)) {
							$filterReplacement = '*****';
							if (!empty($filters[$key])) {
								$filterReplacement = $filters[$key];
							}

							$data[$key] = $filterReplacement;
						} else {
							if (!empty($data[$key]) && is_array($data[$key])) {
								self::_filterData($data[$key], $maxDepth, $currentDepth++);
							}
						}
					}
				}
			}
		}

		return $data;
	}
}
