<?php

App::uses('EvEventAppModel', 'EvEvent.Model');

class Event extends EvEventAppModel {

	public $belongsTo = array(
		'Country' => array(
			'className' => 'EvCountry.Country'
		),
		'EventCategory' => array(
			'className' => 'EvEvent.EventCategory'
		)
	);

	public $actsAs = array(
		'EvTemplates.Template' => array(
			'formInject' => true
		),
		'Routable.Routable' => array(
			'alias' => 'events/:displayField',
			'actual' => ':plugin/:controller/view/:primaryKey'
		),
		'MetaData.Meta'
	);

	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => array(
				'alt'
			)
		),
		'main' => array(
			'slots' => -1,
			'fields' => array(
				'alt'
			)
		),
	);

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 100),
				'message' => 'No more than 100 characters long'
			)
		),
		'date' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Required'
			),
			'datetime' => array(
				'rule' => array('datetime')
			)
		),
		'end_date' => array(
			'datetime' => array(
				'rule' => array('datetime'),
				'allowEmpty' => true
			),
			'afterStartDate' => array(
				'rule' => 'afterStartDate',
				'message' => 'The end date must be after the start date or blank if not required'
			)
		),
		'organiser' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'organiser_url' => array(
			'url' => array(
				'rule' => array('url', true),
				'message' => 'Valid URLs only',
				'allowEmpty' => true
			),
			'maxLength' => array(
				'rule' => array('maxLength', 255),
				'message' => 'No more than 255 characters long'
			)
		),
		'address_line_1' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'address_line_2' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'address_line_3' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'city' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'county' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'postcode' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 15),
				'message' => 'No more than 15 characters long',
				'allowEmpty' => true
			)
		),
		'country_id' => array(
			'required' => array(
				'rule' => array('maxLength', 15),
				'message' => 'Required'
			)
		)
	);

	public $order = 'Event.date DESC';

/**
 * Validation rule for checking the end date is after the start date
 */
	public function afterStartDate() {
		return $this->data[$this->alias]['end_date'] > $this->data[$this->alias]['date'];
	}

	public function readForView($id, $query = []) {
		$query['contain'][] = 'EventCategory';
		$query['contain'][] = 'Country';

		return parent::readForView($id, $query);
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);
		$length = Configure::read('EvEvent.snippet_length');

		foreach ($results as &$event) {

			if (!empty($event['Event'])) {

				if (isset($event['Event']['body'])) {
					if ($length > 0) {

						// Set a reduced text snippet.
						$snippet = strip_tags($event['Event']['body']);
						if (strlen($snippet) > $length) {
							$snippet = substr($snippet, 0, $length) . ' &hellip;';
						}

						$event['Event']['snippet'] = $snippet;

					} else {

						// Snippets want to be the full text.
						$event['Event']['snippet'] = $event['Event']['body'];

					}
				}

				if (! empty($event['Event']['map_data'])) {

					$event['Event']['map_params'] = json_decode($event['Event']['map_data'], true);

				}

			}

		}

		return $results;
	}

	public function beforeSave($options = array()) {
		if (! empty($this->data['Event']['date']) && ! empty($this->data['Event']['date_gmt'])) {
			// Convert date to GMT.
			$this->data['Event']['date'] = CakeTime::toServer($this->data['Event']['date'], Configure::read('Config.timezone'));
		}

		if (! empty($this->data['Event']['end_date']) && ! empty($this->data['Event']['end_date_gmt'])) {
			// Convert date to GMT.
			$this->data['Event']['end_date'] = CakeTime::toServer($this->data['Event']['end_date'], Configure::read('Config.timezone'));
		}

		return parent::beforeSave($options);
	}

}
