<?php

App::uses('EvEventAppController', 'EvEvent.Controller');

class EventsController extends EvEventAppController {

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view',
			'archive'
		));

		return;
	}

/**
 * Main events page.
 *
 * @return void
 */
	public function index() {
		$this->loadModel('Page');

		// See if there is a linked content page.
		$pageId = Configure::read('EvEvent.page_id');
		if (!empty($pageId)) {
			$this->assignPage($pageId, 'page');
		}

		$params = array(
			'conditions' => array(
				'OR' => array(
					'Event.date >=' => date('Y-m-d H:i:s'),
					'Event.end_date >=' => date('Y-m-d H:i:s'),
				),
			)
		);

		$this->paginate = $this->_setPaginationParams($params);

		$this->set('data', $this->paginate());

		$this->widgetFutureEvents();
		$this->widgetCategories();

		$this->view = 'EvEvent.index';

		return;
	}

/**
 * Event listing.
 *
 * @param integer $id event ID
 * @return void
 */
	public function view($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$data = $Model->readForViewOrFail($id);

		if (empty($data) || (!$this->Auth->user() && $data[$modelAlias]['logged_in_only'])) {
			throw new NotFoundException();
		}

		$this->Meta->set($data);
		$this->set('data', $data);

		$this->widgetFutureEvents();
		$this->widgetCategories();

		//Preparing plugin for templates.
		if ($Model->hasBehavior('EvTemplates.Template')) {
			$this->view = $this->Tpl->getTemplate($data);
		}
		if (!$this->view) {
			$this->view = 'EvEvent.Events/view';
		}

		return;
	}

/**
 * Events archive to display events belonging to a particular month.
 *
 * @param string $date month to retrieve events for
 * @return void
 */
	public function archive($date) {
		$params = array(
			'conditions' => array(
				'Event.date >=' => date('Y-m-01 00:00:00', strtotime($date)),
				'Event.date <=' => "$date 23:59:59"
			)
		);

		$this->paginate = array_merge_recursive(
			$this->_processEventListingFilter(),
			$this->_eventListingPaginateDefaults(),
			$params
		);

		$data = $this->paginate();

		if (empty($data)) {
			throw new NotFoundException();
		}

		$this->set('data', $data);
		$this->set('date', $date);

		$this->widgetFutureEvents();
		$this->widgetCategories();

		$this->view = 'EvEvent.archive';

		return;
	}

/**
 * Whitelist columns to display on the admin listing page.
 *
 * @return array
 * @see EvCoreController::_adminIndexColumnsWhitelist()
 */
	protected function _adminIndexColumnsWhitelist() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = parent::_adminIndexColumnsWhitelist();

		// Include a column for the event date.
		$columns[] = "$modelAlias.date";

		return $columns;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);

		if (empty($this->request->data['Event']['country_id'])) {
			$defaultCountryId = Configure::read('EvEvent.default_country_id');
			if (!empty($defaultCountryId)) {
				$this->request->data['Event']['country_id'] = $defaultCountryId;
			}
		}

		return;
	}

/**
 * Populare admin lookups.
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		$eventCategories = [0 => __('Uncategorised')];
		$eventCategories += $Model->EventCategory->find('list');
		$this->set(compact('eventCategories'));

		$this->set('countries', $Model->Country->find('list'));

		return;
	}

	protected function _adminFormFields() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = parent::_adminFormFields();

		$fields[$modelAlias . '.organiser_url']['label'] = 'Organiser\'s URL';

		$fields[$modelAlias . '.logged_in_only']['label'] = 'Visible only to registered users';
		$fields[$modelAlias . '.logged_in_only']['info'] = 'When this option is checked, only registered users will be able to see this event.';

		return $fields;
	}

/**
 * Creates the query for the pagination
 *
 * @param array $params The params for the query
 *
 * @return array
 */
	protected function _setPaginationParams($params = array()) {
		return array_merge_recursive(
			$this->_processEventListingFilter(),
			$this->_eventListingPaginateDefaults(),
			$params
		);
	}

}
