<?php

class ExportController extends AppController {

	public $uses = array();

	public function beforeFilter() {
		$this->adminActions[] = 'admin_index';
		parent::beforeFilter();
	}

/**
 * Exports orders placed in the system to a CSV for download
 * @return null
 */
	public function admin_index() {
		// Rather memory intensive so increase memory before starting
		ini_set('memory_limit', '512M');

		if ($this->request->is('post') || $this->request->is('put')) {
			if (empty($this->request->data['Order']['start_date']) || empty($this->request->data['Order']['end_date'])) {
				$this->Session->setFlash('Please enter a start and end date range', 'flash_fail');
				return $this->redirect($this->here);
			}

			$exportOrderRows = [];

			$orders = $this->_getExportOrders($this->request->data['Order']['start_date'], $this->request->data['Order']['end_date']);

			if (empty($orders)) {
				$this->Session->setFlash('There are no orders to export for this date range', 'flash_fail');
				return $this->redirect($this->here);
			}

			foreach ($orders as $order) {
				$exportOrderRows = array_merge($exportOrderRows, $this->_buildOrderRows($order));
			}

			// Force CSV to download
			$filename = 'orders-export-' . date('j-m-y H:m') . '.csv';
			$this->response->download($filename);

			$data = $exportOrderRows;
			$_header = array_keys($exportOrderRows[0]);
			$_serialize = 'data';
			$this->viewClass = 'CsvView.Csv';
			$this->set(compact('data', '_serialize', '_header'));
		} else {
			// Displays listing page
		}
	}

/**
 * Gets the orders that need to be exported
 * @return array Array of orders that match the criteria, by default PAID status
 */
	protected function _getExportOrders($startDate, $endDate) {
		$Order = EvClassRegistry::init('EvCheckout.Order');
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$unPaidStatusId = $OrderStatus->getConstant('UNPAID');

		$orders = $Order->find('all', [
			'conditions' => [
				'Order.order_status_id != ' => $unPaidStatusId,
				'Order.created >= ' => $startDate,
				'Order.created <= ' => $endDate
			],
			'contain' => [
				'OrderItem',
				'OrderTotal',
				'OrderData',
				'OrderStatus'
			],
			'order' => 'Order.id DESC'
		]);

		return $orders;
	}

/**
 * Builds export rows for a given order
 */
	protected function _buildOrderRows($order) {
		//If the order has no order items, skip
		if (empty($order['OrderItem'])) {
			return [];
		}

		// Build general order information array
		$generalOrderInformation = [
			'order_ref' => $order['Order']['id'],
			'order_status' => $order['OrderStatus']['name'],
			'name' => $order['Order']['name'],
			'email' => $order['Order']['email'],
			'devliery_address' => str_replace("<br>", "\n", $order['Order']['delivery_address']),
			'guest_checkout' => $order['Order']['is_guest_checkout'] ? 'yes' : 'no',
			'order_date' => $order['Order']['created']
		];

		$exportRows = array();
		//Loop through each order item and create row
		foreach ($order['OrderItem'] as $orderItem) {
			$orderItemRow = [];
			$orderItemRow += $generalOrderInformation;
			$orderItemRow += [
				// Order item specific information
				'row_type' => 'item',
				'item_name' => $orderItem['name'],
				'item_quantity' => $orderItem['quantity'],
				'item_tax_rate' => $orderItem['tax_rate'],
				'item_unit_price' => $orderItem['unit_price_inc_tax'],
				'item_row_total' => $orderItem['row_total_inc_tax']
			];

			$exportRows[] = $orderItemRow;
		}

		// Go through the order totals and add delivery and discount rows
		$orderTotals = Hash::combine($order['OrderTotal'], '{n}.name', '{n}');
		$orderData = Hash::combine($order['OrderData'], '{n}.name', '{n}');

		// Add delivery row
		if (isset($orderTotals[Configure::read('EvBasket.labels.delivery')])) {
			$orderDeliveryRow = [];
			$orderDeliveryRow += $generalOrderInformation;
			$orderDeliveryRow += [
				// Delivery specific information
				'row_type' => 'delivery',
				'item_name' => Configure::read('EvBasket.labels.delivery'),
				'item_quantity' => 1,
				'item_tax_rate' => '',
				'item_unit_price' => $orderTotals[Configure::read('EvBasket.labels.delivery')]['display_inc_tax'],
				'item_row_total' => $orderTotals[Configure::read('EvBasket.labels.delivery')]['display_inc_tax']
			];

			$exportRows[] = $orderDeliveryRow;
		}

		// Add discount row
		if (isset($orderTotals[Configure::read('EvBasket.labels.discount')])) {
			$orderDiscountRow = [];
			$orderDiscountRow += $generalOrderInformation;
			$orderDiscountRow += [
				// Delivery specific information
				'row_type' => 'discount',
				'item_name' => !empty($orderData['discount_code']['data']) ? $orderData['discount_code']['data'] : "-",
				'item_quantity' => 1,
				'item_tax_rate' => '',
				'item_unit_price' => $orderTotals[Configure::read('EvBasket.labels.discount')]['display_inc_tax'],
				'item_row_total' => $orderTotals[Configure::read('EvBasket.labels.discount')]['display_inc_tax']
			];

			$exportRows[] = $orderDiscountRow;
		}

		return $exportRows;
	}
}
