<?php

App::uses('DpdCSVExport', 'EvDpdExport.Lib');

/**
 * DPD Export Controller
 *
 * Adds the ability to export your paid orders as a CSV for DPD
 *
 * @package       EvDPDExport.Controller
 */
class DpdExportController extends AppController {

	public $uses = array();

	public $adminActions = array(
		'admin_export'
	);

	public function admin_export() {
		// Get the orders to export
		$orders = $this->_getExportOrders();

		// Go through the orders and build up CSV export format
		$formattedOrders = [];
		foreach ($orders as $order) {
			$formattedOrders[$order['Order']['id']] = $this->_formatOrder($order);
		}

		// Force CSV to download
		$filename = 'dpd-export-' . date('j-m-y H:m') . '.csv';
		$this->response->download($filename);

		$data = DpdCSVExport::generateCSVArray($formattedOrders);

		$_serialize = 'data';
		$this->viewClass = 'CsvView.Csv';
		$this->set(compact('data', '_serialize'));
	}

/**
 * Formats the order data into fields required for DPD export
 * Override to add additional data
 */
	protected function _formatOrder($order) {
		//
		$orderData = Hash::combine($order, 'OrderData.{n}.name', 'OrderData.{n}.data');

		if (!empty($orderData['delivery_address'])) {
			// this is the latest way of storing broken down addresses
			$address = json_decode($orderData['delivery_address'], true);

			// address3 should contain city according to DPD docs
			if (!empty($address['Address']['address3'])) {
				$address['Address']['address3'] .= ', ' . $address['Address']['city'];
			} else {
				$address['Address']['address3'] = $address['Address']['city'];
			}
		} else {
			// If this project isn't using the latest checkout or has old data
			$address = $this->_decodeAddress($order['Order']['delivery_address']);
		}

		$formattedOrder = [];

		// Address
		$formattedOrder['Address Line 1'] = $address['Address']['address1'];
		$formattedOrder['Address Line 2'] = $address['Address']['address2'];
		$formattedOrder['Address Line 3'] = $address['Address']['address3'];
		$formattedOrder['Postcode'] = $address['Address']['post_code'];
		$formattedOrder['Country Code'] = $address['Country']['iso2'];

		// Contact information
		$formattedOrder['Contact Name'] = $order['Order']['name'];
		$formattedOrder['Contact Telephone'] = $order['Order']['phone'];

		// Contents Description
		$formattedOrder['Content Description'] = $this->_getContentsDescription($order);

		// Order Reference and nofication
		$formattedOrder['Customer Ref 1'] = '#' . $order['Order']['id'];
		$formattedOrder['Notification Email'] = $order['Order']['email'];
		// If a valid UK phone number add to the notification column
		if (preg_match("/^(\+44\s?7\d{3}|\(?07\d{3}\)?)\s?\d{3}\s?\d{3}$/", $order['Order']['phone'])) {
			$formattedOrder['Notification SMS'] = $order['Order']['phone'];
		}

		return $formattedOrder;
	}

/**
 * Takes the String address and breaks it into an array
 * @return Returns an annotated array of the address
 */
	protected function _decodeAddress($address) {
		$formattedAddress = [];

		$address = explode("<br>", $address);

		// Remove blank line on the end
		array_pop($address);
		$formattedAddress['Country'] = array_pop($address);
		$formattedAddress['Address']['post_code'] = array_pop($address);
		$formattedAddress['Address']['address1'] = array_shift($address);
		$formattedAddress['Address']['address2'] = array_shift($address);
		$formattedAddress['Address']['address3'] = implode(", ", $address);

		$Country = EvClassRegistry::init('EvCountry.Country');
		$country = $Country->findByName($formattedAddress['Country']);

		$formattedAddress['Country'] = $country['Country'];
		$formattedAddress['Address']['country_id'] = $country['Country']['id'];

		return $formattedAddress;
	}

/**
 * Gets the orders that need to be exported
 * @return array Array of orders that match the criteria, by default PAID status
 */
	protected function _getExportOrders($queryParams = []) {
		$Order = EvClassRegistry::init('EvCheckout.Order');
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$paidStatusId = $OrderStatus->getConstant('PAID');

		$defaultQueryParams = [
			'conditions' => [
				'Order.order_status_id' => $paidStatusId,
			],
			'contain' => 'OrderData',
		];
		$queryParams = Hash::merge($defaultQueryParams, $queryParams);
		$orders = $Order->find('all', $queryParams);
		return $orders;
	}

/**
 * Gets the contents description for a given order
 * Unless overrridden this uses the config variable from EvDpdExport.genericContentDescription
 * @return string Gets a description of the contents
 */
	protected function _getContentsDescription($order) {
		return Configure::read('EvDpdExport.genericContentDescription');
	}
}
