<?php

App::uses('EvEmailAppModel', 'EvEmail.Model');
App::uses('LibTidy', 'EvCore.Lib');

class Email extends EvEmailAppModel {

/**
 * Belongs to associations
 * @var array
 */
	public $belongsTo = [
		'EmailGroup' => [
			'className' => 'EvEmail.EmailGroup'
		]
	];

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = [
		'name' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Required',
				'on' => 'create'
			],
			'maxLength' => [
				'rule' => ['maxLength', 45],
				'message' => 'No more than 45 characters long'
			]
		],
		'system_name' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Required',
				'on' => 'create'
			],
			'maxLength' => [
				'rule' => ['maxLength', 100],
				'message' => 'No more than 100 characters long'
			]
		],
		'email_group_id' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Required'
			]
		],
		'subject' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Required'
			],
			'maxLength' => [
				'rule' => ['maxLength', 100],
				'message' => 'No more than 100 characters long'
			]
		],
		'content' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Required'
			]
		],
		'subject' => [
			'required' => [
				'rule' => 'notBlank',
				'message' => 'Required'
			],
			'maxLength' => [
				'rule' => ['maxLength', 100],
				'message' => 'No more than 100 characters long'
			]
		],
		'from_name' => [
			'maxLength' => [
				'rule' => ['maxLength', 45],
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			]
		],
		'from_email' => [
			'email' => [
				'rule' => 'email',
				'message' => 'Not a valid email address',
				'allowEmpty' => true
			],
			'maxLength' => [
				'rule' => ['maxLength', 254],
				'message' => 'No more than 254 characters long'
			]
		],
		'cc' => [
			'email' => [
				'rule' => 'validateEmailList',
				'message' => 'Not a list of valid email addresses',
				'allowEmpty' => true
			]
		],
		'bcc' => [
			'email' => [
				'rule' => 'validateEmailList',
				'message' => 'Not a list of valid email addresses',
				'allowEmpty' => true
			]
		]
	];

/**
 * After validate, check that tokens have been set in content.
 * This must come after beforeValidate and before the beforeSave callback to
 * ensure all validation rules are checked to display to the user.
 *
 * @param array $options
 * @return void
 */
	public function afterValidate($options = []) {
		parent::afterValidate();
		if (!empty($this->data[$this->alias]['content'])) {
			$this->validateRequiredTokens($this->data[$this->alias]['content'], 'content');
		}
		return;
	}

/**
 * Check that a list of valid email addresses has been supplied
 *
 * @param string $check
 * @param bool $deep Perform a deeper validation (if true), by also checking availability of host
 * @param string $regex Regex to use (if none it will use built in regex)
 * @return bool
 */
	public function validateEmailList($check, $deep = false, $regex = null) {
		$check = array_pop($check);
		$emails = $this->splitEmails($check);
		foreach ($emails as $email) {
			if (Validation::email($email, $deep, $regex) === false) {
				return false;
			}
		}

		return true;
	}

/**
 * Split out a deliminated list of email addresses (used for CC and BCC fields)
 *
 * @param string $emails String of email addresses
 * @return array Email addresses
 */
	public static function splitEmails($emails) {
		$emails = preg_split('/(\ |,|;)/', $emails);
		return array_filter($emails);
	}

/**
 * Check that all the required tokens exist in the content
 *
 * @param string $check
 * @param string $field
 * @return bool
 */
	public function validateRequiredTokens($check, $field) {
		if (!empty($this->id)) {
			$missingTokens = [];
			if (!array_key_exists('required_tokens', $this->data[$this->alias])) {
				$requiredTokens = explode(',', $this->field('required_tokens'));
			} else {
				$requiredTokens = explode(',', $this->data[$this->alias]['required_tokens']);
			}
			$requiredTokens = array_filter(($requiredTokens));
			if (!empty($requiredTokens) && is_array($requiredTokens)) {
				foreach ($requiredTokens as $token) {
					$token = trim($token);
					if (strpos($check, '{' . $token . '}') === false) {
						$missingTokens[] = '{' . $token . '}';
					}
				}
			}
			// Flag the missing tokens as a validation error.
			if (!empty($missingTokens)) {
				$this->invalidate(
					$field,
					__d(
						'ev_email',
						'The following tokens are missing: %s',
						[implode(', ', $missingTokens)]
					)
				);
				return false;
			}
		}
		return true;
	}

	/**
	 * Generate email content replacing tokens with passed data.
	 *
	 * @param int|string $id Email record ID or system name
	 * @param array $data Data to replace tokens
	 * @return array|bool
	 */
	public function generateEmailData($id, array $data = []) {
		$result = is_int($id) ? $this->findById($id) : $this->findBySystemName($id);
		if (empty($result) || (bool)$result[$this->alias]['is_active'] === false) {
			return false;
		}

		// Replace required tokens.
		$result['Email']['subject'] = $this->_replaceTokens(
			$result['Email']['subject'],
			$result['Email']['required_tokens'],
			$data
		);

		// Replace optional tokens.
		$result['Email']['subject'] = $this->_replaceTokens(
			$result['Email']['subject'],
			$result['Email']['optional_tokens'],
			$data
		);

		// Remove paragraph tags from block-style tokens caused by WYSIWYG editor but still match if they're not there
		$pattern = '#(<p>(?=\s*{==\w+==}\s*</p>))?\s*{==(\w+)==}\s*?(?:(?(1)</p>|))??#isU';
		$result['Email']['content'] = preg_replace($pattern, '{$2}', $result['Email']['content']);

		// Replace required tokens.
		$result['Email']['content'] = $this->_replaceTokens(
			$result['Email']['content'],
			$result['Email']['required_tokens'],
			$data
		);
		unset($result['Email']['required_tokens']);

		// Replace optional tokens.
		$result['Email']['content'] = $this->_replaceTokens(
			$result['Email']['content'],
			$result['Email']['optional_tokens'],
			$data
		);
		unset($result['Email']['optional_tokens']);

		if (!empty(Configure::read('EvEmail.strip_content_whitespace'))) {
			// Remove white space between html tags as this causes problems in gmail.
			$result['Email']['content'] = $this->_splitHtmlString(LibTidy::minify($result['Email']['content'], ['html' => true]), 980);
		}

		// Configure where email will be sent from.
		$fromName = $result['Email']['from_name'] ?: Configure::read('SiteSetting.general.site_title');
		$fromEmail = $result['Email']['from_email'] ?: Configure::read('SiteSetting.general.admin_email');
		$result['Email']['from'] = [$fromEmail => $fromName];
		unset($result['Email']['from_name']);
		unset($result['Email']['from_email']);

		$result['Email']['to'] = null;
		if ($result['Email']['override_to'] === true) {
			$toName = $result['Email']['to_name'] ?: Configure::read('SiteSetting.general.site_title');
			$toEmail = $result['Email']['to_email'] ?: Configure::read('SiteSetting.general.admin_email');
			$result['Email']['to'] = [$toEmail => $toName];
		}
		unset($result['Email']['to_name']);
		unset($result['Email']['to_email']);

		if (!empty($result['Email']['cc'])) {
			$result['Email']['cc'] = $this->splitEmails($result['Email']['cc']);
		}

		if (!empty($result['Email']['bcc'])) {
			$result['Email']['bcc'] = $this->splitEmails($result['Email']['bcc']);
		}

		return $result;
	}

/**
 * Convert an HTML string to plain text.
 *
 * @param string $html HTML content
 * @return string Plain text content
 */
	public function textContent($html) {
		// Convert <br> to line breaks.
		$text = preg_replace('/<br(\s+)?\/?>/i', "\n", $html);
		// Convert </p> to line breaks.
		$text = preg_replace('/<\/p>/i', "\n\n", $html);
		// Remove HTML markup.
		$text = trim(strip_tags($text));

		return $text;
	}

/**
 * Replaces tokens in content.
 *
 * Tokens are in the format of {key}. Use {==key==} to represent a token that needs to remove
 * wrapping markup (for example a basket representation); this can be used to inject a View
 * element into an email.
 *
 * @param string $content Content containing tokens
 * @param string $tokensJson JSON encoded tokens
 * @param array $data Data to replace tokens
 * @return string
 */
	protected function _replaceTokens($content, $tokensJson, array $data = []) {
		$tokens = explode(',', $tokensJson);
		if (!empty($tokens) && is_array($tokens)) {
			$tokens = array_map(
				function ($val) {
					$val = trim($val);
					return str_replace('==', '', $val);
				},
				$tokens
			);
			// Replace each token with the passed data.
			foreach ($tokens as $token) {
				if (array_key_exists($token, $data) === true) {
					$content = str_replace('{' . $token . '}', $data[$token], $content);
				}
			}
		}
		return $content;
	}

/**
 * Convinience method for adding any email to the queue.
 *
 * @param string $subject
 * @param string $content
 * @param array $to
 * @param array $from
 * @param string $template
 * @param array $cc Carbon copy emails
 * @param array $bcc Blind carbon copy emails
 * @param array $replyTo Reply-to email address
 * @param array $attachemnts
 * @return bool
 * @deprecated use Email::addToQueue()
 */
	public function queueEmail($subject, $content, array $to, array $from = null, $template = null, $cc = null, $bcc = null, $replyTo = null, $helpers = [], $headContent = null, $layout = null, $attachments = null) {
		return $this->addToQueue(
			$subject,
			$content,
			$to,
			[
				'from' => $from ?: [Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')],
				'replyTo' => $replyTo,
				'subject' => $subject,
				'content' => $content,
				'headContent' => $headContent,
				'template' => $template,
				'layout' => $layout,
				'helpers' => $helpers,
				'domain' => Configure::read('App.fullBaseUrl'),
				'attachments' => $attachments,
				'cc' => $cc,
				'bcc' => $bcc
			]
		);
	}

/**
 * Convinience method for adding any email to the queue.
 *
 * @param string $subject The email subject
 * @param string $content Main content of the email
 * @param array $to An array of emails the email will be sent to
 * @param array $additionalParams An array containing additional email parameters
 * @see /Plugin/EvEmail/readme.md for an example of what is expected in the $additionalParams array
 * @return bool
 */
	public function addToQueue($subject, $content, array $to, $additionalParams = []) {
		$notBefore = null;
		if (!empty($additionalParams['queueNotBefore'])) {
			$notBefore = $additionalParams['queueNotBefore'];
			unset($additionalParams['queueNotBefore']);
		}

		$group = null;
		if (!empty($additionalParams['queueGroup'])) {
			$group = $additionalParams['queueGroup'];
			unset($additionalParams['queueGroup']);
		}

		$reference = null;
		if (!empty($additionalParams['queueReference'])) {
			$reference = $additionalParams['queueReference'];
			unset($additionalParams['queueReference']);
		}

		return (bool)ClassRegistry::init('Queue.QueuedTask')->createJob(
			'EvEmail',
			[
				'to' => $to,
				'cc' => !empty($additionalParams['cc']) ? $additionalParams['cc'] : null,
				'bcc' => !empty($additionalParams['bcc']) ? $additionalParams['bcc'] : null,
				'from' => !empty($additionalParams['from']) ? $additionalParams['from'] : [Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')],
				'replyTo' => !empty($additionalParams['replyTo']) ? $additionalParams['replyTo'] : null,
				'subject' => $subject,
				'content' => $content,
				'headContent' => !empty($additionalParams['headContent']) ? $additionalParams['headContent'] : null,
				'template' => !empty($additionalParams['template']) ? $additionalParams['template'] : null,
				'layout' => !empty($additionalParams['layout']) ? $additionalParams['layout'] : null,
				'helpers' => !empty($additionalParams['helpers']) ? $additionalParams['helpers'] : [],
				'domain' => Configure::read('App.fullBaseUrl'),
				'attachments' => !empty($additionalParams['attachments']) ? $additionalParams['attachments'] : null,
				'viewVars' => !empty($additionalParams['viewVars']) ? $additionalParams['viewVars'] : []
			],
			$notBefore,
			$group,
			$reference
		);
	}

/**
 * Splits a line of HTML respecting the tags where possible.
 *
 * @param string $str The string to split
 * @param int $limit The maximum number of characters on a line
 * @return string The string with additional line breaks
 */
	protected function _splitHtmlString($str, $limit) {
		$newStrParts = [];
		$remainingString = $str;
		$remainingLength = strlen($remainingString);

		while ($remainingLength > $limit) {
			// First see if there's an existing line break
			$nextSplit = strrpos($remainingString, PHP_EOL, $limit - $remainingLength);

			// Split at the last tag possible
			if (empty($nextSplit)) {
				$nextSplit = strrpos($remainingString, '>', $limit - $remainingLength);
			}

			// If there's no tags we're likely in a long bit of content. Split at the last space
			if (empty($nextSplit)) {
				$nextSplit = max($nextSplit, strrpos($remainingString, ' ', $limit - $remainingLength));
			}

			// If non are found just split at the character limit
			if (empty($nextSplit)) {
				$nextSplit = $limit;
			}

			// Split the string and try again with the remainder
			$newStrParts[] = trim(substr($remainingString, 0, $nextSplit + 1));
			$remainingString = substr($remainingString, $nextSplit + 1);
			$remainingLength = strlen($remainingString);
		}

		$newStrParts[] = trim($remainingString);
		return implode(PHP_EOL, $newStrParts);
	}

}
