<?php

App::uses('EvEmailAppController', 'EvEmail.Controller');

class EmailsController extends EvEmailAppController {

	/**
	 * Supported admin actions
	 * @var array
	 */
	public $adminActions = [
		'admin_index',
		'admin_edit',
		'admin_delete'
	];

	/**
	 * Admin edit form
	 * @param int $id Email ID
	 * @return void
	 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);
		if ($id !== null) {
			$data = $this->{$this->modelClass}->findById($id);
			if (!$this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
				// We need to make sure that the email tokens (that we're not saving) don't get lost
				// between page loads. So we fetch them again. Only a super user can change these.
				$this->request->data['Email']['required_tokens'] = $data['Email']['required_tokens'];
				$this->request->data['Email']['optional_tokens'] = $data['Email']['optional_tokens'];
			}

			if (
				!$this->request->data['Email']['override_to']
				|| (!empty($data) && $data['Email']['override_to'] !== true)
			) {
				unset($this->viewVars['fields']['Email.to_name']);
				unset($this->viewVars['fields']['Email.to_email']);
			}
		} else {
			if (empty($this->request->data['Email']['override_to'])) {
				unset($this->viewVars['fields']['Email.to_name']);
				unset($this->viewVars['fields']['Email.to_email']);
			}
		}
		return;
	}

	/**
	 * Delete action, only permit super users.
	 * @param int $id Email ID
	 * @return void
	 */
	public function admin_delete($id) {
		if (!$this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			throw new UnauthorizedException();

		}
		return parent::admin_delete($id);
	}

	/**
	 * Used to populate form drop down selects
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$this->set('emailGroups', $this->{$this->modelClass}->EmailGroup->find('list'));
		return;
	}

	/**
	 * Admin index paginate
	 * @return array
	 */
	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();
		$paginate['contain'] = [
			'EmailGroup'
		];
		return $paginate;
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		// Ensure token fields use textareas.
		$fields['Email.required_tokens']['type'] = 'text_plain';
		$fields['Email.optional_tokens']['type'] = 'text_plain';

		if (Configure::read('EvEmail.show_cc_field')) {
			$fields['Email.cc']['type'] = 'string';
			$fields['Email.cc']['label'] = __d('ev_email', 'CC Emails');
		} else {
			unset($fields['Email.cc']);
		}
		if (Configure::read('EvEmail.show_bcc_field')) {
			$fields['Email.bcc']['type'] = 'string';
			$fields['Email.bcc']['label'] = __d('ev_email', 'BCC Emails');
		} else {
			unset($fields['Email.bcc']);
		}

		$fields['Email.from_name']['placeholder'] = Configure::read('SiteSetting.general.site_title');
		$fields['Email.from_email']['placeholder'] = Configure::read('SiteSetting.general.admin_email');

		if (!$this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			// Hide system name, this should never be changed and is set by the super user on
			// creation).
			unset($fields['Email.system_name']);
			// Hide the override_to column (this is to be set by a super user only).
			unset($fields['Email.override_to']);
			// The email name should only be set by a super user and is used to help other admin
			// user's identify the email template.
			$fields['Email.name']['displayonly'] = 'displayonly';
			// The tokens should only be set by the super user.
			$fields['Email.required_tokens']['displayonly'] = 'displayonly';
			$fields['Email.optional_tokens']['displayonly'] = 'displayonly';
		}

		return $fields;
	}

	/**
	 * Admin index columns
	 *
	 * @return array
	 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();
		// Remove modified columns.
		unset($columns['Email.created']);
		$newColumns = array(
			'EmailGroup.name' => [
				'label' => __d('ev_email', 'Group'),
				'type' => 'string'
			]
		);
		return ArrayUtil::addAfter($columns, 'Email.subject', $newColumns);
	}

	/**
	 * Defines a whitelist of columns to be displayed on the admin index.
	 * @return array whitelisted columns
	 */
	protected function _adminIndexColumnsWhitelist() {
		$Email = $this->{$this->modelClass};
		$alias = $Email->alias;
		$columns = parent::_adminIndexColumnsWhitelist();
		$columns[] = $alias . '.subject';
		return $columns;
	}

	/**
	 * Filters
	 * @return array
	 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		unset($filters['Email.created']);
		unset($filters['Email.modified']);

		$newFilters = array(
			'Email.subject' => array(
				'label' => __d('ev_email', 'Subject'),
				'type' => 'string',
				'compare' => array('Email.subject LIKE' => "%%%s%%")
			),
			'Email.email_group_id' => array(
				'label' => __d('ev_email', 'Group'),
				'type' => 'select',
				'compare' => array('Email.email_group_id' => "%s")
			)
		);

		return ArrayUtil::addAfter($filters, 'Email.name', $newFilters);
	}

	/**
	 * Admin index actions
	 * @return void
	 */
	protected function _adminIndexActions() {
		$actions = parent::_adminIndexActions();
		if (!$this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			unset($actions['Delete']);
		}
		return $actions;
	}

	protected function _adminIndexToolbar() {
		if (!$this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			return [];
		}
		return parent::_adminIndexToolbar();
	}

	protected function _adminFormToolbar($id = null) {
		if (!$this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			return [];
		}
		return parent::_adminFormToolbar($id);
	}

}
