<?php

App::uses('EvDiscountAppModel', 'EvDiscount.Model');
App::uses('CakeNumber', 'Utility');
App::uses('CakeTime', 'Utility');

class DiscountCode extends EvDiscountAppModel {

	// If set to true, code validation will accept any existing code regardless of is_active or valid between dates
	public $allowDisabledCodes = false;

	public $actsAs = array(
		'EvCurrency.Currency' => array(
			'formInject' => true
		)
	);

	/**
	 * validate the code
	 */
	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have a name'
			)
		),
		'code' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have a code'
			),
			'isUnique' => array(
				'rule' => array('isUnique'),
				'message' => 'The discount code must be unique'
			)
		),
		'discount_type_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have a type'
			)
		),
		'amount' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have an amount set'
			)
		),
		'maximum_uses' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The maximum amount of uses must be set'
			)
		),
		'currency_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please select a currency'
			)
		)
	);

	/**
	 * hasMany Relationships
	 */
	public $hasMany = array(
		'CodeRestriction' => array(
			'className' => 'EvDiscount.CodeRestriction',
			'foreignKey' => 'discount_model_id',
			'conditions' => [
				'discount_model' => 'DiscountCode'
			]
		)
	);

	/**
	 * belongsTo Relationships
	 */
	public $belongsTo = array(
		'DiscountType' => array(
			'className' => 'EvDiscount.DiscountType'
		),
		'Currency' => array(
			'className' => 'EvCurrency.Currency'
		)
	);

	/**
	 * after find to format amount
	 *
	 * Called after each find operation. Can be used to modify any results returned by find().
	 * Return value should be the (modified) results.
	 *
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
	 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as $key => $item) {
			if (isset($item[$this->alias]['amount'])) {
				/*
				 * Passing in options to the formatter as the default options turn the amount into a string which can
				 * lead to issues when using the amount in calculations later.
				 */
				$results[$key][$this->alias]['amount'] = CakeNumber::format(
					$item[$this->alias]['amount'],
					[
						'places' => 2,
						'before' => false,
						'thousands' => false,
					]
				);
			}

			if (isset($item[$this->alias]['min_order'])) {
				/*
				 * Passing in options to the formatter as the default options turn the amount into a string which can
				 * lead to issues when using the amount in calculations later.
				 */
				$results[$key][$this->alias]['min_order'] = CakeNumber::format(
					$item[$this->alias]['min_order'],
					[
						'places' => 2,
						'before' => false,
						'thousands' => false,
					]
				);
			}

			if (!empty($item[$this->alias]['extra_data'])) {
				$results[$key]['DiscountData'] = json_decode($item[$this->alias]['extra_data'], true);
			}
		}

		return $results;
	}

	public function beforeBeforeSave($data, $options) {
		$this->restrictionsToDelete = [];

		if (!empty($data['CodeRestriction'])) {
			foreach ($data['CodeRestriction'] as $restrictionAlias => $restriction) {
				$restrictionsToDelete = [];
				$currentCodeRestrictions = json_decode($restriction['current_model_id'], true);

				if (!empty($restriction['model_id'])) {
					// Add new restriction to data
					foreach ($restriction['model_id'] as $modelId) {
						$codeRestriction = [
							'discount_model' => $this->alias,
							'model' => $restriction['model'],
							'model_id' => $modelId
						];

						// If we are updating a restriction then we need to pass the id in too
						if (in_array($modelId, $currentCodeRestrictions)) {
							$codeRestriction['id'] = array_search($modelId, $currentCodeRestrictions);
						}

						$data['CodeRestriction'][] = $codeRestriction;
					}

					// Find which restrictions need deleting
					foreach ($currentCodeRestrictions as $restrictionId => $restrictionModelId) {
						if (!in_array($restrictionModelId, $restriction['model_id'])) {
							$restrictionsToDelete[] = $restrictionId;
						}
					}

					// Add the restrictions to be deleted to the deleted array
					if (!empty($restrictionsToDelete)) {
						$this->restrictionsToDelete = array_merge($this->restrictionsToDelete, $restrictionsToDelete);
					}

				} else {
					// There aren't any restrictions to save to remove any that did exist
					$this->restrictionsToDelete = array_merge($this->restrictionsToDelete, array_keys($currentCodeRestrictions));
				}

				// Clean up data array
				unset($data['CodeRestriction'][$restrictionAlias]);
			}
		}

		if (!empty($data['DiscountData'])) {
			$data[$this->alias]['extra_data'] = json_encode($data['DiscountData']);
			unset($data['DiscountData']);
		}

		return $data;
	}

/**
 * {@inheritDoc}
 */
	public function beforeSave($options = []) {
		$beforeSave = parent::beforeSave($options);

		if ($beforeSave === false) {
			return false;
		}

		/*
		 * Set the amount of the discount code to 1 if the discount code is set
		 * as a free gift. Free gifts can't have an amount set on them in the
		 * CMS but break if an amount is not set. The amount is set to 1 so
		 * that a maximum of 1 free gift is provided to each basket.
		 */
		$freeGiftDiscountTypeId = $this->DiscountType->field(
			'id',
			[
				$this->DiscountType->alias . '.name' => 'Free gift',
			]
		);

		if (
			!empty($freeGiftDiscountTypeId)
			&& !empty($this->data[$this->alias]['discount_type_id'])
			&& $this->data[$this->alias]['discount_type_id'] === $freeGiftDiscountTypeId
		) {
			$this->data[$this->alias]['amount'] = 1;
		}

		return true;
	}

	/**
	 * Called after each successful save operation.
	 *
	 * @param bool $created True if this save created a new record
	 * @param array $options Options passed from Model::save().
	 * @return void
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#aftersave
	 * @see Model::save()
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		// process any custom field deletions
		if (!empty($this->restrictionsToDelete)) {
			$this->CodeRestriction->deleteAll(
				array(
					'CodeRestriction.id' => $this->restrictionsToDelete
				),
				true,
				false
			);
		}
	}

	/**
	 * readForEdit - retrieve the zone extra data
	 *
	 * @param integer $id ID of row to edit
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForEdit($id, $query = array()) {
		$query['contain'][] = 'CodeRestriction';

		return parent::readForEdit($id, $query);
	}

	/**
	 * Checks if the given codes are valid
	 *
	 * @param  array $code codes
	 * @return array returns the valid codes
	 */
	public function validateCodes($code) {
		$codes = $this->getCode(array_keys($code));

		if (empty($codes)) {
			return array();
		}

		$codes = Hash::combine($codes, '{n}.DiscountCode.code', '{n}');

		return $codes;
	}

	/**
	 * get the code data by the code
	 * If the validateDisabledCodes flag has been set then any existing code for the current currency will be returned.
	 * Otherwise only a currently valid code will be returned.
	 *
	 * @param   string|array  $code   The code that was entered
	 * @return  array   $data   The code and restriction data
	 */
	public function getCode($code) {
		$conditions = [
			'DiscountCode.currency_id' => CakeSession::read('EvCurrency.currencyId'),
		];

		// if it's an empty array, don't have this condition.
		if (!empty($code)) {
			$conditions['DiscountCode.code'] = $code;
		}

		if (!$this->allowDisabledCodes) {
			// Add restrictions based on if a code is currently active and has not been used up
			$conditions += $this->_getDiscountRestrictionsConditions();
		}

		return $this->find(
			(is_array($code) ? 'all' : 'first'),
			array(
				'contain' => array(
					'DiscountType',
					'CodeRestriction'
				),
				'conditions' => $conditions
			)
		);
	}

/**
 * Gets the current count of available discount codes.
 *
 * @param array $query any additional query parameters
 *
 * @return int
 */
	public function getNumActiveDiscountCodes($query = []) {
		$defaultConditions = $this->_getDiscountRestrictionsConditions();

		if (array_key_exists('conditions', $query)) {
			$query['conditions'] = array_merge($query['conditions'], $defaultConditions);
		} else {
			$query['conditions'] = $defaultConditions;
		}

		return $this->find('count', $query);
	}

/**
 * Checks if there's any active discount codes.
 *
 * @param array $query any additional query parameters
 *
 * @return bool
 */
	public function hasActiveDiscountCodes($query = []) {
		return (bool)$this->getNumActiveDiscountCodes($query);
	}
/**
 * Returns restriction conditions for discount codes.
 *
 * @return array
 */
	protected function _getDiscountRestrictionsConditions() {
		$now = CakeTime::toServer(time(), Configure::read('Config.timezone'), 'Y-m-d H:i:s');
		return [
			'DiscountCode.is_active' => true,
			[
				'OR' => [
					'DiscountCode.times_used < DiscountCode.maximum_uses',
					'DiscountCode.maximum_uses <= 0',
					'DiscountCode.maximum_uses IS NULL',
				]
			],
			[
				'OR' => [
					[
						'DiscountCode.start_date IS NOT NULL',
						'DiscountCode.start_date <=' => $now
					],
					'DiscountCode.start_date IS NULL'
				]
			],
			[
				'OR' => [
					[
						'DiscountCode.end_date IS NOT NULL',
						'DiscountCode.end_date >=' => $now
					],
					'DiscountCode.end_date IS NULL'
				]
			]
		];
	}

	/**
	 * increases the number of times the discount code has been used by one.
	 * @param  integer $id id of the discount code to increment
	 * @return booolean
	 */
	public function increaseTimesUsed($id) {
		$discount = $this->findById($id);

		if (! empty($discount)) {
			$this->set('id', $id);
			$this->set('times_used', $discount['DiscountCode']['times_used'] + 1);

			return $this->save();
		}
		return false;
	}

	/**
	 * increases the number of times the discount code has been used by one.
	 * @param  integer $code Discount code, code to increment
	 * @return booolean
	 */
	public function increaseTimesUsedByCode($code) {
		$discount = $this->findByCode($code);
		if (! empty($discount)) {
			$this->set('id', $discount['DiscountCode']['id']);
			$this->set('times_used', $discount['DiscountCode']['times_used'] + 1);

			return $this->save();
		}
		return false;
	}
}
