<?php
namespace Evoluted\PriceModifier\DiscountModifiers;

use Evoluted\PriceModifier\DiscountModifiers\BaseDiscountModifier;
use Evoluted\PriceModifier\Interfaces\DiscountModifierInterface;

/**
 * A discount handler for the PriceModifer package. Discounts the price of [amount] items in the
 * basket to [fixed_price] for every [amount] items bought.
 *
 * @package 	PriceModifier
 * @author		Evoluted New Media <developers@evoluted.net>
 * @license     http://mit-license.org/
 *
 * @link		https://github.com/evolutednewmedia/pricemodifier
 *
 */
class XForFixedPriceOfDiscountModifier extends BaseDiscountModifier implements DiscountModifierInterface {

/**
 * Applies the discount to the loaded basket.
 *
 * @return BasketInterface Returns the modified basket
 */
	public function applyDiscount() {
		$validItems = [];
		$totalCount = 0;

		foreach ($this->basket->items() as $basketItem) {
			if (! $this->basket->validDiscount($this->id, $basketItem)) {
				continue;
			}
			for ($i = 0; $i < $basketItem->quantity; $i++) {
				$validItems[] = $basketItem;
			}
			$totalCount += $basketItem->quantity;
		}

		if ($totalCount < $this->params['amount']) {
			// We don't have enough items to qualify for this discount
			return $this->basket;
		}

		// first sort so the cheapest are at the front of the array
		usort($validItems, function ($a, $b) {
			if ($a->unitPrice === $b->unitPrice) {
				return 0;
			}

			return $a->unitPrice > $b->unitPrice ? 1 : -1;
		});

		$groupedValidItems = array_chunk($validItems, $this->params['amount'], true);

		if (! isset($this->basket->discountBreakdown[$this->id]['amount'])) {
			$this->basket->discountBreakdown[$this->id]['amount'] = 0;
		}
		if (! isset($this->basket->discountBreakdown[$this->id]['tax'])) {
			$this->basket->discountBreakdown[$this->id]['tax'] = 0;
		}

		foreach ($groupedValidItems as $items) {
			if (count($items) < $this->params['amount']) {
				continue;
			}

			extract($this->_determineDiscountForGroup($items, $this->params['extra_data']['fixed_price']));

			$this->basket->discountBreakdown[$this->id]['amount'] -= $discountSubtotal;
			$this->basket->discountBreakdown[$this->id]['tax'] -= $discountTax;
		}

		return $this->basket;
	}

/**
 * Determine the discount applicable to the items in the discount group
 *
 * @param BasketItemInterface $items Group of items to apply discount to
 * @param float $fixedPrice Price to pay for group of items
 * @return array Discount subtotal and discount tax
 */
	protected function _determineDiscountForGroup($items, $fixedPrice) {
		$subtotal = 0;
		$taxTotal = 0;

		foreach ($items as $item) {
			$tax = round(($item->unitPrice * ($item->taxRate() / 100)), 2);
			$itemPrice = $item->unitPrice;

			$subtotal += $itemPrice;
			$taxTotal += $tax;
		}

		$total = $subtotal + $taxTotal;

		if ($total < $fixedPrice) {
			return [
				'discountSubtotal' => 0,
				'discountTax' => 0,
			];
		}

		return $this->_normaliseDiscount(
			($fixedPrice * $subtotal / $total) - $subtotal,
			($fixedPrice * $taxTotal / $total) - $taxTotal
		);
	}
}
