<?php
namespace Evoluted\EvDiscount\DiscountRestrictions;

use Evoluted\PriceModifier\DiscountRestrictions\Basic;

/**
 * The basic discount restriction handler allows all discount types through without
 * restricting them.
 *
 * @package 	PriceModifier
 * @author 		Sam Biggins <sam@evoluted.net>
 * @author		Evoluted New Media <developers@evoluted.net>
 * @license     http://mit-license.org/
 *
 * @link		https://github.com/evolutednewmedia/pricemodifier
 *
 */
class DiscountRestrictions extends Basic {

	/**
	 * Sets the restrictions on the basket.
	 *
	 * @return void
	 */
	public function setRestrictions() {
		// Setup discounts array incase there are no discount restrictions
		$this->basket->validDiscounts = [];

		foreach ($this->basket->discountRules as $rule) {
			$ruleEnabled = true;

			if ($rule['min_order'] > 0) {
				if ($rule['min_order'] > $this->basket->getBasketTotal()) {
					$ruleEnabled = false;
				}
			}

			if (isset($rule['code_restrictions']) && ! empty($rule['code_restrictions'])) {

				// Go through each of the basket items and see if they rule applies to them.
				foreach ($this->basket->items() as $basketItemId => $basketItem) {
					// This code has restrictions that need checking
					$checkDiscountRestriction = new \CakeEvent('EvBasket.Component.BasketItem.CheckDiscountRestrictions', $basketItem, array(
						'discount' => $rule,
						'basket_item' => array_merge(['id' => $basketItemId], $basketItem->toArray()),
					));
					\CakeEventManager::instance()->dispatch($checkDiscountRestriction);
					// If false is returned discount isn't available for this item
					$results = $checkDiscountRestriction->result;

					// If no items are valid it won't add the discount rule to the list of valid discounts
					if (!empty($results['discountValue']) && $ruleEnabled) {
						$this->basket->validDiscounts[$rule['id']][] = $basketItemId;
					}
				}

				// Check for delivery discount restrictions
				$checkDeliveryDiscountRestriction = new \CakeEvent('EvBasket.Component.BasketDelivery.CheckDiscountRestrictions', $this->basket, array(
						'discount' => $rule,
						'basket' => $this->basket->toArray(),
				));
				\CakeEventManager::instance()->dispatch($checkDeliveryDiscountRestriction);
				// If false is returned discount isn't available for the selected delivery method on this basket
				$results = $checkDeliveryDiscountRestriction->result;

				if (!empty($results['discountValue']) && $ruleEnabled) {
					$this->basket->validDiscounts[$rule['id']] = true;
				}

			} else {
				//Check for any general discount restrictions on each item
				foreach ($this->basket->items() as $basketItemId => $basketItem) {
					// This code has restrictions that need checking
					$checkGeneralItemDiscountRestriction = new \CakeEvent(
						'EvBasket.Component.BasketItem.CheckGeneralDiscountRestrictions',
						$basketItem,
						array(
							'discount' => $rule,
							'basket_item' => array_merge(['id' => $basketItemId], $basketItem->toArray()),
						)
					);
					\CakeEventManager::instance()->dispatch($checkGeneralItemDiscountRestriction);

					// If false is returned discount isn't available for this item
					$results = $checkGeneralItemDiscountRestriction->result;

					// If no items are valid it won't add the discount rule to the list of valid discounts
					if (!empty($results['discountValue']) && $ruleEnabled) {
						$this->basket->validDiscounts[$rule['id']][] = $basketItemId;
					} elseif (isset($results['discountValue']) && $results['discountValue'] === false) {
						$ruleEnabled = false;
					}
				}

				//If a basket item didn't meet the criteria for a item discount then it doesn't meet the criteria for a
				//general basket discount. Use of this event would be very limited as much of it's functionality can be
				//created through the code restrictions and or promotions. This is more for hard checks against the
				//basket. Scenario might be that you have a free delivery for a basket but one of the basket items
				//requires payment for delivery so the basket delivery needs to be paid.
				if ($ruleEnabled) {
					//Check for any general discount restriction on the whole basket.
					$checkBasketDiscountRestriction = new \CakeEvent(
						'EvBasket.Component.Basket.CheckGeneralDiscountRestrictions',
						$this->basket,
						array(
							'discount' => $rule,
							'basket' => $this->basket->toArray(),
						)
					);
					\CakeEventManager::instance()->dispatch($checkBasketDiscountRestriction);
					// If false is returned discount isn't available for the basket
					$results = $checkBasketDiscountRestriction->result;

					if (!empty($results['discountValue']) && $ruleEnabled) {
						$this->basket->validDiscounts[$rule['id']] = true;
					} elseif (isset($results['discountValue']) && $results['discountValue'] === false) {
						$ruleEnabled = false;
					}
				}

				// Not restricted by item or basket so add discount code with no product restrictions
				if ($ruleEnabled) {
					$this->basket->validDiscounts[$rule['id']] = true;
				}
			}
		}
	}
}
