<?php
namespace Evoluted\PriceModifier\DiscountModifiers;

use Evoluted\PriceModifier\DiscountModifiers\BaseDiscountModifier;
use Evoluted\PriceModifier\Interfaces\DiscountModifierInterface;

/**
 * A discount handler for the PriceModifer package. For every [amount] products, the cheapest is discounted from the total.
 *
 * @package 	PriceModifier
 * @author		Evoluted New Media <developers@evoluted.net>
 * @license     http://mit-license.org/
 *
 * @link		https://github.com/evolutednewmedia/pricemodifier
 *
 */
class CheapestFreeDiscountModifier extends BaseDiscountModifier implements DiscountModifierInterface {

	/**
	 * Applies the discount to the loaded basket.
	 *
	 * @return BasketInterface Returns the modified basket
	 */
	public function applyDiscount() {
		//
		$validItems = [];
		$totalCount = 0;

		foreach ($this->basket->items() as $basketItem) {
			if ($this->basket->validDiscount($this->id, $basketItem)) {
				$validItems[] = $basketItem;
				$totalCount += $basketItem->quantity;
			}
		}

		// We have enough - take some money off the basket
		if ($totalCount >= $this->params['amount']) {

			// first sort so the cheapest are at the front of the array
			usort($validItems, function ($a, $b) {
				if ($a->unitPrice > $b->unitPrice) {
					return 1;
				} elseif ($a->unitPrice < $b->unitPrice) {
					return -1;
				} else {
					return 0;
				}
			});

			$numToDiscount = floor($totalCount / $this->params['amount']);
			foreach ($validItems as $item) {
				if ($item->quantity < $numToDiscount) {
					extract($this->_normaliseDiscount($item->unitPrice * $item->quantity, $item->unitPrice * $item->quantity * ($item->taxRate() / 100)));
					$numToDiscount -= $item->quantity;
				} else {
					extract($this->_normaliseDiscount($item->unitPrice * $numToDiscount, $item->unitPrice * $numToDiscount * ($item->taxRate() / 100)));
					$numToDiscount = 0;
				}

				if (!isset($this->basket->discountBreakdown[$this->id]['amount'])) {
					$this->basket->discountBreakdown[$this->id]['amount'] = 0;
				}
				if (!isset($this->basket->discountBreakdown[$this->id]['tax'])) {
					$this->basket->discountBreakdown[$this->id]['tax'] = 0;
				}

				$this->basket->discountBreakdown[$this->id]['amount'] += $discountSubtotal;
				$this->basket->discountBreakdown[$this->id]['tax'] += $discountTax;

				if ($numToDiscount == 0) {
					// Nothing else to do
					break;
				}
			}

		}

		return $this->basket;
	}
}
