<?php

App::uses('AppComponent', 'Controller/Component');

/**
 * Discount code component - listens in all post data to see if a discount code
 * is being applied.
 */
class DiscountCodesComponent extends AppComponent {

	/**
	 * Component startup - check for the form being submitted
	 *
	 * @param Controller $Controller Controller instance
	 * @return bool Returns true if discount applied. A flash status message is also set
	 */
	public function startup(Controller $Controller) {
		if ($this->_controller->request->is('post') && ! empty($this->_controller->request->data['Discount'])) {

			//If the discount is submitted as part of another form then check a code has been supplied
			if (!empty($this->_controller->request->data['Discount']['DiscountCode']['code'])) {
				$this->_applyDiscountCode('Discount');
			}
			//If the voucher is submitted as part of another form then check a code has been supplied
			if (!empty($this->_controller->request->data['Voucher']['DiscountCode']['code'])) {
				$this->_applyDiscountCode('Voucher');
			}
		}

		return true;
	}

/**
 * Method to handle processing a discount code. The type of
 * discount is passed in to control which flash to populate
 * @param  string $discountType used to populate the correct flash (Discount | Voucher)
 * @return null
 */
	protected function _applyDiscountCode($discountName) {
		$flashKey = strtolower($discountName) . '-flash';

		if (! is_object($this->_controller->BasketManager)) {
			$this->_controller->loadComponent('BasketManager', ['className' => 'EvBasket.Basket']);
		}

		$DiscountCode = EvClassRegistry::init('EvDiscount.DiscountCode');
		$code = $this->_controller->request->data[$discountName]['DiscountCode']['code'];
		// Clean up the discount code as it's going to be pretty common
		// for people to copy/paste and accidentaly get a space round it.
		$code = trim($code);

		if (empty($code)) {
			return false;
		}

		// Check the basket data to see if we already have a discount code applied
		$basketData = $this->_controller->BasketManager->getBasketData();
		$discountCode = Hash::extract($basketData, "{n}[name=discount_code].data");

		$data = $DiscountCode->getCode($code);

		if (empty($data)) {
			$this->_controller->Flash->fail(
				__('The code entered is not valid'),
				array(
					'key' => $flashKey
				)
			);

			return false;
		}

		if (! empty($discountCode) && Configure::read('EvDiscount.multipleCodes') !== true && $data['DiscountCode']['is_voucher'] == 0) {
			$this->_controller->Flash->fail(
				__('A discount code has already been applied'),
				array(
					'key' => $flashKey
				)
			);
			return false;
		}

		// dispatch event
		$this->_controller->getEventManager()->dispatch(
			new CakeEvent('EvDiscount.Component.DiscountCodes.codeAdded', $this, array(
				'code' => $code,
				'data' => $data
			))
		);

		$discountTotal = $this->_controller->BasketManager->findTotalRow(Configure::read('EvBasket.labels.discount'));
		if (!empty($discountTotal)) {
			$this->_controller->Flash->success(
				__($discountName . ' code applied'),
				array(
					'key' => $flashKey
				)
			);
		} else {
			$this->_controller->Flash->fail(
				__($discountName . ' code not applied'),
				array(
					'key' => $flashKey
				)
			);
		}
	}

}
