<?php

App::uses('EvDiscountAppModel', 'EvDiscount.Model');
App::uses('CakeNumber', 'Utility');
App::uses('CakeTime', 'Utility');

class DiscountCode extends EvDiscountAppModel {

	/**
	 * validate the code
	 */
	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have a name'
			)
		),
		'code' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have a code'
			)
		),
		'discount_type_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have a type'
			)
		),
		'amount' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'The discount code must have an amount set'
			)
		),
	);

	/**
	 * hasMany Relationships
	 */
	public $hasMany = array(
		'CodeRestriction' => array(
			'className' => 'EvDiscount.CodeRestriction'
		)
	);

	/**
	 * belongsTo Relationships
	 */
	public $belongsTo = array(
		'DiscountType' => array(
			'className' => 'EvDiscount.DiscountType'
		)
	);

	/**
	 * after find to format amount
	 *
	 * Called after each find operation. Can be used to modify any results returned by find().
	 * Return value should be the (modified) results.
	 *
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
	 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as $key => $item) {
			if (isset($item[$this->alias]['amount'])) {
				$results[$key][$this->alias]['amount'] = CakeNumber::format($item[$this->alias]['amount'], 2);
			}
		}

		return $results;
	}

	/**
	 * Called after each successful save operation.
	 *
	 * @param bool $created True if this save created a new record
	 * @param array $options Options passed from Model::save().
	 * @return void
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#aftersave
	 * @see Model::save()
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		// process any custom field deletions
		if (! empty($this->data[$this->alias]['CodeRestrictions']['toDelete'])) {

			$ids = explode(',', $this->data[$this->alias]['CodeRestrictions']['toDelete']);
			$ids = ArrayUtil::clearArray($ids);

			$this->CodeRestriction->deleteAll(
				array(
					'CodeRestriction.id' => $ids
				),
				true,
				true
			);
		}
	}

	/**
	 * readForEdit - retrieve the zone extra data
	 *
	 * @param integer $id ID of row to edit
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForEdit($id, $query = array()) {
		$query['contain'][] = 'CodeRestriction';

		return parent::readForEdit($id, $query);
	}

	/**
	 * get the code data by the code
	 *
	 * @param 	string 	$code 	The code that was entered
	 * @return 	array 	$data 	The code and restriction data
	 */
	public function getCode($code) {
		$now = CakeTime::toServer(time(), Configure::read('Config.timezone'), 'Y-m-d H:i:s');

		return $this->find(
			'first',
			array(
				'contain' => array(
					'DiscountType',
					'CodeRestriction'
				),
				'conditions' => array(
					'DiscountCode.code' => $code,
					'DiscountCode.is_active' => 1,
					array(
						'OR' => array(
							array(
								'DiscountCode.start_date IS NOT NULL',
								'DiscountCode.start_date <=' => $now
							),
							array(
								'DiscountCode.start_date IS NULL'
							)
						)
					),
					array(
						'OR' => array(
							array(
								'DiscountCode.end_date IS NOT NULL',
								'DiscountCode.end_date >=' => $now
							),
							array(
								'DiscountCode.end_date IS NULL'
							)
						)
					)
				)
			)
		);
	}

	/**
	 * given the array from the cookie, validate the codes and return the valid ones
	 *
	 * @param 	array 	$codeCookie 	Array of codes from the cookie
	 * @return 	array
	 */
	public function validateCodes($codeCookie) {
		$codes = $this->find(
			'all',
			array(
				'conditions' => array(
					'DiscountCode.code' => array_keys($codeCookie)
				)
			)
		);

		if (empty($codes)) {
			return array();
		}

		$codes = Hash::combine($codes, '{n}.DiscountCode.code', '{n}');

		foreach ($codeCookie as $code => $hash) {
			if (empty($codes[$code]) || $hash !== $this->hashForCookie($codes[$code])) {
				unset($codes[$code]);
			}
		}

		return $codes;
	}

	/**
	 * hash a row for storage in cookie
	 *
	 * @param 	array 	$codeRow 	A row from the discountCode rtable
	 * @return 	string  $hash 		The hashed result
	 */
	public function hashForCookie($codeRow) {
		return sha1($codeRow['DiscountCode']['id'] . $codeRow['DiscountCode']['code'] . $codeRow['DiscountCode']['amount']);
	}
}
